/**
***************************************************************************
* @file CompleterConfig.cc
* Source file defining CompleterConfig class.
*
* Copyright (C) 2006 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: $
* $Date: $
* 
* $Log: $
***************************************************************************
**/

#include <fstream>
#include <sstream>
#include <dlrCommon/exception.h>
#include <dlrCommon/inputStream.h>
#include <dlrUtilities/stringManipulation.h>
#include "CompleterConfig.hh"
#include "version.hh"

// Note(xxx): sort members for quicker runtime checks.

namespace {

  std::string
  s_separatorCharacters =
    "~!@#$%^&*()+{}|:\"<>?`=[]\\;',./-";

  std::string
  s_wordCharacters =
    "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890_";

  inline const std::vector<std::string>&
  s_getWindowNamePatterns()
  {
    static std::vector<std::string> windowNamePatterns;
    if(windowNamePatterns.empty()) {
      windowNamePatterns.push_back(".*\\.c");
      windowNamePatterns.push_back(".*\\.h");
    }
    return windowNamePatterns;
  }

} // namespace


namespace xComplete {

  CompleterConfig::
  CompleterConfig(std::string configFileName)
    : m_backspaceKeys(),
      m_completionKeyNeedsBackspace(true),
      m_completionKeys(),
      m_minimumStubSize(3),
      m_minimumWordSize(5),
      m_modifierKeys(),
      m_separatorCharacters(),
      m_separatorKeys(),
      m_windowNamePatterns(),
      m_wordCharacters(),
      m_wordKeys()
  {
    if(configFileName != "") {
      this->load(configFileName);
    } else {
      m_separatorCharacters.resize(s_separatorCharacters.size());
      for(size_t characterIndex0 = 0;
          characterIndex0 < s_separatorCharacters.size();
          ++characterIndex0) {
        m_separatorCharacters[characterIndex0] =
          static_cast<KeyDescription::CharacterType>(
            s_separatorCharacters[characterIndex0]);
      }

      m_windowNamePatterns = s_getWindowNamePatterns();

      m_wordCharacters.resize(s_wordCharacters.size());
      for(size_t characterIndex1 = 0;
          characterIndex1 < s_wordCharacters.size();
          ++characterIndex1) {
        m_wordCharacters[characterIndex1] =
          static_cast<KeyDescription::CharacterType>(
            s_wordCharacters[characterIndex1]);
      }
    }
  }


  CompleterConfig::
  ~CompleterConfig()
  {
    // Empty.
  }


  bool
  CompleterConfig::
  isCompletionKeyBackspaceRequired()
  {
    return m_completionKeyNeedsBackspace;
  }

  
  void
  CompleterConfig::
  addWindowNamePattern(const std::string& regularExpression)
  {
    m_windowNamePatterns.push_back(regularExpression);
  }
  
    
  void
  CompleterConfig::
  load(const std::string& fileName)
  {
    std::ifstream inputStream(fileName.c_str());
    if(!inputStream) {
      std::ostringstream message;
      message << "Couldn't open input file: " << fileName;
      DLR_THROW(dlr::IOException, "CompleterConfig::load()",
                message.str().c_str());
    }
    this->deserialize(inputStream);
    if(!inputStream) {
      inputStream.close();
      std::ostringstream message;
      message << "Failed to read complete config from open input file: "
              << fileName;
      DLR_THROW(dlr::IOException, "CompleterConfig::load()",
                message.str().c_str());
    }
    inputStream.close();
  }


  void
  CompleterConfig::
  removeWindowNamePattern(const std::string& regularExpression)
  {
    std::vector<std::string>::iterator patternIterator =
      m_windowNamePatterns.begin();
    while(patternIterator != m_windowNamePatterns.end()) {
      if(*patternIterator == regularExpression) {
        m_windowNamePatterns.erase(patternIterator);
        return;
      }
    }
    std::ostringstream message;
    message << "Pattern " << regularExpression
            << " does not appear to match any in the current list.";
    DLR_THROW(dlr::LogicException,
              "CompleterConfig::removeWindowNamePattern()",
              message.str().c_str());
  }

  
  void
  CompleterConfig::
  save(const std::string& fileName)
  {
    std::ofstream outputStream(fileName.c_str());
    if(!outputStream) {
      DLR_THROW(dlr::IOException, "CompleterConfig::save()",
                "Couldn't open output file.");
    }
    outputStream << this->serialize();
    if(!outputStream) {
      outputStream.close();
      DLR_THROW(dlr::IOException, "CompleterConfig::save()",
                "Error writing to open output file.");
    }
    outputStream.close();
  }


  void
  CompleterConfig::
  setBackspaceKeys(
    const std::vector<KeyDescription>& backspaceKeys)
  {
    m_backspaceKeys = backspaceKeys;
  }


  void
  CompleterConfig::
  setCompletionKeyBackspaceRequired(bool isRequired)
  {
    m_completionKeyNeedsBackspace = isRequired;
  }

    
  void
  CompleterConfig::
  setCompletionKeys(
    const std::vector<KeyDescription>& completionKeys)
  {
    m_completionKeys = completionKeys;
  }


  void
  CompleterConfig::
  setModifierKeys(
    const std::vector<KeyDescription>& modifierKeys)
  {
    m_modifierKeys = modifierKeys;
  }


  void
  CompleterConfig::
  setSeparatorKeys(
    const std::vector<KeyDescription>& separatorKeys)
  {
    m_separatorKeys = separatorKeys;
  }
  


  std::istream&
  CompleterConfig::
  deserialize(std::istream& inputStream)
  {
    // It's a lot easier to use a try block than to be constantly
    // testing whether the IO has succeeded, so we tell inputStream to
    // complain if anything goes wrong.
    std::ios_base::iostate oldExceptionState = inputStream.exceptions();
    inputStream.exceptions(
      std::ios_base::badbit | std::ios_base::failbit | std::ios_base::eofbit);

    // Now on with the show.
    try{
      // Construct an InputStream instance so we can use our
      // convenience functions.
      dlr::InputStream stream(inputStream, dlr::InputStream::SKIP_WHITESPACE);

      // The preamble is short and sweet, but must be there.
      stream.expect("CompleterConfig");
      stream.expect("{");

      // Read the individual data members.
      stream.expect("version");
      stream.expect("=");
      std::string versionString;
      inputStream >> versionString;
      
      stream.expect("backspaceKeys");
      stream.expect("=");
      m_backspaceKeys = this->deserializeVector(inputStream);

      stream.expect("completionKeys");
      stream.expect("=");
      m_completionKeys = this->deserializeVector(inputStream);

      stream.expect("completionKeyNeedsBackspace");
      stream.expect("=");
      int tempInt;
      inputStream >> tempInt;
      m_completionKeyNeedsBackspace = static_cast<bool>(tempInt);
      
      stream.expect("minimumStubSize");
      stream.expect("=");
      inputStream >> m_minimumStubSize;

      stream.expect("minimumWordSize");
      stream.expect("=");
      inputStream >> m_minimumWordSize;

      stream.expect("modifierKeys");
      stream.expect("=");
      m_modifierKeys = this->deserializeVector(inputStream);

      stream.expect("separatorCharacters");
      stream.expect("=");
      m_separatorCharacters = this->deserializeCharacterVector(inputStream);

      stream.expect("separatorKeys");
      stream.expect("=");
      m_separatorKeys = this->deserializeVector(inputStream);

      stream.expect("windowNamePatterns");
      stream.expect("=");
      m_windowNamePatterns = this->deserializeStringVector(inputStream);

      stream.expect("wordCharacters");
      stream.expect("=");
      m_wordCharacters = this->deserializeCharacterVector(inputStream);

      stream.expect("wordKeys");
      stream.expect("=");
      m_wordKeys = this->deserializeVector(inputStream);
      
      // Warning(xxx): instead of buffering info above and copying it
      // here, we simply copy above, which could leave us in a
      // half-read state if the config file is hosed.
      // 
      // And finally, copy the data.
      // Code here...
    } catch(std::ios_base::failure) {
      // Empty
    }

    inputStream.exceptions(oldExceptionState);
    return inputStream;
  }
  

  std::vector<KeyDescription::CharacterType>
  CompleterConfig::
  deserializeCharacterVector(std::istream& inputStream)
  {
    // Construct an InputStream instance so we can use our
    // convenience functions.
    dlr::InputStream stream(inputStream, dlr::InputStream::SKIP_WHITESPACE);

    // Construct a vector to hold the result.
    std::vector<KeyDescription::CharacterType> outputVector;
    
    stream.expect("{");

    while(1) {
      KeyDescription::CharacterType character;
      stream.skipWhiteSpace();
      if(stream.peek() == '}') {
        break;
      }
      inputStream >> character;
      if(character == '\\') {
        inputStream >> character;
      }
      outputVector.push_back(character);
    }

    stream.expect("}");

    return outputVector;
  }


  std::vector<std::string>
  CompleterConfig::
  deserializeStringVector(std::istream& inputStream)
  {
    // Construct an InputStream instance so we can use our
    // convenience functions.
    dlr::InputStream stream(inputStream, dlr::InputStream::SKIP_WHITESPACE);

    // Construct a vector to hold the result.
    std::vector<std::string> outputVector;
    
    stream.expect("{");

    while(1) {
      std::string nextString;
      char character;
      stream.skipWhiteSpace();
      if(stream.peek() == '}') {
        break;
      }
      stream.expect("\"");
      while(1) {
        inputStream >> character;
        if(character == '\\') {
          inputStream >> character;
        }
        if(character == '"') {
          break;
        }
        nextString.push_back(character);
      }
      outputVector.push_back(nextString);
    }

    stream.expect("}");
    return outputVector;
  }
  

  std::vector<KeyDescription>
  CompleterConfig::
  deserializeVector(std::istream& inputStream)
  {
    // Construct an InputStream instance so we can use our
    // convenience functions.
    dlr::InputStream stream(inputStream, dlr::InputStream::SKIP_WHITESPACE);

    // Construct a vector to hold the result.
    std::vector<KeyDescription> outputVector;
    
    stream.expect("{");

    while(1) {
      KeyDescription keyDescription;
      stream.skipWhiteSpace();
      if(stream.peek() == '}') {
        break;
      }
      inputStream >> keyDescription;
      outputVector.push_back(keyDescription);
    }

    stream.expect("}");

    return outputVector;
  }


  std::string
  CompleterConfig::
  serialize()
  {
    std::ostringstream outputStream;
    outputStream
      << "CompleterConfig {\n\n"
      << "  version = "
      << getVersionString() << "\n\n"
      << "  backspaceKeys = "
      << this->serializeVector(m_backspaceKeys) << "\n\n"
      << "  completionKeys = "
      << this->serializeVector(m_completionKeys) << "\n\n"
      << "  completionKeyNeedsBackspace = "
      << static_cast<int>(m_completionKeyNeedsBackspace) << "\n\n"
      << "  minimumStubSize = "
      << m_minimumStubSize << "\n\n"
      << "  minimumWordSize = "
      << m_minimumWordSize << "\n\n"
      << "  modifierKeys = "
      << this->serializeVector(m_modifierKeys) << "\n\n"
      << "  separatorCharacters = "
      << this->serializeCharacterVector(m_separatorCharacters) << "\n\n"
      << "  separatorKeys = "
      << this->serializeVector(m_separatorKeys) << "\n\n"
      << "  windowNamePatterns = "
      << this->serializeStringVector(m_windowNamePatterns) << "\n\n"
      << "  wordCharacters = "
      << this->serializeCharacterVector(m_wordCharacters) << "\n\n"
      << "  wordKeys = "
      << this->serializeVector(m_wordKeys) << "\n\n"
      << "}" << std::endl;
    return outputStream.str();
  }
  

  std::string
  CompleterConfig::
  serializeCharacterVector(
    const std::vector<KeyDescription::CharacterType>& inputVector)
  {
    std::ostringstream outputStream;
    outputStream << "{";
    for(size_t vectorIndex = 0;
        vectorIndex < inputVector.size();
        ++vectorIndex) {
      if(inputVector[vectorIndex] == '\\'
         || inputVector[vectorIndex] == '}') {
        outputStream << '\\';
      }
      outputStream << inputVector[vectorIndex];
    }
    outputStream << "}";
    return outputStream.str();
  }


  std::string
  CompleterConfig::
  serializeStringVector(const std::vector<std::string>& inputVector)
  {
    std::ostringstream outputStream;
    outputStream << "{\n";
    for(size_t vectorIndex = 0;
        vectorIndex < inputVector.size();
        ++vectorIndex) {
      std::string escapedString =
        dlr::replaceString(
          dlr::replaceString(inputVector[vectorIndex], "\\", "\\\\"),
          "\"", "\\\"");
      outputStream << "    \"" << escapedString << "\"\n";
    }
    outputStream << "  }";
    return outputStream.str();
  }


  std::string
  CompleterConfig::
  serializeVector(const std::vector<KeyDescription>& inputVector)
  {
    std::ostringstream outputStream;
    outputStream << "{\n";
    for(size_t vectorIndex = 0;
        vectorIndex < inputVector.size();
        ++vectorIndex) {
      outputStream << "    " << inputVector[vectorIndex] << "\n";      
    }
    outputStream << "  }";
    return outputStream.str();
  }
  
  
} // namespace xComplete
