/**
***************************************************************************
* @file CompleterApp.hh
* Header file declaring CompleterApp class.
*
* Copyright (C) 2006 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: $
* $Date: $
* 
* $Log: $
***************************************************************************
**/

#ifndef _XCOMPLETE_COMPLETERAPP_H_
#define _XCOMPLETE_COMPLETERAPP_H_

#include <string>
#include <vector>
#include "CompleterConfig.hh"
#include "CompletionGenerator.hh"
#include "KeyDescription.hh"
#include "KeyHistory.hh"
#include "WindowDescription.hh"
#include "WindowRule.hh"

namespace xComplete {

  /**
   ** Classes which implement completion apps should derive from this
   ** pure virtual class.
   **/
  class CompleterApp {
  public:

    /** 
     * The default constructor initializes the application to a sane
     * initial configuration.
     */
    CompleterApp();


    /** 
     * This constructor initializes the application from a previously
     * saved configuration.
     * 
     * @param configDirectoryName This argument specifies the
     * configuration directory from which to read.
     */
    CompleterApp(const std::string& configDirectoryName);


    /** 
     * The destructor destroys the CompleterApp instance.
     */
    virtual
    ~CompleterApp();


    /** 
     * This member function instructs the application to notice which
     * window receives the next keystroke, and add that window to the
     * list of windows in which completion should be performed.
     */
    virtual void
    addNextWindow();


    /** 
     * This member function enables completion in any window whose
     * name matches the supplied (perl-style) regular expression.
     * 
     * @param regularExpression This argument is a regular expression
     * against which window names will be compared.
     */
    virtual void
    addWindowNamePattern(const std::string& regularExpression);


    /** 
     * This member function checks to see if a keystroke has been
     * typed by the user, returns a bool indicating whether a
     * keystroke was available, and if so returns information about
     * that keystroke by reference.
     * 
     * @param keyDescription This argument returns by reference a
     * KeyDescription instance describing the keystroke.  If no
     * keystroke was available, this argument is not changed.
     * 
     * @param windowDescription This argument returns by reference a
     * WindowDescription instance describing which window received the
     * keystroke.  If no keystroke was available, this argument is not
     * changed.
     * 
     * @return The return value is true if a keystroke was available,
     * and otherwise false.
     */
    virtual bool
    checkForKeystroke(KeyDescription& keyDescription,
                      WindowDescription& windowDescription) = 0;


    
    /** 
     * This member function removes all regular expressions from the
     * name-matching list, so that the name of a window can no longer
     * enable completion in that window.  See also member function
     * addWindowNamePattern().
     */
    virtual void
    clearWindowNamePatterns();


    /** 
     * This member function returns a short string describing version,
     * authorship, etc. of the program.  This string is suitable for
     * display in an "about" dialog box.
     * 
     * @return The return value is a short string.
     */
    virtual std::string
    getCredits();
    

    /** 
     * This member function returns the name of (but not the complete
     * path to) the default configuration file for CompleterApp
     * instances.  It is used to locate the correct configuration file
     * within a specified configuration directory.
     * 
     * @return The return value is a filename.
     */
    virtual std::string
    getDefaultConfigFileName() {return "xComplete2Config.txt";}


    /** 
     * This member function returns the name of (but not the complete
     * path to) the default "database" file containing works from
     * which to draw completions.  It is used to locate the correct
     * database file within a specified configuration directory.
     * 
     * @return The return value is a filename.
     */
    virtual std::string
    getDefaultDatabaseFileName() {return "completionDb.txt";}


    /** 
     * This function returns a string containing basic help
     * information for the application.
     * 
     * @return The return value is the help string.
     */
    virtual std::string
    getHelpString();
    

    /** 
     * This member function accepts keystroke information such as is
     * returned by member function checkForKeystroke(), and does the
     * appropriate thing with that keystroke, such as updating
     * internal state or sending a completion.
     * 
     * @param keyDescription This argument describes the keystroke.
     * 
     * @param windowDescription This argument describes the window in
     * which the keystroke was typed.
     */
    virtual void
    processKeystroke(const KeyDescription& keyDescription,
                     const WindowDescription& windowDescription);


    /** 
     * This method suspends monitoring of keystrokes until member
     * function resume() is called.
     */
    virtual void
    pause();
    

    /** 
     * This member function instructs the application to notice which
     * window receives the next keystroke, and remove that window from
     * the list of windows in which completion should be performed.
     */
    virtual void
    removeNextWindow();


    /** 
     * This member function removes one regular expression from the
     * name-matching list, so that will no longer be used to select
     * windows in which to perform completion.
     * 
     * @param regularExpression This is the regular expression to remove.
     */
    virtual void
    removeWindowNamePattern(const std::string& regularExpression);


    /** 
     * This member function runs an event loop within the application
     * checking for new keystokes, and dispatching them as appropriate
     * when they are available.
     */
    virtual void
    run();


    /** 
     * If operation of the application has been suspended by the
     * pause() method, then this member fucntion will cause it to
     * resume.
     */
    virtual void
    resume();
    

    /** 
     * This member function will write the currently learned database
     * of words to the appropriate configuration file.
     */
    virtual void
    saveHistory();
    
  protected:

    virtual void
    clearKeystrokeQueue();

    
    virtual void
    displayStatus(const std::string& message);

    
    virtual void
    displayStatus(const std::vector<KeyDescription>& message);
    

    virtual bool
    generateAndSendSuggestion(const std::vector<KeyDescription>& wordBeginning,
                              const WindowDescription& windowDescription,
                              size_t completionIndex);


    virtual void
    generateAndShowSuggestion(const std::vector<KeyDescription>& wordBeginning,
                              size_t completionIndex);


    virtual const std::vector< std::pair<KeyDescription, WindowDescription> >&
    getLearnedKeystrokes();

    
    void
    initializeWindowRuleArray();

    
    virtual bool
    isBackspaceKey(const KeyDescription& keyDescription);


    virtual bool
    isCompletionKey(const KeyDescription& keyDescription);

  
    virtual bool
    isKeyDescriptionInCharacterList(
      const KeyDescription& keyDescription,
      const std::vector<KeyDescription::CharacterType>& characterVector);


    virtual bool
    isKeyDescriptionInList(
      const KeyDescription& keyDescription,
      const std::vector<KeyDescription>& keyDescriptionVector);


    bool
    isModifierKey(const KeyDescription& keyDescription);

    
    virtual bool
    isSeparatorKey(const KeyDescription& keyDescription);


    virtual bool
    isTaboo(const KeyDescription& keyDescription,
            const WindowDescription& windowDescription);


    virtual bool
    isWordKey(const KeyDescription& keyDescription);



    virtual void
    loadConfig();


    virtual void
    loadHistory();


    void
    learnKeystroke(KeyDescription& keyDescription,
                   WindowDescription& windowDescription);

    
    virtual void
    processBackspaceKey(const KeyDescription& keyDescription,
                        const WindowDescription& windowDescription,
                        bool isSynthetic);

    
    virtual void
    processCompletionKey(const KeyDescription& keyDescription,
                         const WindowDescription& windowDescription);


    virtual void
    processDisorientingKey(const KeyDescription& keyDescription,
                           const WindowDescription& windowDescription);


    virtual void
    processNormalKey(const KeyDescription& keyDescription,
                     const WindowDescription& windowDescription,
                     bool isSynthetic,
                     bool isSeparator=false);


    virtual void
    processSeparatorKey(const KeyDescription& keyDescription,
                         const WindowDescription& windowDescription,
                         bool isSynthetic);


    virtual std::vector< std::pair<KeyDescription, WindowDescription> >
    removeModifiers(
      const std::vector< std::pair<KeyDescription, WindowDescription> >&
      keystrokeVector);

    
    void
    removeWindowRule(const WindowRule& windowRule);

    
    virtual bool
    runOneIteration(KeyDescription& keyDescription,
                    WindowDescription& windowDescription);

    

    virtual void
    sendCompletion(const std::vector<KeyDescription>& currentWord,
                   const std::vector<KeyDescription>& newWord,
                   const WindowDescription& windowDescription);


    virtual void
    sendKeySequence(const std::vector<KeyDescription>& keySequence,
                    const WindowDescription& windowDescription) = 0;


    virtual void
    setBackspaceKeystrokes(
      const std::vector< std::pair<KeyDescription, WindowDescription> >&
      keystrokesVector);


    virtual void
    setCompletionKeystrokes(
      const std::vector< std::pair<KeyDescription, WindowDescription> >&
      keystrokesVector);


    virtual void
    setModifierKeystrokes(
      const std::vector< std::pair<KeyDescription, WindowDescription> >&
      keystrokesVector);


    virtual void
    setSeparatorKeystrokes(
      const std::vector< std::pair<KeyDescription, WindowDescription> >&
      keystrokesVector);
    
    
    virtual void
    showSuggestion(const std::vector<KeyDescription>& suggestion);


    virtual void
    showSuggestion(const std::string& suggestion) = 0;


    virtual void
    startLearningKeystrokes();


    virtual void
    stopLearningKeystrokes();
    
    
    virtual bool
    updateSyntheticKeystrokeCount(const KeyDescription& keyDescription,
                                  const WindowDescription& windowDescription);


  protected:

    bool m_addNextWindowFlag;
    CompletionGenerator m_completionGenerator;
    size_t m_completionKeyPressCount;
    CompleterConfig m_config;
    std::string m_configFileName;
    std::string m_databaseFileName;
    KeyHistory m_keyHistory;
    bool m_learnKeystrokesFlag;
    std::vector< std::pair<KeyDescription, WindowDescription> >
      m_learnedKeystrokesVector;
    bool m_pauseFlag;
    bool m_removeNextWindowFlag;
    std::map<WindowDescription::WindowID,
             std::deque<KeyDescription> > m_syntheticKeystrokeQueueMap;
    std::vector<WindowRule*> m_windowRuleArray;
    std::vector<KeyDescription> m_wordBeginning;
    
  }; // class CompleterApp


  /* ============== Non-member function declarations ============== */
  

} // namespace xComplete

/* ============ Definitions of inline & template functions ============ */

namespace xComplete {

  // Empty.
  
} // namespace xComplete

#endif /* #ifndef _XCOMPLETE_COMPLETERAPP_H_ */
