/**
***************************************************************************
* @file CompleterApp.hh
* Header file declaring CompleterApp class.
*
* Copyright (C) 2006 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: $
* $Date: $
* 
* $Log: $
***************************************************************************
**/

// Debug(xxx)
#include <iostream>

#include <sstream>
#include <dlrCommon/functional.h>
#include <dlrPortability/timeUtilities.h>
#include <dlrUtilities/path.h>
#include "CompleterApp.hh"
#include "version.hh"

namespace {

  std::string
  s_getDefaultCreditString()
  {
    std::ostringstream creditStream;
    creditStream << "xComplete2\n"
                 << "Version " << xComplete::getVersionString() << "\n\n"
                 << "Primary author: Dave LaRose, dlr@cs.cmu.edu\n\n"
                 << "Regular expression parsing implemented using\n"
                 << "boost::regex (http://www.boost.org).";
    return creditStream.str();
  }


  std::string
  s_getDefaultHelpString()
  {
    std::ostringstream helpStream;
    helpStream
      << "Introduction:\n"
      << "\n"
      << "xComplete2 is a tool which watches what you type, and helpfully "
      << "suggests completions for long words.  Documentation will go here "
      << "soon.";
    return helpStream.str();
  }
    
} // namespace


namespace xComplete {

  CompleterApp::
  CompleterApp()
    :  m_addNextWindowFlag(false),
       m_completionGenerator(),
       m_completionKeyPressCount(0),
       m_config(),
       m_configFileName(),
       m_databaseFileName(),
       m_keyHistory(),
       m_learnKeystrokesFlag(false),
       m_learnedKeystrokesVector(),
       m_pauseFlag(false),
       m_removeNextWindowFlag(false),
       m_syntheticKeystrokeQueueMap(),
       m_windowRuleArray(),
       m_wordBeginning()
  {
    this->initializeWindowRuleArray();
  }


  CompleterApp::
  CompleterApp(const std::string& configDirectoryName)
    :  m_addNextWindowFlag(false),
       m_completionGenerator(),
       m_completionKeyPressCount(0),
       m_config(),
       m_configFileName(),
       m_databaseFileName(),
       m_keyHistory(),
       m_learnKeystrokesFlag(false),
       m_learnedKeystrokesVector(),
       m_pauseFlag(false),
       m_removeNextWindowFlag(false),
       m_syntheticKeystrokeQueueMap(),
       m_windowRuleArray(),
       m_wordBeginning()
  {
    m_configFileName = dlr::joinPath(
      configDirectoryName, this->getDefaultConfigFileName());
    m_databaseFileName = dlr::joinPath(
      configDirectoryName, this->getDefaultDatabaseFileName());
    this->loadConfig();
    this->loadHistory();
  }


  CompleterApp::
  ~CompleterApp()
  {
    this->saveHistory();
    for(size_t ruleIndex = 0;
        ruleIndex < m_windowRuleArray.size();
        ++ruleIndex) {
      delete m_windowRuleArray[ruleIndex];
    }
  }


  void
  CompleterApp::
  addNextWindow()
  {
    m_addNextWindowFlag = true;
  }

  
  void
  CompleterApp::
  addWindowNamePattern(const std::string& regularExpression)
  {
    m_config.addWindowNamePattern(regularExpression);
    m_windowRuleArray.push_back(new WindowRuleNameMatch(regularExpression));
  }

  
  void
  CompleterApp::  
  clearWindowNamePatterns()
  {
    m_config.clearWindowNamePatterns();
    for(size_t ruleIndexPlusOne = m_windowRuleArray.size();
        ruleIndexPlusOne > 0;
        --ruleIndexPlusOne) {
      if(m_windowRuleArray[ruleIndexPlusOne - 1]->getRuleType()
         == WindowRule::NAME_MATCH) {
        delete m_windowRuleArray[ruleIndexPlusOne - 1];
        m_windowRuleArray.erase(
          m_windowRuleArray.begin() + ruleIndexPlusOne - 1);
      }
    }
  }

    
  std::string
  CompleterApp::  
  getCredits()
  {
    return s_getDefaultCreditString();
  }
    
    
  std::string
  CompleterApp::  
  getHelpString()
  {
    return s_getDefaultHelpString();
  }
    
    
  void
  CompleterApp::  
  processKeystroke(const KeyDescription& keyDescription,
                   const WindowDescription& windowDescription)
  {
    // If we're supposed to add the next window to the list of valid
    // windows, or remove the next window from the list of valid
    // windows, do so now.
    if(m_addNextWindowFlag) {
      m_windowRuleArray.push_back(new WindowRuleIDMatch(windowDescription));
      m_addNextWindowFlag = false;
    }
    if(m_removeNextWindowFlag) {
      this->removeWindowRule(WindowRuleIDMatch(windowDescription));
      m_removeNextWindowFlag = false;
    }
    
    // Only process the key if were not in a window that's permanently
    // or temporarily forbidden.
    if(this->isTaboo(keyDescription, windowDescription)) {
      // I think I'll retire the expiring taboo idea.
      // // Some taboos go away after a certain number of keystrokes.
      // // Make sure that this keystroke gets charged to the relevant
      // // taboos.
      // this->updateTaboos(keyDescription);
      // Since this was a taboo keystroke/window, stop processing.
      this->displayStatus("Ignoring keystroke...");
      return;
    }

    // Keystrokes which are just modifier keys being pressed can be
    // safely ignored.
    if(this->isModifierKey(keyDescription)) {
      return;
    }

    // If we've recently sent a completion, this keystroke might be one
    // of the ones we sent, not a user generated keystroke.  We'll call
    // these "synthetic keystrokes."  We keep track of which synthetic
    // keystrokes are outstanding, so we check the records here and
    // update them as appropriate.
    bool isSynthetic = this->updateSyntheticKeystrokeCount(
      keyDescription, windowDescription);

    // Looks like we're allowed to process this keystroke.  We have to do
    // different things, depending on which key was pressed.
    if(this->isCompletionKey(keyDescription)) {
      // The user pressed the completion key.  Go send the appropriate
      // characters.
      this->processCompletionKey(keyDescription, windowDescription);
    } else if(this->isBackspaceKey(keyDescription)) {
      // This backspace might be from the user, or it might be from
      // this application.  No way to be sure.
      this->processBackspaceKey(keyDescription, windowDescription, isSynthetic);
    } else if(this->isSeparatorKey(keyDescription)) {
      // Separators (such as the space key) demark the end of words,
      // so we have a seperate procedure for processing separator
      // keys.
      this->processSeparatorKey(keyDescription, windowDescription,
                                isSynthetic);
    } else if(this->isWordKey(keyDescription)) {
      // If the key is a valid part of a word, update word lists
      // appropriately.
      this->processNormalKey(keyDescription, windowDescription, isSynthetic);
    } else {
      // The user pressed a key that we don't recognize.  Reset word
      // lists.
      this->processDisorientingKey(keyDescription, windowDescription);
    }
  
  }


  void
  CompleterApp::
  pause()
  {
    m_pauseFlag = true;
  }

  
  void
  CompleterApp::
  run()
  {
    KeyDescription keyDescription;
    WindowDescription windowDescription;
    while(1) {
      this->runOneIteration(keyDescription, windowDescription);
      dlr::portability::preciseSleep(30);
    }
  }

  
  void
  CompleterApp::
  resume()
  {
    m_pauseFlag = false;
  }

  
  void
  CompleterApp::
  removeNextWindow()
  {
    m_removeNextWindowFlag = true;
  }


  void
  CompleterApp::
  removeWindowNamePattern(const std::string& regularExpression)
  {
    // Update config.
    m_config.removeWindowNamePattern(regularExpression);

    // Make sure the change is reflected in our local sequence of
    // window selection rules.
    WindowRuleNameMatch targetRule(regularExpression);
    this->removeWindowRule(targetRule);
  }

    
  void
  CompleterApp::
  saveHistory()
  {
    m_completionGenerator.save(m_databaseFileName);
  }
  

  void
  CompleterApp::
  clearKeystrokeQueue()
  {
    KeyDescription keyDescription;
    WindowDescription windowDescription;
    while(this->checkForKeystroke(keyDescription, windowDescription)) {
      // Empty.
    }
  }

  
  void
  CompleterApp::
  displayStatus(const std::string& message)
  {
    std::cout << "Status: " << message << std::endl;
  }

  
  void
  CompleterApp::
  displayStatus(const std::vector<KeyDescription>& messageVector)
  {
    std::ostringstream messageStream;
    for(size_t index0 = 0; index0 < messageVector.size(); ++index0) {
      messageStream << (messageVector[index0]).getRepresentation();
    }
    this->displayStatus(messageStream.str());
  }

  
  bool
  CompleterApp::
  generateAndSendSuggestion(const std::vector<KeyDescription>& wordBeginning,
                            const WindowDescription& windowDescription,
                            size_t completionIndex)
  {
    if(wordBeginning.size() < m_config.getMinimumStubSize()) {
      return false;
    }

    const std::vector<KeyDescription>& completion =
      m_completionGenerator.getCompletion(wordBeginning, completionIndex);

    if(completion.empty()) {
      return false;
    }
    
    if(completion.size() < m_config.getMinimumWordSize()) {
      DLR_THROW(dlr::LogicException,
                "CompleterApp::generateAndSendSuggestion()",
                "Completion has unreasonably small size.");
    }

    // Get the latest word fragment.  This is not the same as
    // wordBeginning because a previous completion key press may have
    // sent synthetic keystrokes which change the final word fragment,
    // but don't update wordBeginning.
    const std::vector<KeyDescription>& currentWordFragment =
      m_keyHistory.getFinalWordFragment(windowDescription.getWindowID());

    // Delete the old word and send the new one.
    this->sendCompletion(currentWordFragment, completion, windowDescription);
    return true;
  }


  void
  CompleterApp::
  generateAndShowSuggestion(const std::vector<KeyDescription>& wordBeginning,
                            size_t completionIndex)
  {
    if(wordBeginning.size() >= m_config.getMinimumStubSize()) {
      const std::vector<KeyDescription>& completion =
        m_completionGenerator.getCompletion(wordBeginning, completionIndex);
      this->showSuggestion(completion);
    } else {
      this->showSuggestion("");
    }
  }


  const std::vector< std::pair<KeyDescription, WindowDescription> >&
  CompleterApp::
  getLearnedKeystrokes()
  {
    return m_learnedKeystrokesVector;
  }


  void
  CompleterApp::
  initializeWindowRuleArray()
  {
    m_windowRuleArray.clear();
    std::vector<std::string> windowNamePatterns =
      m_config.getWindowNamePatterns();
    for(size_t patternIndex = 0; patternIndex < windowNamePatterns.size();
        ++patternIndex) {
      m_windowRuleArray.push_back(
        new WindowRuleNameMatch(windowNamePatterns[patternIndex]));
    }
  }
    
  
  bool
  CompleterApp::
  isBackspaceKey(const KeyDescription& keyDescription)
  {
    return isKeyDescriptionInList(
      keyDescription, m_config.getBackspaceKeys());
  }


  bool
  CompleterApp::
  isKeyDescriptionInCharacterList(
    const KeyDescription& keyDescription,
    const std::vector<KeyDescription::CharacterType>& characterVector)
  {
    // Note(xxx): Ugly way to make sure modifiers are _not_ pressed.
    // Make up something better soon.
    if(checkModifier(keyDescription.getState(), MODIFIER_CONTROL)
       || checkModifier(keyDescription.getState(), MODIFIER_MODIFIER1)
       || checkModifier(keyDescription.getState(), MODIFIER_MODIFIER2)
       || checkModifier(keyDescription.getState(), MODIFIER_MODIFIER3)
       || checkModifier(keyDescription.getState(), MODIFIER_MODIFIER4)
       || checkModifier(keyDescription.getState(), MODIFIER_MODIFIER5)) {
      return false;
    }

    // At this point, we know that modifier is either, NONE, SHIFT, or
    // CAPS_LOCK.
    for(std::vector<KeyDescription::CharacterType>::const_iterator
          vectorIterator = characterVector.begin();
        vectorIterator != characterVector.end();
        ++vectorIterator) {
      if(*vectorIterator == keyDescription.getRepresentation()) {
        return true;
      }
    }
    return false;
  }
  
  bool
  CompleterApp::
  isCompletionKey(const KeyDescription& keyDescription)
  {
    return isKeyDescriptionInList(
      keyDescription, m_config.getCompletionKeys());
  }

  
  bool
  CompleterApp::
  isKeyDescriptionInList(
    const KeyDescription& keyDescription,
    const std::vector<KeyDescription>& keyDescriptionVector)
  {
    for(std::vector<KeyDescription>::const_iterator
          vectorIterator = keyDescriptionVector.begin();
        vectorIterator != keyDescriptionVector.end();
        ++vectorIterator) {
      if(isSymmetricMatch(*vectorIterator, keyDescription)) {
        return true;
      }
    }
    return false;
  }


  bool
  CompleterApp::
  isModifierKey(const KeyDescription& keyDescription)
  {
    return isKeyDescriptionInList(
      keyDescription, m_config.getModifierKeys());
  }


  bool
  CompleterApp::
  isSeparatorKey(const KeyDescription& keyDescription)
  {
    if(isKeyDescriptionInList(keyDescription, m_config.getSeparatorKeys())) {
      return true;
    }
    if(isKeyDescriptionInCharacterList(
         keyDescription, m_config.getSeparatorCharacters())) {
      return true;
    }
    return false;
  }


  bool
  CompleterApp::
  isTaboo(const KeyDescription& keyDescription,
          const WindowDescription& windowDescription)
  {
    // Keystrokes are only permissible if they come to one of the
    // windows we've been explicitly told to access.
    for(size_t windowRuleIndex = 0;
        windowRuleIndex < m_windowRuleArray.size();
        ++windowRuleIndex) {
      if(m_windowRuleArray[windowRuleIndex]->isMatch(windowDescription)) {
        return false;
      }
    }
    return true;
  }


  bool
  CompleterApp::
  isWordKey(const KeyDescription& keyDescription)
  {
    if(isKeyDescriptionInList(keyDescription, m_config.getWordKeys())) {
      return true;
    }
    if(isKeyDescriptionInCharacterList(
         keyDescription, m_config.getWordCharacters())) {
      return true;
    }
    return false;
  }


  void
  CompleterApp::
  loadConfig()
  {
    this->clearWindowNamePatterns();
    m_config.load(m_configFileName);
    this->initializeWindowRuleArray();
  }

  
  void
  CompleterApp::
  loadHistory()
  {
    m_completionGenerator.load(m_databaseFileName);
  }

  
  void
  CompleterApp::
  learnKeystroke(KeyDescription& keyDescription,
                 WindowDescription& windowDescription)
  {
    m_learnedKeystrokesVector.push_back(
      std::make_pair(keyDescription, windowDescription));
  }

    
  void
  CompleterApp::
  processBackspaceKey(const KeyDescription& keyDescription,
                      const WindowDescription& windowDescription,
                      bool isSynthetic)
  {
    m_keyHistory.deleteKeystroke(windowDescription.getWindowID());

    if(!isSynthetic) {
      // Reset count to reflect that any upcoming completion key presses
      // will be the first in their series, and then suggest a completion.
      m_completionKeyPressCount = 0;

      // Get the latest word fragment.
      m_wordBeginning =
        m_keyHistory.getFinalWordFragment(windowDescription.getWindowID());

      // And use this to provide a suggestion.
      this->generateAndShowSuggestion(
        m_wordBeginning, m_completionKeyPressCount);

      this->displayStatus(m_wordBeginning);
    }
  
  }


  void
  CompleterApp::
  processCompletionKey(const KeyDescription& keyDescription,
                       const WindowDescription& windowDescription)
  {
    // If we need to send a backspace to delete the completion key (in
    // other words, if the completion key shows up in the typed text
    // on-screen), then we need to update our internal record of
    // what's typed so that it's consistent.  That is, if the
    // completion key shows up on screen, we need to make it show up
    // in our keyHistory, so we should add it here.
    if(m_config.isCompletionKeyBackspaceRequired()) {
      m_keyHistory.addKeystroke(
        keyDescription, windowDescription.getWindowID(), false);
    }

    // Sending the completion has the side effect of updating the lists
    // of synthesized keystrokes.  These lists are used in
    // updateSyntheticKeystrokeCount().
    bool completionSent = this->generateAndSendSuggestion(
      m_wordBeginning, windowDescription, m_completionKeyPressCount);

    if(completionSent) {
      // Display a suggestion for what will be sent on the next press of
      // the completion key.
      ++m_completionKeyPressCount;
      this->generateAndShowSuggestion(
        m_wordBeginning, m_completionKeyPressCount);
      this->displayStatus("");
    }
  }


  void
  CompleterApp::
  processDisorientingKey(const KeyDescription& keyDescription,
                         const WindowDescription& windowDescription)
  {
    // Since disorienting keys can delimit words, we record the most
    // recently typed word.
    std::vector<KeyDescription> newWord =
      m_keyHistory.getFinalWordFragment(windowDescription.getWindowID());
    if(newWord.size() >= m_config.getMinimumWordSize()) {
      m_completionGenerator.addWord(newWord);    
    }

    // But since the disorienting key is... well... disorienting,
    // we've now lost our place and should forget the key history.
    m_keyHistory.reset(windowDescription.getWindowID());
    m_completionKeyPressCount = 0;
    this->showSuggestion("");
  }


  void
  CompleterApp::
  processNormalKey(const KeyDescription& keyDescription,
                   const WindowDescription& windowDescription,
                   bool isSynthetic,
                   bool isSeparator)
  {
    m_keyHistory.addKeystroke(
      // keyDescription, windowDescription.getWindowID(), false, isSeparator);
      keyDescription, windowDescription.getWindowID(), isSeparator);

    if(!isSynthetic) {
      // Reset count to reflect that any upcoming completion key presses
      // will be the first in their series, and then suggest a completion.
      m_completionKeyPressCount = 0;

      // Get the latest word fragment.
      m_wordBeginning =
        m_keyHistory.getFinalWordFragment(windowDescription.getWindowID());

      // And use this to provide a suggestion.
      this->generateAndShowSuggestion(
        m_wordBeginning, m_completionKeyPressCount);

      this->displayStatus(m_wordBeginning);
    }
  }


  void
  CompleterApp::
  processSeparatorKey(const KeyDescription& keyDescription,
                       const WindowDescription& windowDescription,
                       bool isSynthetic)
  {
    // Since separator delimits words, we can record the most recently
    // typed word.
    std::vector<KeyDescription> newWord =
      m_keyHistory.getFinalWordFragment(windowDescription.getWindowID());
    if(newWord.size() >= m_config.getMinimumWordSize()) {
      m_completionGenerator.addWord(newWord);    
    }

    // Now we can treat it just like an alphanumeric key press.
    this->processNormalKey(keyDescription, windowDescription, isSynthetic, true);
  }


  std::vector< std::pair<KeyDescription, WindowDescription> >
  CompleterApp::
  removeModifiers(
    const std::vector< std::pair<KeyDescription, WindowDescription> >&
    keystrokeVector)
  {
    std::vector< std::pair<KeyDescription, WindowDescription> > filteredVector;
    for(size_t index0 = 0; index0 < keystrokeVector.size(); ++index0) {
      if(!this->isModifierKey(keystrokeVector[index0].first)) {
        filteredVector.push_back(keystrokeVector[index0]);
      }
    }
    return filteredVector;
  }

  
  void
  CompleterApp::
  removeWindowRule(const WindowRule& windowRule)
  {
    bool isMatchingRuleFound = false;
    std::vector<WindowRule*>::iterator ruleIterator =
      m_windowRuleArray.begin();
    while(ruleIterator != m_windowRuleArray.end()) {
      if(windowRule.isEquivalent(*ruleIterator)) {
        delete *ruleIterator;
        m_windowRuleArray.erase(ruleIterator);
        isMatchingRuleFound = true;
        break;
      }
      ++ruleIterator;
    }
    if(!isMatchingRuleFound) {
      DLR_THROW(dlr::LogicException, "CompleterApp::removeWindowRule()",
                "No matching rule found.");
    }
  }

    
  bool
  CompleterApp::
  runOneIteration(KeyDescription& keyDescription,
                  WindowDescription& windowDescription)
  {
    if(m_pauseFlag) {
      this->clearKeystrokeQueue();
    } else {
      if(this->checkForKeystroke(keyDescription, windowDescription)) {
        if(this->m_learnKeystrokesFlag) {
          this->learnKeystroke(keyDescription, windowDescription);
        } else {
          this->processKeystroke(keyDescription, windowDescription);
        }
        return true;
      }
    }
    return false;
  }


  void
  CompleterApp::
  setBackspaceKeystrokes(
    const std::vector< std::pair<KeyDescription, WindowDescription> >&
    keystrokesVector)
  {
    std::vector<KeyDescription> processedKeystrokesVector(
      keystrokesVector.size());

    // Note(xxx): Not sure what's wrong with this line.
//     std::transform(
//       keystrokesVector.begin(), keystrokesVector.begin(),
//       processedKeystrokesVector.begin(),
//       dlr::ExtractFirstFunctor<KeyDescription, WindowDescription>());

    // so instead we use this...
    dlr::ExtractFirstFunctor<KeyDescription, WindowDescription> efFunctor;
    for(size_t keystrokeIndex = 0; keystrokeIndex < keystrokesVector.size(); ++keystrokeIndex) {
      processedKeystrokesVector[keystrokeIndex] =
        efFunctor(keystrokesVector[keystrokeIndex]);
    }

    m_config.setBackspaceKeys(processedKeystrokesVector);
  }


  void
  CompleterApp::
  setCompletionKeystrokes(
    const std::vector< std::pair<KeyDescription, WindowDescription> >&
    keystrokesVector)
  {
    std::vector<KeyDescription> processedKeystrokesVector(
      keystrokesVector.size());

    // Note(xxx): Not sure what's wrong with this line.
//     std::transform(
//       keystrokesVector.begin(), keystrokesVector.begin(),
//       processedKeystrokesVector.begin(),
//       dlr::ExtractFirstFunctor<KeyDescription, WindowDescription>());

    // so instead we use this...
    dlr::ExtractFirstFunctor<KeyDescription, WindowDescription> efFunctor;
    for(size_t keystrokeIndex = 0; keystrokeIndex < keystrokesVector.size(); ++keystrokeIndex) {
      processedKeystrokesVector[keystrokeIndex] =
        efFunctor(keystrokesVector[keystrokeIndex]);
    }

    m_config.setCompletionKeys(processedKeystrokesVector);
  }


  void
  CompleterApp::
  setModifierKeystrokes(
    const std::vector< std::pair<KeyDescription, WindowDescription> >&
    keystrokesVector)
  {
    std::vector<KeyDescription> processedKeystrokesVector(
      keystrokesVector.size());

    // Note(xxx): Not sure what's wrong with this line.
//     std::transform(
//       keystrokesVector.begin(), keystrokesVector.begin(),
//       processedKeystrokesVector.begin(),
//       dlr::ExtractFirstFunctor<KeyDescription, WindowDescription>());

    // so instead we use this...
    dlr::ExtractFirstFunctor<KeyDescription, WindowDescription> efFunctor;
    for(size_t keystrokeIndex = 0; keystrokeIndex < keystrokesVector.size(); ++keystrokeIndex) {
      processedKeystrokesVector[keystrokeIndex] =
        efFunctor(keystrokesVector[keystrokeIndex]);
    }
    
    m_config.setModifierKeys(processedKeystrokesVector);
  }


  void
  CompleterApp::
  setSeparatorKeystrokes(
    const std::vector< std::pair<KeyDescription, WindowDescription> >&
    keystrokesVector)
  {
    std::vector<KeyDescription> processedKeystrokesVector(
      keystrokesVector.size());

    // Note(xxx): Not sure what's wrong with this line.
//     std::transform(
//       keystrokesVector.begin(), keystrokesVector.begin(),
//       processedKeystrokesVector.begin(),
//       dlr::ExtractFirstFunctor<KeyDescription, WindowDescription>());

    // so instead we use this...
    dlr::ExtractFirstFunctor<KeyDescription, WindowDescription> efFunctor;
    for(size_t keystrokeIndex = 0; keystrokeIndex < keystrokesVector.size(); ++keystrokeIndex) {
      processedKeystrokesVector[keystrokeIndex] =
        efFunctor(keystrokesVector[keystrokeIndex]);
    }

    m_config.setSeparatorKeys(processedKeystrokesVector);
  }
    
    
  void
  CompleterApp::
  sendCompletion(const std::vector<KeyDescription>& currentWord,
                 const std::vector<KeyDescription>& newWord,
                 const WindowDescription& windowDescription)
  {
    // There's no sense in sending characters which are common between
    // the two words, so we search to find the first difference
    // between the two.  When we're done with the "if" below, start0
    // will point to the first character at which currentWord differs
    // from n newWord, and start1 will point to the corresponding
    // character of newWord.
    typedef std::vector<KeyDescription>::const_iterator IterType;
    std::pair<IterType, IterType> start0_start1;
    if(currentWord.size() <= newWord.size()) {
      start0_start1 =
        std::mismatch(currentWord.begin(), currentWord.end(), newWord.begin(),
                      isSymmetricMatch);
    } else {
      start0_start1 =
        std::mismatch(newWord.begin(), newWord.end(), currentWord.begin(),
                      isSymmetricMatch);
      std::swap(start0_start1.first, start0_start1.second);
    }
    IterType start0 = start0_start1.first;
    IterType start1 = start0_start1.second;
    
    // Now we need to assemble a sequence which has enough backspaces
    // to delete those characters of currentWord which need replacing,
    // plus all of the characters of newWord which need sending.
    size_t sequenceLength =
      (currentWord.end() - start0) + (newWord.end() - start1);
    std::vector<KeyDescription> completionSequence(sequenceLength);
    std::vector<KeyDescription>::iterator sequenceIterator =
      completionSequence.begin();
    KeyDescription backspaceCharacter = (m_config.getBackspaceKeys())[0];
    while(start0 < currentWord.end()) {
      *sequenceIterator = backspaceCharacter;
      ++sequenceIterator;
      ++start0;
    }
    while(start1 < newWord.end()) {
      *sequenceIterator = *start1;
      ++sequenceIterator;
      ++start1;
    }

    // Find the appropriate list of pending synthetic keystrokes, and
    // record the ones we're about to send.
    std::deque<KeyDescription>& syntheticKeystrokeQueue =
      m_syntheticKeystrokeQueueMap[windowDescription.getWindowID()];
    std::copy(completionSequence.begin(), completionSequence.end(),
              std::back_inserter(syntheticKeystrokeQueue));
    
    // Now we dispatch to the pure virtual function sendSequence,
    // which knows how to send the characters under whatever windowing
    // system we're using.
    this->sendKeySequence(completionSequence, windowDescription);
  }
  
    
  void
  CompleterApp::
  showSuggestion(const std::vector<KeyDescription>& suggestion)
  {
    std::ostringstream suggestionStream;
    for(size_t index0 = 0; index0 < suggestion.size(); ++index0) {
      suggestionStream << (suggestion[index0]).getRepresentation();
    }
    this->showSuggestion(suggestionStream.str());
  }
  

  void
  CompleterApp::
  startLearningKeystrokes()
  {
    // Side effect (xxx)
    m_learnedKeystrokesVector.clear();
    m_learnKeystrokesFlag = true;
  }


  void
  CompleterApp::
  stopLearningKeystrokes()
  {
    // Side effect (xxx)
    m_learnKeystrokesFlag = false;
  }    
    
    
  bool
  CompleterApp::
  updateSyntheticKeystrokeCount(const KeyDescription& keyDescription,
                                const WindowDescription& windowDescription)
  {
    // Find the appropriate list of pending synthetic keystrokes.
    std::deque<KeyDescription>& syntheticKeystrokeQueue =
      m_syntheticKeystrokeQueueMap[windowDescription.getWindowID()];
  
    // Are there any pending synthetic keystrokes?
    if(syntheticKeystrokeQueue.empty()) {
      return false;
    }

    // If so, pop the next one and return.
    // 
    // Warning(xxx): we should really check to see if keyDescription
    // matches what we're expecting.
    syntheticKeystrokeQueue.pop_front();
    return true;
  }

/* ============== Non-member function declarations ============== */
  

} // namespace xComplete

/* ============ Definitions of inline & template functions ============ */

namespace xComplete {

  // Empty.
  
} // namespace xComplete
