/**
***************************************************************************
* @file dlrUtilities/stringManipulation.h
*
* Header file declaring routines for working with strings.
*
* Copyright (C) 2003-2007, David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 946 $
* $Date: 2007-07-09 20:24:50 -0400 (Mon, 09 Jul 2007) $
***************************************************************************
**/

#ifndef _DLR_STRINGMANIPULATION_H_
#define _DLR_STRINGMANIPULATION_H_

#include <iostream>
#include <sstream>
#include <string>
#include <vector>
#include <dlrCommon/exception.h>
#include <dlrCommon/tags.h>
#include <dlrUtilities/exception.h>

namespace dlr {

  namespace utilities {
    
    /** 
     * This function takes an input string and returns a string in which
     * all special shell characters have been escaped.  The list of
     * special characters is specified as a string, and defaults to the
     * following characters, including the double quote character:
     * "~#$&*()\|[]{};'`<>/?".  The third argument specifies the quote
     * character, and defaults to the backslash character ("\").  The
     * third argument affects how quote characters in the input are
     * handled.  By default, quote characters are escaped just like any
     * other special character.  If the third argument is set to true,
     * then it is assumed that parts of the string have already been
     * quoted, and quote characters are not quoted.  For example:
     *
     *   cleanString("~foo\;bar\b;az", "\;~", '\', false);
     *   will return the string "\~foo\\\;bar\\b\;az",
     *
     * and
     *
     *   cleanString("~foo\;bar\b;az", "\;~", '\', true);
     *   will return the string "\~foo\;bar\b\;az".
     * 
     * @param inputString 
     * @return 
     */
    std::string
    cleanString(const std::string& inputString,
                const std::string& 
                specialCharacters="\"~#$&*()\\|[]{};'`<>/?",
                char quoteCharacter='\\',
                bool alreadyQuoted=false);
  

    /**
     * This function converts the string inputString to Type.  For
     * example:
     *
     *   int intValue = convertString<int>("23");
     *
     * Conversion is done using the stream input operator of the
     * target class.  If the conversion fails, a
     * dlr::utilities::ConversionException is thrown.
     *
     * @param inputString This argument is the string to be converted.
     * 
     * @return The return value an instance of the target type which has
     * had its value set by stream input from the specified string.
     */
    template <class Type>
    Type
    convertString(const std::string& inputString);

  
    /**
     * This function converts the string inputString to Type.  For
     * example:
     *
     *   int intValue = convertString("23", dlr::type_tag<int>());
     *
     * or, using the predefined tag "Int",
     *
     *   int intValue = convertString("23", dlr::Int);
     *
     * Conversion is done using the stream input operator of the
     * target class.  If the conversion fails, a
     * dlr::utilities::ConversionException is thrown.
     *
     * @param inputString This argument is the string to be converted.
     * 
     * @return The return value an instance of the target type which has
     * had its value set by stream input from the specified string.
     */
    template <class Type>
    inline Type
    convertString(const std::string& inputString, type_tag<Type>);

  
    /**
     * This function returns a single string comprising copies of all of
     * the strings in inputStringVector, interposed by the copies of
     * separator.  For example:
     *
     *   joinString(["Hi", "there", "honey", "bear"], "/");
     *
     * will return "Hi/there/honey/bear".
     *
     * @param inputStringVector This argument specifies the strings to
     * be concatenated.
     *
     * @param separator This argument specifies text to be repeated
     * between consecutive elements of inputStringVector.
     *
     * @return The return value is the concatenation of the elements of
     * inputStringVector, with the contents of separator interposed
     * between elements.
     */
    std::string
    joinString(const std::vector<std::string>& inputStringVector,
               const std::string& separator="");
  

    /**
     * This function returns a copy of the input argument in which every
     * uppercase character has been replaced with its lowercase
     * equivalent.  For example:
     *
     *   lowerCaseString("hI TheRe!")
     *
     * will return "hi there!"
     *
     * @param inputString This argument specifies the string which will
     * be converted to lower case.
     *
     * @return The return value is the converted string.
     */
    std::string
    lowerCaseString(const std::string& inputString);
  

    /**
     * This function copies a string, replacing non-overlapping
     * occurrences of a target string with the specified replacement.
     * For example:
     *
     *   replaceString("hohoohooohooooho", "oo", "ii")
     *
     * gives
     *
     *   "hohiihiiohiiiiho"
     * 
     * @param inputString This argument species the string which is to
     * be copied.
     * 
     * @param target This argument specifies the substring which is to
     * be replaced.
     * 
     * @param replacement This argument specifies the string which is to
     * be substituted for each instance of the target.
     * 
     * @return The return value is a string in which all non-overlapping
     * instances of target have been replaced with replacement.
     */
    std::string
    replaceString(const std::string& inputString,
                  const std::string& target,
                  const std::string& replacement);


    /**
     * This function divides inputString around instances of delimiter.
     * For example:
     *
     *   splitString("04//22/2001", "/")
     *
     * gives
     *
     *   ["04", "22", "2001"]
     *
     * and
     *
     *   splitString("04 - 22 - 2001", " - ")
     *
     * gives
     *
     *   ["04", "22", "2001"]
     *
     * Setting includeNullStrings to true makes the return value include
     * even strings of zero length.
     *
     *   splitString("04//22/2001", "/", true)
     *
     * gives
     *
     *   ["04", "", "22", "2001"]
     *
     * and
     *
     *   splitString("hellohehoundhe", "he", true)
     *
     * gives
     *
     *   ["", "llo", "hound", ""]
     * 
     * @param inputString This argument specifies the string which is to
     * be split.
     * 
     * @param delimiter This argument the substring around which
     * inputString should be divided.
     * 
     * @param includeNullStrings This argument specifies whether
     * zero-length substrings should be included in the output.
     *
     * @param maxSplit If this argument is non-zero, it limits the
     * number of splits performed.  If maxSplit is non-zero, then the
     * returned vector of strings will have at most (maxSplit + 1)
     * elements, with the first maxSplit elements being formed by
     * splitting chunks off the beginning of the string, and the final
     * element being the remainder of the string after the first
     * maxSplit splits are performed.  If this argument is zero, then
     * the number of splits will not be limited.
     * 
     * @return The return value is a vector of substrings as described
     * above.
     */
    std::vector<std::string>
    splitString(const std::string& inputString,
                const std::string& delimiter,
                bool includeNullStrings=false,
                size_t maxSplit=0);

  
    /** 
     * This function removes whitespace from the beginning and end of
     * a string.  If the second argument is specified, it indicates which
     * characters should be considered to be whitespace.
     * 
     * @param inputString This is the string to be stripped.
     *
     * @param whiteSpace The characters of this string define what qualifies
     * as whitespace.
     *
     * @return A copy of inputString from which leading and trailing
     * whitespace has been removed.
     */
    std::string
    stripString(const std::string& inputString,
                const std::string& whiteSpace=" \t\n");


    /** 
     * This function returns a copy of the input argument in which every
     * lowercase character has been replaced with its uppercase
     * equivalent.  For example:
     *
     *   upperCaseString("hI TheRe!")
     *
     * will return "HI THERE!"
     * 
     * @param inputString This argument specifies the string which will
     * be converted to upper case.
     * 
     * @return The return value is the converted string.
     */
    std::string
    upperCaseString(const std::string& inputString);


    /** 
     * This function returns a copy of the input argument in which
     * end-of-line markers have been inserted to wrap the string at a
     * specified line length.  Wrapping will only occur at places where
     * there is whitespace.  This means that long sections with no
     * whitespace will not be broken, and may exceed the requested line
     * length.  For example:
     *
     *   wrapString("This is a string to be wrapped.\nYesyesyes indeed it is",
     *              "", 8, " ", "\n")
     *
     * will return
     *
     *   "This is\na string\nto be\nwrapped.\nYesyesyes\nindeed\nit is"
     *
     * and
     *
     *   wrapString("This is a string to be wrapped.\nYesyesyes indeed",
     *              "> ", 10, " ", "\n")
     *
     * will return
     *
     *   "This is a\n> string\n> to be\n> wrapped.\n> Yesyesyes\n> indeed"
     * 
     * @param inputString This argument is the string to be wrapped.
     * 
     * @param fillPrefix This prefix will be inserted immediately after
     * each newline in the output string.
     * 
     * @param width This argument specifies the target line length for
     * the returned string.  Individual lines will only exceed this
     * length if there are words in the input string which have more
     * than width characters.
     *
     * @param whitespace This argument specifies which characters should
     * count as whitespace when breaking inputString.  The string may
     * be broken around an character present in whitespace.
     *
     * @param eolString This argument specifies the end-of-line marker.
     * This is used for two things: its presence is used to identify
     * existing newlines in the input string, and it is inserted into
     * the output string to introduce a newline.
     *
     * @return The return value is a wrapped string.
     */
    std::string
    wrapString(const std::string& inputString,
               const std::string& fillPrefix="",
               size_t width=78,
               const std::string& whitespace=" \t\n",
               const std::string& eolString="\n");


    /** 
     * @deprecated {Please use the 5-argument version of wrapString().}
     *
     * This is a deprecated interface to the wrapString function.  It
     * does not allow specifying a fill prefix, and is retained to
     * support legacy code.
     * 
     * @param inputString This argument is the string to be wrapped.
     * 
     * @param width This argument specifies the target line length for
     * the returned string.  Individual lines will only exceed this
     * length if there are words in the input string which have more
     * than width characters.
     * 
     * @param whitespace This argument specifies which characters should
     * count as whitespace when breaking inputString.  The string may
     * be broken around an character present in whitespace.
     *
     * @param eolString This argument specifies the end-of-line marker.
     * This is used for two things: its presence is used to identify
     * existing newlines in the input string, and it is inserted into
     * the output string to introduce a newline.
     *
     * @return The return value is a wrapped string.
     */
    inline std::string
    wrapString(const std::string& inputString,
               size_t width,
               const std::string& whitespace=" \t\n",
               const std::string& eolString="\n") {
      return wrapString(inputString, "", width, whitespace, eolString);
    }

  } // namespace utilities
  
} // namespace dlr


/* ******************************************************************
 * Declarations to maintain compatibility with legacy code.
 * ***************************************************************** */

namespace dlr {

  using utilities::cleanString;
  using utilities::convertString;
  using utilities::joinString;
  using utilities::lowerCaseString;
  using utilities::replaceString;
  using utilities::splitString;
  using utilities::stripString;
  using utilities::upperCaseString;
  using utilities::wrapString;

} // namespace dlr


/* ******************************************************************
 * Definitions of inline and template functions.
 * ***************************************************************** */

namespace dlr {

  namespace utilities {
    
    // Converts the string inputString to Type.
    template <class Type>
    Type
    convertString(const std::string& inputString)
    {
      Type returnValue;
      std::istringstream inputStream(inputString);
      inputStream >> returnValue;
      if(!inputStream) {
        std::ostringstream message;
        message << "Can't convert malformed input string: " << inputString
                << std::endl;
        DLR_THROW(ConversionException, "convertString()", message.str().c_str());
      }
      return returnValue;
    }

  
    // Converts the string inputString to Type.
    template <class Type>
    inline Type
    convertString(const std::string& inputString, type_tag<Type>)
    {
      return convertString<Type>(inputString);
    }

  } // namespace utilities
  
} // namespace dlr

#endif // #ifndef _DLR_STRINGMANIPULATION_H_
