/**
***************************************************************************
* @file dlrUtilities/optionParser.cpp
* Source file defining the OptionParser class.
*
* Copyright (C) 2006 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 706 $
* $Date: 2006-08-04 19:41:11 -0400 (Fri, 04 Aug 2006) $
***************************************************************************
**/

#include <algorithm>
#include <sstream>
#include <dlrCommon/exception.h>
#include <dlrUtilities/optionParser.h>
#include <dlrUtilities/stringManipulation.h>

namespace dlr {

  namespace utilities {
  
    // Default constructor.
    OptionParser::
    OptionParser(bool allowExtraArguments,
                 bool allowStackedShortOptions,
                 bool allowOptionishArguments)
      : m_allowExtraArguments(allowExtraArguments),
        m_allowOptionishArguments(allowOptionishArguments),
        m_allowStackedShortOptions(allowStackedShortOptions),
        m_errorBehavior(ThrowOnError),
        m_exitCode(0),
        m_extraArgumentValues(),
        m_handleHelp(false),
        m_numberOfPosArgRequired(0),
        m_numberOfPosArgParsed(0),
        m_optionCounts(),
        m_optionDescriptions(),
        m_optionValues(),
        m_positionalArgumentNames(),
        m_positionalArgumentDefaultValues(),
        m_positionalArgumentDocs(),
        m_programName()
    {
      // Empty.
    }


    // Constructor.
    OptionParser::
    OptionParser(int exitCode,
                 bool handleMinusMinusHelp,
                 bool printUsage,
                 bool allowExtraArguments,
                 bool allowStackedShortOptions,
                 bool allowOptionishArguments)
      : m_allowExtraArguments(allowExtraArguments),
        m_allowOptionishArguments(allowOptionishArguments),
        m_allowStackedShortOptions(allowStackedShortOptions),
        m_errorBehavior(printUsage ? UsageAndExitOnError : ExitOnError),
        m_exitCode(exitCode),
        m_extraArgumentValues(),
        m_handleHelp(handleMinusMinusHelp),
        m_numberOfPosArgRequired(0),
        m_numberOfPosArgParsed(0),
        m_optionCounts(),
        m_optionDescriptions(),
        m_optionValues(),
        m_positionalArgumentNames(),
        m_positionalArgumentDefaultValues(),
        m_positionalArgumentDocs(),
        m_programName()
    {
      // Empty.
    }

    
    // Destructor.
    OptionParser::
    ~OptionParser()
    {
      // Empty.
    }


    void
    OptionParser::
    addOption(const std::string& name,
              const std::string& shortVersion,
              const std::string& longVersion,
              const std::string& docString,
              bool allowPartialMatch)
    {
      if(m_optionDescriptions.find(name) != m_optionDescriptions.end()) {
        std::ostringstream message;
        message << "Option \"" << name << "\" has already been specified.";
        DLR_THROW(StateException, "OptionParser::addOption()",
                  message.str().c_str());
      }
      OptionDescription newOption(
        name, shortVersion, longVersion, false, allowPartialMatch, docString);
      m_optionDescriptions[name] = newOption;
      m_optionCounts[name] = 0;
    }


    void
    OptionParser::
    addOptionWithValue(const std::string& name,
                       const std::string& shortVersion,
                       const std::string& longVersion,
                       const std::string& defaultValue,
                       const std::string& docString,
                       bool requireArgument,
                       bool allowPartialMatch,
                       bool allowOptionishValue)
    {
      if(requireArgument == false) {
        DLR_THROW(NotImplementedException,
                  "OptionParser::addOptionWithValue()",
                  "Options with optional arguments not yet supported.");
      }

      if(m_optionDescriptions.find(name) != m_optionDescriptions.end()) {
        std::ostringstream message;
        message << "Option \"" << name << "\" has already been specified.";
        DLR_THROW(StateException, "OptionParser::addOptionWithValue()",
                  message.str().c_str());
      }
      OptionDescription newOption(
        name, shortVersion, longVersion, true, allowPartialMatch, docString,
        defaultValue);
      m_optionDescriptions[name] = newOption;
      m_optionCounts[name] = 0;
    }

    
    void
    OptionParser::
    addPositionalArgument(const std::string& name,
                          const std::string& docString,
                          bool isRequired,
                          const std::string& defaultValue)
    {
      m_positionalArgumentNames.push_back(name);
      m_positionalArgumentDocs.push_back(docString);
      m_positionalArgumentDefaultValues.push_back(defaultValue);
      if(isRequired) {
        m_numberOfPosArgRequired = m_positionalArgumentNames.size();
      }
    }

    
    size_t
    OptionParser::
    getCount(const std::string& name) const
    {
      std::map<std::string, size_t>::const_iterator iter =
        m_optionCounts.find(name);
      if(iter == m_optionCounts.end()) {
        std::ostringstream message;
        message << "Invalid option name: " << name << ".";
        DLR_THROW(ValueException, "OptionParser::getCount()",
                  message.str().c_str());
      }
      return iter->second;
    }

    
    std::vector<std::string>
    OptionParser::
    getExtraPositionalArguments()
    {
      return m_extraArgumentValues;
    }

    
    std::string
    OptionParser::
    getOptionsDescription()
    {
      std::ostringstream optionsStream;
      if(!m_optionDescriptions.empty()) {
        std::map<std::string, OptionDescription>::const_iterator iter =
          m_optionDescriptions.begin();
        while(iter != m_optionDescriptions.end()) {
          optionsStream << iter->second << "\n";
          ++iter;
        }
      }
      return optionsStream.str();
    }

    
    std::string
    OptionParser::
    getUsage()
    {
      std::string indentString = "        ";
      std::ostringstream usageStream;
      usageStream << "Usage:\n"
                  << "  " << m_programName << " ";
      if(!m_optionDescriptions.empty()) {
        usageStream << "[options] ";
      }
      size_t argIndex = 0;
      while(argIndex < m_numberOfPosArgRequired) {
        usageStream << m_positionalArgumentNames[argIndex] << " ";
        ++argIndex;
      }
      while(argIndex < m_positionalArgumentNames.size()) {
        usageStream << "[" << m_positionalArgumentNames[argIndex] << "] ";
        ++argIndex;
      }
      if(!m_positionalArgumentNames.empty()) {
        usageStream << "\n\nPositional arguments:\n";
        for(argIndex = 0; argIndex < m_numberOfPosArgRequired; ++argIndex) {
          usageStream
            << "  " << m_positionalArgumentNames[argIndex] << "\n"
            << wrapString(indentString + m_positionalArgumentDocs[argIndex],
                          indentString)
            << "\n";
        }
        while(argIndex < m_positionalArgumentNames.size()) {
          usageStream
            << "  " << m_positionalArgumentNames[argIndex] << "\n"
            << wrapString(indentString + m_positionalArgumentDocs[argIndex],
                          indentString) << "\n"
            << wrapString((indentString + "Default value: \""
                           + m_positionalArgumentDefaultValues[argIndex]),
                          indentString) << "\""
            << "\n";
          ++argIndex;          
        }
      }

      if(!m_optionDescriptions.empty()) {
        usageStream << "\nOptions:\n"
                    << this->getOptionsDescription();
      }
      
      return usageStream.str();
    }
    
    
    std::string
    OptionParser::
    getValue(const std::string& name)
    {
      return this->getValue(name, -1);
    }

    
    std::string
    OptionParser::
    getValue(const std::string& name, int valueIndex)
    {
      // Look for the specified name in the recorded option/argument values.
      std::map< std::string, std::vector<std::string> >::const_iterator vIter =
        m_optionValues.find(name);
      if(vIter != m_optionValues.end()) {
        // Found an appropriate entry in the recorded values.
        if(valueIndex >= static_cast<int>((vIter->second).size())) {
          std::ostringstream message;
          message << "Option/argument " << name << " was specified "
                  << (vIter->second).size() << " times, "
                  << "yet getValue() was called with valueIndex set to "
                  << valueIndex << ".";
          DLR_THROW(IndexException, "OptionParser::getValue()",
                    message.str().c_str());
        }
        if(valueIndex < 0
           || valueIndex >= static_cast<int>((vIter->second).size())) {
          valueIndex = (vIter->second).size() - 1;
        }
        return (vIter->second)[valueIndex];
      }

      // Nothing appropriate found in the recorded values.  Check to
      // see if the specified name corresponds to an option, and if
      // so, return the default value for that option.
      std::map<std::string, OptionDescription>::const_iterator odIter =
        m_optionDescriptions.find(name);
      if(odIter != m_optionDescriptions.end()) {
        return (odIter->second).getDefaultValue();
      }

      // No appropriate option found.  Check to see if the specified
      // name corresponds to a positional argument, and if so, return
      // the default value for that positional argument.
      std::vector<std::string>::const_iterator panIter =
        std::find(m_positionalArgumentNames.begin(),
                  m_positionalArgumentNames.end(),
                  name);
      if(panIter != m_positionalArgumentNames.end()) {
        size_t argIndex = panIter - m_positionalArgumentNames.begin();
        return m_positionalArgumentDefaultValues[argIndex];
      }
      
      // // No appropriate positional argument found.  Return the empty
      // // string.
      // return "";

      // No appropriate positional argument found.
      std::ostringstream message;
      message << "The value of Option/argument " << name << " was requested, "
              << "but no option or argument with this name exists.";
      DLR_THROW(ValueException, "OptionParser::getValue()",
                message.str().c_str());
    }

    
    void
    OptionParser::
    parseCommandLine(int argc, char* argv[])
    {
      // An enum representing the possible states of the parser below.
      enum ParseState {
        DLR_OP_GETTING_NEW_ARG,
        DLR_OP_IDENTIFYING_ARG,
        DLR_OP_RECORDING_POS_ARG,
        DLR_OP_RECORDING_OPTION,
        DLR_OP_RECORDING_VALUE,
        DLR_OP_FINISHING_SHORT_OPTION,
        DLR_OP_FINISHING_LONG_OPTION,
        DLR_OP_FINISHED,
        DLR_OP_ERROR
      };

      if(argc < 1) {
        DLR_THROW(ValueException, "OptionParser::parseCommandLine()",
                  "Argument argc must be greater than or equal to 1.");
      }

      // Reset state for new parse.
      m_optionValues.clear();
      std::map<std::string, size_t>::iterator iter = m_optionCounts.begin(); 
      while(iter != m_optionCounts.end()) {
        iter->second = 0;
        ++iter;
      }
      m_extraArgumentValues.clear();

      // Record "special" arguments.
      m_programName = argv[0];

      // Parse the remaining command line, catching any errors.
      try {
        int argIndex = 1;
        std::string currentArgument = "";
        OptionDescription optionDescription;
        size_t typedLength;
        bool isShortMatch;
        std::ostringstream errorMessage;

        ParseState currentState = DLR_OP_GETTING_NEW_ARG;
        while(currentState != DLR_OP_FINISHED) {
          switch(currentState) {
          case DLR_OP_GETTING_NEW_ARG:
            if(argIndex >= argc) {
              currentState = DLR_OP_FINISHED;
            } else {
              currentArgument = argv[argIndex];
              ++argIndex;
              currentState = DLR_OP_IDENTIFYING_ARG;
            }
            break;
          case DLR_OP_IDENTIFYING_ARG:
            // Special case for --help arguments.
            if(m_handleHelp && (currentArgument == "--help")) {
              std::cout << this->getUsage() << std::endl;
              exit(0);
            }

            // See if any of the user-specified options match.
            {
              bool optionFound = this->findOptionDescription(
                currentArgument, optionDescription, typedLength, isShortMatch);
              if(!optionFound) {
                currentState = DLR_OP_RECORDING_POS_ARG;
              } else {
                currentState = DLR_OP_RECORDING_OPTION;
              }
            }
            break;
          case DLR_OP_RECORDING_POS_ARG:
            // This argument doesn't match any of our options.  It must
            // be a positional argument.
            if(!m_allowOptionishArguments && currentArgument[0] == '-') {
              errorMessage << "Unrecognized option \"" << currentArgument
                           << "\".";
              currentState = DLR_OP_ERROR;
            } else {
              this->recordPositionalArgument(currentArgument);
              currentState = DLR_OP_GETTING_NEW_ARG;
            }
            break;
          case DLR_OP_RECORDING_OPTION:
            // Increment the count for this option.
            ++((this->m_optionCounts.find(optionDescription.getName()))
               ->second);

            // And make sure we do the appropriate followup.
            if(optionDescription.requiresValue()) {
              currentState = DLR_OP_RECORDING_VALUE;
            } else if(isShortMatch) {
              currentState = DLR_OP_FINISHING_SHORT_OPTION;
            } else {
              currentState = DLR_OP_FINISHING_LONG_OPTION;
            }
            break;
          case DLR_OP_RECORDING_VALUE:
            // This argument does match one of our options.
            if(typedLength < currentArgument.size()) {
              this->m_optionValues[optionDescription.getName()].push_back(
                currentArgument.substr(typedLength, std::string::npos));
              currentState = DLR_OP_GETTING_NEW_ARG;
            } else if(argIndex < argc) {
              this->m_optionValues[optionDescription.getName()].push_back(
                argv[argIndex]);
              ++argIndex;
              currentState = DLR_OP_GETTING_NEW_ARG;
            } else {
              errorMessage << "Missing value for option \"" << currentArgument
                           << "\".";
              currentState = DLR_OP_ERROR;
            }
            break;
          case DLR_OP_FINISHING_SHORT_OPTION:
            if(typedLength >= currentArgument.size()) {
              currentState = DLR_OP_GETTING_NEW_ARG;
            } else if(m_allowStackedShortOptions
                      && typedLength >= 2
                      && currentArgument[0] == '-') {
              currentArgument =
                "-" + currentArgument.substr(typedLength, std::string::npos);
              currentState = DLR_OP_IDENTIFYING_ARG;                
            } else {
              errorMessage
                << "Option \""
                << currentArgument.substr(0, typedLength)
                << "\" does not take a value, yet was specified as \""
                << currentArgument << ".";
              currentState = DLR_OP_ERROR;
            }
            break;
          case DLR_OP_FINISHING_LONG_OPTION:
            if(typedLength >= currentArgument.size()) {
              currentState = DLR_OP_GETTING_NEW_ARG;
            } else {
              errorMessage
                << "Option \""
                << currentArgument.substr(0, typedLength)
                << "\" does not take a value, yet was specified as \""
                << currentArgument << ".";
              currentState = DLR_OP_ERROR;
            }
            break;
          case DLR_OP_ERROR:
            DLR_THROW(IOException, "OptionParser::parseCommandLine()",
                      errorMessage.str().c_str());
            break;
          default:
            // Should never get here.
            break;
          }
        }

        // Make sure we got the right number of positional arguments.
        if(m_numberOfPosArgParsed < m_numberOfPosArgRequired) {
          std::ostringstream message;
          message << "Expected " << m_numberOfPosArgRequired
                  << " positional argument(s), but only found "
                  << m_numberOfPosArgParsed << ".";
          DLR_THROW(IOException, "OptionParser::parseCommandLine()",
                    message.str().c_str());                      
        }
        if(!m_allowExtraArguments && (m_extraArgumentValues.size() != 0)) {
          std::ostringstream message;
          message << "Expected no more than "
                  << m_positionalArgumentNames.size()
                  << " positional arguments, but found "
                  << m_numberOfPosArgParsed << ".";
          DLR_THROW(IOException, "OptionParser::parseCommandLine()",
                    message.str().c_str());                      
        }

      } catch(const IOException& caughtException) {
        switch(m_errorBehavior) {
        case UsageAndExitOnError:
          // Print diagnostic info and usage, then fall through into
          // ExitOnError code.
          std::cout << caughtException.what() << "\n\n";
          std::cout << this->getUsage() << std::endl;
        case ExitOnError:
          // Exit with the user specified error code.
          exit(m_exitCode);
          break;
        case ThrowOnError:
        default:
          // Default behavior is to rethrow the exception.
          throw;
          break;
        }
      }
    }
    

    bool
    OptionParser::
    findOptionDescription(const std::string& argument,
                          OptionDescription& optionDescription,
                          size_t& typedLength,
                          bool& isShortMatch)
    {
      // Ugly algorithm here makes arg parsing be O(N^2) where N the
      // number of args.  For now, the 80/20 rule dictates that we not
      // sweat about it.
      typedLength = 0;
      bool foundOption = false;
      std::map<std::string, OptionDescription>::const_iterator iter =
        m_optionDescriptions.begin();
      while(iter != m_optionDescriptions.end()) {
        bool localIsShortMatch;
        size_t localTypedLength =
          (iter->second).getMatchLength(argument, localIsShortMatch);
        if(localTypedLength != 0) {
          if(foundOption) {
            std::ostringstream message;
            message << "Argument \"" << argument << "\" is ambiguous.";
            DLR_THROW(IOException, "OptionParser::findOptionDescription()",
                      message.str().c_str());                      
          } else {
            typedLength = localTypedLength;
            foundOption = true;
            optionDescription = iter->second;
            isShortMatch = localIsShortMatch;
          }
        }
        ++iter;
      }
      return foundOption;
    }


    void
    OptionParser::
    recordPositionalArgument(const std::string& argument)
    {
      if(m_numberOfPosArgParsed < m_positionalArgumentNames.size()) {
        std::string argumentName =
          m_positionalArgumentNames[m_numberOfPosArgParsed];
        m_optionValues[argumentName].push_back(argument);
      } else {
        m_extraArgumentValues.push_back(argument);
      }
      ++m_numberOfPosArgParsed;
    }


  } // namespace utilities

} // namespace dlr
