/**
***************************************************************************
* @file monitorTest.cpp
* Source file defining tests for dlr::thread::Monitor.
*
* Copyright (C) 2006 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 925 $
* $Date: 2007-05-21 01:49:01 -0400 (Mon, 21 May 2007) $
***************************************************************************
**/

#include <string>

#include <dlrPortability/timeUtilities.h>
#include <dlrRandom/pseudoRandom.h>
#include <dlrTest/testFixture.h>
#include <dlrThread/monitor.h>
#include <dlrThread/threadFunctor.h>

using namespace dlr::thread;
using dlr::portability::getCurrentTime;
using dlr::portability::portableSleep;

namespace dlr {


  class MonitorTest : public TestFixture<MonitorTest> {

  public:

    MonitorTest();
    ~MonitorTest() {}

    void setUp(const std::string& testName) {}
    void tearDown(const std::string& testName) {}

    // Tests of member functions.
    void testNormalFunction0();

  }; // class MonitorTest


  class TestMonitor0
    : public Monitor
  {
  public:
    TestMonitor0()
      : Monitor(),
        m_flag0(false),
		m_flag0False(Monitor::createCondition()),
        m_tokenAssigned(false) {}

    int
    testMethod0(const std::string& threadName,
                double sleepTime, int verbosity=0) {
      Token token = this->getToken();
      this->verboseWrite(threadName + ": Entered0...\n", 2, verbosity);
      if(m_tokenAssigned) {
        this->verboseWrite(threadName + ": Token error...\n", 1, verbosity);
        m_tokenAssigned = false;
        return 1;
      }
      m_tokenAssigned = true;
      
      if(m_flag0) {
        this->verboseWrite(threadName + ": Waiting...\n", 2, verbosity);
        m_tokenAssigned = false;
        this->wait(m_flag0False, token);
        if(m_tokenAssigned) {
          this->verboseWrite(threadName + ": Token error2...\n", 1, verbosity);
          m_tokenAssigned = false;
          return 3;
        }
        m_tokenAssigned = true;
        this->verboseWrite(threadName + ": Awake...\n", 2, verbosity);
        if(m_flag0) {
          this->verboseWrite(threadName + ": Flag error...\n", 1, verbosity);
          m_tokenAssigned = false;
          return 2;
        }
      }
      this->verboseWrite(threadName + ": Sleeping0...\n", 2, verbosity);
      portableSleep(sleepTime);
      m_flag0 = true;
      this->verboseWrite(threadName + ": Leaving0...\n", 2, verbosity);
      m_tokenAssigned = false;
      return 0;
    }

    int
    testMethod1(const std::string& threadName,
                double sleepTime, int verbosity=0) {
      Token token = this->getToken();
      this->verboseWrite(threadName + ": Entered1...\n", 2, verbosity);
      if(m_tokenAssigned) {
        this->verboseWrite(threadName + ": Token error...\n", 1, verbosity);
        m_tokenAssigned = false;
        return 1;
      }
      m_tokenAssigned = true;
      
      portableSleep(sleepTime);
      m_flag0 = false;
      this->signalOne(m_flag0False);
      this->verboseWrite(threadName + ": Leaving1...\n", 2, verbosity);
      m_tokenAssigned = false;
      return 0;
    }

    void
    verboseWrite(const std::string& message, int threshold, int verbosity) {
      if(verbosity >= threshold) {
        std::cout << message << std::flush;
      }
    }
        
  private:
    bool m_flag0;
    Condition m_flag0False;
    bool m_tokenAssigned;
  };


  enum ThreadExitStatus {DLR_MT_NOSTATUS,
                         DLR_MT_NORMAL,
                         DLR_MT_EXCEPTION};
                         
  
  class TestThreadFunctor
    : public ThreadFunctor
  {
  public:
    TestThreadFunctor(const std::string& threadName,
                      TestMonitor0& monitor0,
                      size_t numberOfIterations,
                      double minSleep,
                      double maxSleep,
                      int verbosity)
      : ThreadFunctor(threadName),
        m_monitor(monitor0),
        m_maxSleep(maxSleep),
        m_minSleep(minSleep),
        m_numberOfIterations(numberOfIterations),
        m_pseudoRandom(),
        m_verbosity(verbosity),
        m_exitStatus(DLR_MT_NOSTATUS) {
      // Sleep for a millisecond so that subsequent TestThreadFunctor
      // instances will get a different seed for m_pseudoRandom.
      portableSleep(0.001);
    }

    
    ThreadExitStatus
    getExitStatus() {return m_exitStatus;}

    virtual void
    main() {
      try {
        this->testMain();
        this->m_exitStatus = DLR_MT_NORMAL;
      } catch(const dlr::Exception& caughtException) {
	if(m_verbosity > 0) {
	  std::cout << "[" << this->getThreadName() << "] "
		    << caughtException.what() << std::endl;
	}
        this->m_exitStatus = DLR_MT_EXCEPTION;
      }
    }
    
    virtual void
    testMain() = 0;
    
  protected:
    TestMonitor0& m_monitor;
    double m_maxSleep;
    double m_minSleep;
    size_t m_numberOfIterations;
    PseudoRandom m_pseudoRandom;
    int m_verbosity;
    
    ThreadExitStatus m_exitStatus;
  };


  
  class NormalTest0ThreadFunctor
    : public TestThreadFunctor
  {
  public:
    NormalTest0ThreadFunctor(
      const std::string& threadName,
      TestMonitor0& monitor0,
      size_t numberOfIterations,
      double minSleep,
      double maxSleep,
      int verbosity=0)
      : TestThreadFunctor(threadName, monitor0, numberOfIterations,
                          minSleep, maxSleep, verbosity) {}
        
    virtual void
    testMain() {
      for(size_t index0 = 0; index0 < m_numberOfIterations; ++index0) {
        double sleepTime = m_pseudoRandom.uniform(m_minSleep, m_maxSleep);
	int returnValue = m_monitor.testMethod0(m_threadName, sleepTime, 0);
        if(returnValue != 0) {
	  std::ostringstream message;
	  message << "m_monitor.testMethod0() returned " << returnValue
		  << ".";
          DLR_THROW(ValueException, "NormalTest0ThreadFunctor::testMain()",
		    message.str().c_str());
        }
        portableSleep(0.001);
	returnValue = m_monitor.testMethod1(m_threadName, sleepTime, 0);
        if(returnValue != 0) {
	  std::ostringstream message;
	  message << "m_monitor.testMethod1() returned " << returnValue
		  << ".";
          DLR_THROW(ValueException, "NormalTest0ThreadFunctor::testMain()",
		    message.str().c_str());
        }
        portableSleep(0.001);
      }
    }

  };


  /* ============== Member Function Definititions ============== */
    
  MonitorTest::
  MonitorTest()
    : TestFixture<MonitorTest>("MonitorTest")
  {
    DLR_TEST_REGISTER_MEMBER(testNormalFunction0);
  }


  void
  MonitorTest::
  testNormalFunction0()
  {
    TestMonitor0 testMonitor0;
    NormalTest0ThreadFunctor thread0(
      "Thread0", testMonitor0, 50, 0.001, 0.01, 1);
    NormalTest0ThreadFunctor thread1(
      "Thread1", testMonitor0, 50, 0.001, 0.01, 1);
    NormalTest0ThreadFunctor thread2(
      "Thread2", testMonitor0, 50, 0.001, 0.01, 1);
    NormalTest0ThreadFunctor thread3(
      "Thread3", testMonitor0, 50, 0.001, 0.01, 1);

    thread0.run();
    thread1.run();
    thread2.run();
    thread3.run();
    thread3.join();
    thread2.join();
    thread1.join();
    thread0.join();

    DLR_TEST_ASSERT(thread0.getExitStatus() == DLR_MT_NORMAL);
    DLR_TEST_ASSERT(thread1.getExitStatus() == DLR_MT_NORMAL);
    DLR_TEST_ASSERT(thread2.getExitStatus() == DLR_MT_NORMAL);
    DLR_TEST_ASSERT(thread3.getExitStatus() == DLR_MT_NORMAL);
  }

} // namespace dlr


#if 0

int main(int argc, char** argv)
{
  dlr::MonitorTest currentTest;
  bool result = currentTest.run();
  return (result ? 0 : 1);
}

#else

namespace {

  dlr::MonitorTest currentTest;

}

#endif
