/**
***************************************************************************
* @file distributionQueueTest.cpp
* Source file defining tests for dlr::thread::DistributionQueue.
*
* Copyright (C) 2006 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 706 $
* $Date: 2006-08-04 19:41:11 -0400 (Fri, 04 Aug 2006) $
***************************************************************************
**/

#include <string>

#include <dlrPortability/timeUtilities.h>
#include <dlrRandom/pseudoRandom.h>
#include <dlrTest/testFixture.h>
#include <dlrThread/distributionQueue.h>
#include <dlrThread/threadFunctor.h>

using namespace dlr::thread;
using dlr::portability::getCurrentTime;
using dlr::portability::portableSleep;

namespace dlr {


  class DistributionQueueTest : public TestFixture<DistributionQueueTest> {

  public:

    DistributionQueueTest();
    ~DistributionQueueTest() {}

    void setUp(const std::string& testName) {}
    void tearDown(const std::string& testName) {}

    // Tests of member functions.
    void testNormalFunction0();
    void testNormalFunction1();

  }; // class DistributionQueueTest


  enum ThreadExitStatus {DQT_NOSTATUS,
                         DQT_NORMAL,
                         DQT_OVERFLOWEXCEPTION,
                         DQT_STATEEXCEPTION};
                         
  
  class SourceSink
    : public Monitor {
  public:
    SourceSink(size_t size, int verbosity = 0)
      : m_flags(size, false), m_index(0), m_verbosity(verbosity) {}
    ~SourceSink() {}

    bool
    isComplete() {
      bool completeFlag =
	(std::count(m_flags.begin(), m_flags.end(), false) == 0);
      if(!completeFlag) {
	if(m_verbosity > 0) {
	  for(size_t index0 = 0; index0 < m_flags.size(); ++index0) {
	    if(m_flags[index0] != true) {
	      std::cout << "[SourceSink] Flag #" << index0 << " is false."
			<< std::endl;
	    }
	  }
	}
      }
      return completeFlag;
    }


    size_t
    getSize() {return m_flags.size();}
    
    
    bool
    source(unsigned int& item) {
      Token token = this->getToken();
      if(m_index == m_flags.size()) {
        return false;
      }
      item = m_index++;
      return true;
    }

    
    void
    sink(unsigned int item) {
      if(item >= m_flags.size()) {
        DLR_THROW(ValueException, "SourceSink::sink()",
                  "Invalid item number.");
      }
      if(m_flags[item] == true) {
        DLR_THROW(ValueException, "SourceSink::sink()",
                  "Duplicate item number.");
      }
      m_flags[item] = true;
    }

  private:
    std::vector<bool> m_flags;
    unsigned int m_index;
    int m_verbosity;
  };


  class TestThreadFunctor
    : public ThreadFunctor
  {
  public:
    TestThreadFunctor(
      const std::string& threadName,
      const DistributionQueue< std::pair<size_t, double> >& distributionQueue0,
      SourceSink& sourceSink,
      double minSleep,
      double maxSleep,
      int verbosity)
      : ThreadFunctor(threadName),
        m_distributionQueue(distributionQueue0),
        m_sourceSink(sourceSink),
        m_maxSleep(maxSleep),
        m_minSleep(minSleep),
        m_pseudoRandom(),
        m_verbosity(verbosity),
        m_exitStatus(DQT_NOSTATUS) {
      // Sleep for a millisecond so that subsequent TestThreadFunctor
      // instances will get a different seed for m_pseudoRandom.
      portableSleep(0.001);
    }

    
    ThreadExitStatus
    getExitStatus() {return m_exitStatus;}

    virtual void
    main() {
      try {
        this->testMain();
        this->m_exitStatus = DQT_NORMAL;
      } catch(const dlr::thread::OverflowException& caughtException) {
        if(m_verbosity) {
          std::cout << "[TestThreadFunctor] Caught an exception:\n"
                    << caughtException.what() << "\n" << std::endl;
        }
        this->m_exitStatus = DQT_OVERFLOWEXCEPTION;
      } catch(const StateException& caughtException) {
        if(m_verbosity) {
          std::cout << "[TestThreadFunctor] Caught an exception:\n"
                    << caughtException.what() << "\n" << std::endl;
        }
        this->m_exitStatus = DQT_STATEEXCEPTION;
      }
    }
    
    virtual void
    testMain() = 0;
    
  protected:
    DistributionQueue< std::pair<size_t, double> > m_distributionQueue;
    SourceSink& m_sourceSink;
    double m_maxSleep;
    double m_minSleep;
    PseudoRandom m_pseudoRandom;
    int m_verbosity;
    
    ThreadExitStatus m_exitStatus;
  };


  
  class ProducerThreadFunctor
    : public TestThreadFunctor
  {
  public:
    ProducerThreadFunctor(
      const std::string& threadName,
      const DistributionQueue< std::pair<size_t, double> >& distributionQueue0,
      SourceSink& sourceSink,
      double minSleep,
      double maxSleep,
      int verbosity=0)
      : TestThreadFunctor(threadName, distributionQueue0, sourceSink,
                          minSleep, maxSleep, verbosity) {}
        
    virtual void
    testMain() {
      DistributionQueue< std::pair<size_t, double> >::ClientID clientID;
      m_distributionQueue.registerClient(clientID, m_threadName);

      unsigned int item;
      while(m_sourceSink.source(item)) {
        m_distributionQueue.pushFront(
          clientID, std::make_pair(item, getCurrentTime()));
        double sleepTime = m_pseudoRandom.uniform(m_minSleep, m_maxSleep);
        portableSleep(sleepTime);
      }
    }
  };


  class ConsumerThreadFunctor
    : public TestThreadFunctor
  {
  public:
    ConsumerThreadFunctor(
      const std::string& threadName,
      const DistributionQueue< std::pair<size_t, double> >& distributionQueue0,
      SourceSink& sourceSink,
      double minSleep,
      double maxSleep,
      size_t numberOfConsumers,
      int verbosity=0)
      : TestThreadFunctor(threadName, distributionQueue0, sourceSink,
                          minSleep, maxSleep, verbosity),
        m_numberOfConsumers(numberOfConsumers) {}
    
    virtual void
    testMain() {
      DistributionQueue< std::pair<size_t, double> >::ClientID clientID;
      m_distributionQueue.registerClient(clientID, m_threadName);
      
      std::pair<size_t, double> elementValue;
      while(1) {
        // Read next packet.
        // double timeout = 10 * m_numberOfConsumers * m_maxSleep;
	double timeout = 60.0;
        if(m_distributionQueue.bufferBack(clientID, timeout) == false) {
          DLR_THROW(StateException, "ConsumerThreadFunctor::testMain()",
                    "Producer appears to have died.");
        }
        m_distributionQueue.copyBuffer(clientID, elementValue);
        m_sourceSink.sink(static_cast<unsigned int>(elementValue.first));
        if(elementValue.first >=
           (m_sourceSink.getSize() - m_numberOfConsumers)) {
          break;
        }
        this->adaptiveSleep(elementValue.second);
      }
    }
    
    
    void
    adaptiveSleep(double elementTimestamp) {
      // Adjust sleep times to make sure we don't fall too far
      // behind in the queue.
      double elementAge = getCurrentTime() - elementTimestamp;
      double queueLengthBound = elementAge / m_minSleep;
      double slackTime =
        (m_distributionQueue.getMaximumLength() - queueLengthBound - 1)
        * m_minSleep;
      if(slackTime < m_minSleep) {
        slackTime = m_minSleep;
      }
      double safeMaxSleepTime = std::min(m_maxSleep, slackTime);
      double safeMinSleepTime = m_minSleep;

      // Sleep.
      double sleepTime =
        m_pseudoRandom.uniform(safeMinSleepTime, safeMaxSleepTime);
      if(m_verbosity >= 2) {
        std::cout << m_threadName << ": sleep(" << sleepTime << ")"
                  << std::endl;
      }
      portableSleep(sleepTime);
    }

  private:
    size_t m_numberOfConsumers;
    bool m_sleepWhileLocking;
    
  };


  /* ============== Member Function Definititions ============== */
    
  DistributionQueueTest::
  DistributionQueueTest()
    : TestFixture<DistributionQueueTest>("DistributionQueueTest")
  {
    DLR_TEST_REGISTER_MEMBER(testNormalFunction0);
    DLR_TEST_REGISTER_MEMBER(testNormalFunction1);
  }


  void
  DistributionQueueTest::
  testNormalFunction0()
  {
    int verbosity = 0;
    DistributionQueue< std::pair<size_t, double> > distributionQueue(10, 2);
    SourceSink sourceSink(100);
    ConsumerThreadFunctor consumerThread(
      "Consumer0", distributionQueue, sourceSink, 0.01, 0.1, 1, verbosity);
    ProducerThreadFunctor producerThread(
      "Producer0", distributionQueue, sourceSink, 0.01, 0.1, verbosity);

    consumerThread.run();
    producerThread.run();
    producerThread.join();
    consumerThread.join();

    DLR_TEST_ASSERT(sourceSink.isComplete());
    DLR_TEST_ASSERT(consumerThread.getExitStatus() == DQT_NORMAL);
    DLR_TEST_ASSERT(producerThread.getExitStatus() == DQT_NORMAL);
  }


  void
  DistributionQueueTest::
  testNormalFunction1()
  {
    int verbosity = 1;
    DistributionQueue< std::pair<size_t, double> > distributionQueue(10, 2);
    SourceSink sourceSink(1000, 1);
    ConsumerThreadFunctor consumer0Thread(
      "Consumer0", distributionQueue, sourceSink, 0.001, 0.01, 4,
      verbosity);
    ConsumerThreadFunctor consumer1Thread(
      "Consumer1", distributionQueue, sourceSink, 0.001, 0.01, 4,
      verbosity);
    ConsumerThreadFunctor consumer2Thread(
      "Consumer2", distributionQueue, sourceSink, 0.001, 0.01, 4,
      verbosity);
    ConsumerThreadFunctor consumer3Thread(
      "Consumer3", distributionQueue, sourceSink, 0.001, 0.01, 4,
      verbosity);
    ProducerThreadFunctor producerThread(
      "Producer0", distributionQueue, sourceSink, 0.001, 0.01, verbosity);

    consumer0Thread.run();
    consumer1Thread.run();
    consumer2Thread.run();
    consumer3Thread.run();
    producerThread.run();
    producerThread.join();
    consumer3Thread.join();
    consumer2Thread.join();
    consumer1Thread.join();
    consumer0Thread.join();

    DLR_TEST_ASSERT(consumer0Thread.getExitStatus() == DQT_NORMAL);
    DLR_TEST_ASSERT(consumer1Thread.getExitStatus() == DQT_NORMAL);
    DLR_TEST_ASSERT(consumer2Thread.getExitStatus() == DQT_NORMAL);
    DLR_TEST_ASSERT(consumer3Thread.getExitStatus() == DQT_NORMAL);
    DLR_TEST_ASSERT(producerThread.getExitStatus() == DQT_NORMAL);
    DLR_TEST_ASSERT(sourceSink.isComplete());
  }

} // namespace dlr


#if 0

int main(int argc, char** argv)
{
  dlr::DistributionQueueTest currentTest;
  bool result = currentTest.run();
  return (result ? 0 : 1);
}

#else

namespace {

  dlr::DistributionQueueTest currentTest;

}

#endif
