/**
***************************************************************************
* @file dlrThread/monitor.h 
* Header file declaring Monitor class.
*
* Copyright (C) 2006-2007 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 915 $
* $Date: 2007-05-18 01:27:47 -0400 (Fri, 18 May 2007) $
***************************************************************************
**/

#ifndef _DLR_THREAD_MONITOR_H_
#define _DLR_THREAD_MONITOR_H_

#include <vector>
#include <boost/thread.hpp>
#include <dlrCommon/referenceCount.h>

namespace dlr {

  namespace thread {

    // Forward declarations.
    class Monitor;
    class Condition;

    
    /** 
     ** This helper class represents a virtual token which is passed
     ** between threads to prevent more than one thread from
     ** interacting with the Monitor at once.  Someday it may be
     ** promoted to it's own source file.  See the documentation for
     ** class Monitor for more information.
     **/
    class Token {
      friend class Monitor;
      friend class Condition;
      
    public:
      Token();
      Token(const Token& other);
      ~Token();
      operator bool();

    private:
      Token(boost::timed_mutex& mutex);
      Token(boost::timed_mutex& mutex, double timeout);
      boost::timed_mutex::scoped_lock& getLock();
      boost::timed_mutex::scoped_timed_lock& getTimedLock();
      bool isTimed();
      Token& operator=(const Token& other);

      bool m_isTimed;
      boost::timed_mutex::scoped_lock* m_lockPtr;
      boost::timed_mutex::scoped_timed_lock* m_timedLockPtr;

      // We use ReferenceCount directly, rather than through a
      // pointer, because Token objects are never expected to be
      // shared between threads, so thread isn't an issue.
      ReferenceCount m_referenceCount;
    };

    
    /** 
     ** This helper class represents a condition variable as described
     ** in Hoare's paper.  Someday it may be promoted to it's own
     ** source file.  See the documentation for class Monitor for more
     ** information.
     **/
    class Condition {
      friend class Monitor;
    public:
      Condition(const Condition& other);
      ~Condition();
      Condition& operator=(const Condition& other);

    private:
      Condition(boost::condition* conditionPtr);
      void signalOne();
      void signalAll();
      void wait(Token& token);
      bool wait(Token& token, double timeout);

      boost::condition* m_conditionPtr;
    };

    

    /** 
     ** This class implements the Monitor pattern proposed by
     ** C. A. R. Hoare in [1].  Use this class as follows:
     **
     **   - First derive a subclass of Monitor.
     **
     **   - Declare as many private or protected Monitor::Condition
     **     instances as necessary, and use this->createCondition() in
     **     the class constructor to initialize them.
     **
     **   - Make sure all public subclass methods, with the exception
     **     of the copy constructor, begin with a call to
     **     this->getToken().  The copy constructor should begin with a
     **     call to Monitor::operator=(const Monitor&), and then call
     **     this->getToken() and save its return value in a local
     **     variable.
     **
     **   - Make sure the class has no public data members.
     **
     **   - Make sure the class member functions don't access any external
     **     data.  Data passed by value as an argument to the member
     **     function doesn't count as external.
     **
     ** Here's an example of a simple monitor, based on the first
     ** example in Hoares paper.  It turns out that this example could
     ** be more easily implemented using just a semaphore or a lock,
     ** but it's useful anyway.  Note that the signal() method differs
     ** slightly from the behavior described in the paper (see the
     ** documentation for Monitor::signal()) but that this difference
     ** is insignificant if we adhere to the usual convention of only
     ** calling signal() at the end of a function.
     **
     ** @code
     **
     ** class SimpleMonitor: public Monitor {
     ** public:
     **   SimpleMonitor()
     **     : Monitor(),
     **       m_isBusy(false),
     **       m_nonbusyCondition(Monitor::createCondition()) {}
     **
     **   void acquire() {
     **     Token token = this->getToken();
     **     if(this->m_isBusy) {
     **       this->wait(this->m_nonbusyCondition, token);
     **     }
     **     this->m_isBusy = true;
     **   }
     **
     **   void release() {
     **     Token token = this->getToken();
     **     this->m_isBusy = false;
     **     this->signalOne(this->m_nonbusyCondition, token);
     **   }
     **     
     ** private:
     **   bool m_isBusy;
     **   Condition m_nonbusyCondition;
     ** };
     **
     ** @endcode
     **
     ** Monitor is also useful as a parent class for objects to which
     ** access should be serialized.  In this case, you may choose to
     ** completely ignore the wait(), signalOne() and signalAll()
     ** methods and just use the getToken() method to ensure that the
     ** object interacts with only one thread at a time.
     **
     ** Monitor is reference counted, so that interacting with a copy
     ** of a monitor object has the same effect as interacting with
     ** the monitor object itself.  This increases the flexibility of
     ** the class, but also opens the door to lots of headaches: data
     ** members of derived classes will not be shared between copies
     ** unless you make them so by using pointers, reference counting,
     ** etc.  For this reason, copying is disabled by default.  If you
     ** really want to copy a Monitor instance, you can enable copying
     ** by calling the makeCopyable() member function.
     **
     ** If you do decide to copy a Monitor instance, the copy
     ** constructor and assignment operator should be sure to copy the
     ** class in a thread safe way.  The following example illustrates
     ** this, and shows how to avoid wasting resources when copying
     ** Condition instances.
     **
     ** @code
     **   SimpleMonitor(const SimpleMonitor& other)
     **     : Monitor(other),
     **       m_isBusy(),
     **       m_nonbusyCondition(Monitor::createCondition(false)) {
     **         Token token = this->getToken();
     **         m_isBusy = other.m_isBusy;
     **         m_nonBusyCondition = other.m_nonBusyCondition;
     **       }
     ** @endcode
     **/
    class Monitor
    {
    public:

      /**
       * Default constructor.
       */
      Monitor();


      /** 
       * The copy constructor does a shallow copy.  Operating on the
       * copy has exactly the same effect as operating on the
       * original.  For more information about copying monitors,
       * please read the class-level documentation for Monitor.
       * 
       * @param other The Monitor instance to be copied.
       */
      Monitor(const Monitor& other);


      /** 
       * Destroys the Monitor instance and releases any resources.
       */
      virtual
      ~Monitor();


      /** 
       * The assignment operator does a shallow copy.  After assignment,
       * operating on the copy has exactly the same effect as operating
       * on the original.
       * 
       * @param other The Monitor instance to be copied.
       *
       * @return A reference to *this.
       */
      Monitor&
      operator=(const Monitor& other);


      /** 
       * This member function enables (or disables) copying of the
       * Monitor instance.
       * 
       * @param copyableFlag When this argument is left at its default
       * value of true, a call to makeCopyable will enable copying of
       * the Monitor instance.  Explicitly setting this argument to
       * false will reverse the effect of makeCopyable() and disable
       * copying.
       */
      void
      makeCopyable(bool copyableFlag=true) {m_isCopyable = copyableFlag;}

    protected:

      /** 
       * This protected member function is called by the copy
       * constructor and assignment operator in order to copy another
       * Monitor instance in a thread-safe way.
       * 
       * @param other This argument is the Monitor instance to be
       * copied.
       */
      void
      copyOther(const Monitor& other);

      
      /** 
       * This protected member function creates a condition to be used
       * for communicating with other threads.  If argument
       * doInitialize is true, then the resources allocated by
       * createCondition() are not returned to the system until the
       * Monitor instance is destroyed.
       *
       * Special care must be taken when creating Condition objects in
       * copy constructors.  The member initialization list of the
       * copy constructor should create a dummy condition by setting
       * argument doInitialize to false, and the copying of the
       * condition from the to-be-copied Monitor instance should be
       * done in the body of the copy constructor, where it can be
       * protected by creation of a Token instance.  Please refer to
       * the class documentation for Monitor to see an example.
       *
       * @param doInitialize This parameter should be set to false in
       * the parameter initializion lists of copy constructors in
       * order to suppress allocation of resources.  Note(xxx): is
       * this really necessary now that condition has no reference
       * count?
       *
       * @return The return value is a condition object.
       */
      Condition
      createCondition(bool doInitialize = true);


      /** 
       * This protected member function requests, and waits for, a
       * virtual "token" which is shared between threads.  The caller
       * will block until the it gets the token, at which point all
       * other callers will block until the token goes out-of-scope in
       * the callers thread.
       * 
       * @return The return value represents the virtual token.
       */
      Token
      getToken() {return Token(*m_tokenMutexPtr);}

      
      /** 
       * This protected member function requests, and waits for, a
       * virtual "token" which is shared between threads.  It is just
       * like getToken(), with the exception that it will not block
       * for longer than the specified timeout.  If the timout expires
       * before the token is acquired, a TimeoutException is thrown.
       * 
       * @return The return value represents the virtual token.
       */
      Token
      getToken(double timeout) {return Token(*m_tokenMutexPtr, timeout);}


      /** 
       * This protected member function is called by the destructor
       * and assignment operator to decrement reference counts, etc.
       */
      void
      releaseResources();

      
      /** 
       * This protected member function wakes up one of the threads
       * which is waiting on the specified condition.  If no thread is
       * waiting, then signalOne() does nothing.  The awakened thread
       * will become active after the current thread releases the
       * token.  Note that this behavior is different from the
       * behavior of signal() in Hoare's paper: in Hoare's
       * formulation, the awakened thread becomes active immediately,
       * and control returns to the current thread after the awakened
       * thread releases the token.  This difference is immaterial if
       * you follow the usual convention of only calling signalOne() as
       * the last line of a member function.
       * 
       * @param condition This argument is the condition to signal.
       */
      void
      signalOne(Condition& condition) {condition.signalOne();}

      
      /** 
       * This protected member function wakes up all of the threads
       * which are waiting on the specified condition.  If no thread
       * is waiting, then signalAll() does nothing.  The awakened
       * threads will become active (each in turn) after the current
       * thread releases the token.  Note that signalAll() has no
       * analog in Hoare's paper.
       *
       * @param condition This argument is the condition to signal.
       */
      void
      signalAll(Condition& condition) {condition.signalAll();}

      
      /** 
       * This protected member function waits until signalOne() or
       * signalAll() is called by another thread with the same
       * Condition as its argument.
       *
       * @param condition This argument is the condition on which to
       * wait.
       * 
       * @param token This argument is a token instance, which should
       * have been acquired using getToken() at the beginning of the
       * member function which is now calling wait().
       * 
       */
      void
      wait(Condition& condition, Token& token) {condition.wait(token);}
        

      /** 
       * This protected member function waits until signalOne() or
       * signalAll() is called by another thread with the same
       * Condition as its argument.  It is just like wait(Condition&),
       * with the exception that it will not block for longer than the
       * specified timeout.
       * 
       * @param condition This argument is the condition on which to
       * wait.
       * 
       * @param token This argument is a token instance, which should
       * have been acquired using getToken() at the beginning of the
       * member function which is now calling wait().
       * 
       * @param timeout This argument specifies how long to wait for
       * the condition to be satisfied.
       *
       * @return The return value is false if the timout expires, true
       * otherwise.
       */
      bool
      wait(Condition& condition, Token& token, double timeout) {
        return condition.wait(token, timeout);
      }


    private:

      // We need a mutex in order to implement the virtual token.
      boost::timed_mutex* m_tokenMutexPtr;

      // This vector lets us manage deletion of condition objects
      // explicitly, rather than relying on the Condition class to do
      // so.  This is important because the Condition class destructor
      // is not always called in a protected section of code.
      std::vector<boost::condition*>* m_boostConditionVectorPtr;
      
      // This bool indicates whether or not copying should be allowed.
      bool m_isCopyable;
      
      // Note that thread-safety requires us to use a pointer here.
      size_t* m_referenceCountPtr;
    };

  } // namespace thread

} // namespace dlr

#endif /* #ifndef _DLR_THREAD_MONITOR_H_ */
