/**
***************************************************************************
* @file dlrThread/monitor.cpp
* Source file implementing the Monitor class.
*
* Copyright (C) 2006-2007 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 911 $
* $Date: 2007-05-17 15:11:44 -0400 (Thu, 17 May 2007) $
***************************************************************************
**/

#include <dlrThread/exception.h>
#include <dlrThread/monitor.h>
#include <dlrThread/private.h>


namespace dlr {

  namespace thread {


    Token::
    Token()
      : m_isTimed(false),
        m_lockPtr(0),
        m_timedLockPtr(0),
        m_referenceCount(0)
    {
      // Empty.
    }

    
    Token::
    Token(const Token& other)
      : m_isTimed(other.m_isTimed),
        m_lockPtr(other.m_lockPtr),
        m_timedLockPtr(other.m_timedLockPtr),
        m_referenceCount(other.m_referenceCount)
    {
      // Empty.
    }


    Token&
    Token::
    operator=(const Token& other)
    {
      if(this != &other) {
        m_isTimed = other.m_isTimed;
        m_lockPtr = other.m_lockPtr;
        m_timedLockPtr = other.m_timedLockPtr;
        m_referenceCount = other.m_referenceCount;
      }
      return *this;
    }
    

    Token::
    Token(boost::timed_mutex& mutex)
      : m_isTimed(false),
        m_lockPtr(new boost::timed_mutex::scoped_lock(mutex)),
        m_timedLockPtr(0),
        m_referenceCount()
    {
      // Empty.
    }


    Token::
    Token(boost::timed_mutex& mutex, double timeout)
      : m_isTimed(true),
        m_lockPtr(0),
        m_timedLockPtr(0),
        m_referenceCount()
    {
      if(timeout < 0.0) {
        timeout = 0.0;
      }
      boost::xtime xtimeout;
      if(!getXTime(timeout, xtimeout)) {
        DLR_THROW(RunTimeException, "Token::Token()",
                  "Can't get current time.");
      }
      try {
        m_timedLockPtr =
          new boost::timed_mutex::scoped_timed_lock(mutex, xtimeout);
      } catch(boost::lock_error&) {
        // Leave m_timedLockPtr == 0
      }          
    }


    Token::
    ~Token() {
      if (!m_referenceCount.isShared()) {
        if(m_isTimed) {
          delete m_timedLockPtr;
        } else {
          delete m_lockPtr;
        }
      }
    }


    Token::
    operator bool()
    {
      return ((m_lockPtr != 0) || (m_timedLockPtr != 0));
    }

    
    boost::timed_mutex::scoped_lock&
    Token::
    getLock()
    {
      return *m_lockPtr;
    }


    boost::timed_mutex::scoped_timed_lock&
    Token::
    getTimedLock()
    {
      return *m_timedLockPtr;
    }
    

    bool
    Token::
    isTimed()
    {
      return m_isTimed;
    }

    
    Condition::
    Condition(boost::condition* conditionPtr)
      : m_conditionPtr(conditionPtr)
    {
      // Empty.
    }


    Condition::
    Condition(const Condition& other)
      : m_conditionPtr(other.m_conditionPtr)
    {
      // Empty.
    }

    
    Condition::
    ~Condition()
    {
      // Empty.
    }


    Condition&
    Condition::
    operator=(const Condition& other)
    {
      if(&other != this) {
	m_conditionPtr = other.m_conditionPtr;
      }
      return *this;
    }

    
    void
    Condition::
    signalOne()
    {
      if(m_conditionPtr != 0) {
        m_conditionPtr->notify_one();
      }
    }

    
    void
    Condition::
    signalAll()
    {
      if(m_conditionPtr != 0) {
        m_conditionPtr->notify_all();
      }
    }

    
    void
    Condition::
    wait(Token& token)
    {
      if(m_conditionPtr != 0) {
        if(token.isTimed()) {
          m_conditionPtr->wait(token.getTimedLock());
        } else {
          m_conditionPtr->wait(token.getLock());
        }
      }
    }

    
    bool
    Condition::
    wait(Token& token, double timeout)
    {
      if(m_conditionPtr != 0) {
        if(timeout < 0.0) {
          timeout = 0.0;
        }
        boost::xtime xtimeout;
        if(!getXTime(timeout, xtimeout)) {
          return false;
        }
        if(token.isTimed()) {
          return m_conditionPtr->timed_wait(token.getTimedLock(), xtimeout);
        }
        return m_conditionPtr->timed_wait(token.getLock(), xtimeout);
      }
      return false;
    }


    
    Monitor::
    Monitor()
      : m_tokenMutexPtr(new boost::timed_mutex),
	m_boostConditionVectorPtr(new std::vector<boost::condition*>),
	m_isCopyable(false),
        m_referenceCountPtr(new size_t)
    {
      *m_referenceCountPtr = 1;
    }


    // The copy constructor does a shallow copy.
    Monitor::
    Monitor(const Monitor& other)
      : m_tokenMutexPtr(0),
	m_boostConditionVectorPtr(0),
	m_isCopyable(false),
        m_referenceCountPtr(0)
    {
      this->copyOther(other);
    }

    
    // Destroys the Monitor instance and releases any resources.
    Monitor::
    ~Monitor()
    {
      this->releaseResources();
    }

    
    // The assignment operator does a shallow copy.
    Monitor&
    Monitor::
    operator=(const Monitor& other)
    {
      if(&other != this) {
        if(!m_isCopyable) {
          DLR_THROW(LogicException, "Monitor::operator=()",
                    "Attempt to copy a Monitor instance or subclass "
                    "without explicitly enabling copying.");
        }
	this->releaseResources();
	this->copyOther(other);
      }
      return *this;
    }      


    // This protected member function is called by the copy
    // constructor and assignment operator in order to copy another
    // Monitor instance in a thread-safe way.
    void
    Monitor::
    copyOther(const Monitor& other)
    {
      // m_tokenMutexPtr must be valid because we're copying a valid
      // Monitor instance.
      m_tokenMutexPtr = other.m_tokenMutexPtr;
      {
	Token token = this->getToken();
	if(!other.m_isCopyable) {
	  DLR_THROW(LogicException, "Monitor::copyOther()",
		    "Attempt to copy a Monitor instance or subclass "
		    "without explicitly enabling copying.");
	}
	m_boostConditionVectorPtr = other.m_boostConditionVectorPtr;
	m_isCopyable = other.m_isCopyable;
	m_referenceCountPtr = other.m_referenceCountPtr;
	++(*m_referenceCountPtr);
      }
    }

      
    // This protected member function creates a condition to be used
    // for communicating with other threads.
    Condition
    Monitor::
    createCondition(bool doInitialize)
    {
      if(doInitialize) {
	// When calling createCondition from within a constructor,
	// this next line works as long as m_tokenMutexPtr has already
	// been initialized.
	Token token = this->getToken();
	m_boostConditionVectorPtr->push_back(new boost::condition);
	size_t index0 = m_boostConditionVectorPtr->size() - 1;
	return Condition((*m_boostConditionVectorPtr)[index0]);
      }
      return Condition(0);
    }


    // This protected member function is called by the destructor
    // and assignment operator to decrement reference counts, etc.
    void
    Monitor::
    releaseResources()
    {
      bool isShared = true;
      {
	Token token = this->getToken();
        if(--(*m_referenceCountPtr) == 0) {
          isShared = false;
        }
      }

      if(!isShared) {
	// Since *this is the only copy left, we can go ahead
	// and delete.
	for(size_t index0 = 0; index0 < m_boostConditionVectorPtr->size();
	    ++index0) {
	  delete (*m_boostConditionVectorPtr)[index0];	  
	}
	
        delete m_tokenMutexPtr;
	m_tokenMutexPtr = 0;
	delete m_boostConditionVectorPtr;
	m_boostConditionVectorPtr = 0;
        delete m_referenceCountPtr;
	m_referenceCountPtr = 0;
      }
    }
    
  } // namespace thread

} // namespace dlr
