/**
***************************************************************************
* @file dlrThread/distributionQueue.h 
* Header file declaring DistributionQueue class.
*
* Copyright (C) 2005-2007 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 706 $
* $Date: 2006-08-04 19:41:11 -0400 (Fri, 04 Aug 2006) $
***************************************************************************
**/

#ifndef _DLR_THREAD_DISTRIBUTIONQUEUE_H_
#define _DLR_THREAD_DISTRIBUTIONQUEUE_H_

#include <map>
#include <boost/thread.hpp>
#include <dlrThread/clientID.h>
#include <dlrThread/exception.h>
#include <dlrThread/monitor.h>
#include <dlrUtilities/timeUtilities.h>


namespace dlr {

  namespace thread {

    /** 
     ** This class implements FIFO-style communication between one or
     ** more producer threads and multiple consumer threads, so that
     ** each queue item is consumed by exactly one thread.  If you
     ** want each item on the queue to be consumed once by each
     ** consumer thread, you should use a BroacastQueue instead of a
     ** DistributionQueue.
     **
     ** Create the DistributionQueue instance before instantiating the
     ** producer and consumer threads, and provide each thread with a
     ** copy of, pointer to, or reference to the DistributionQueue.
     ** Producers can stuff the FIFO by the pushFront() member
     ** function.  Consumers can extract data from the other end using
     ** the bufferBack() and getBuffer() member functions.  The
     ** bufferBack()/getBuffer() combination is used instead of the
     ** more traditional copyBack()/popBack() interface in order to
     ** allow an interface which is both thread-safe and
     ** exception-safe.  Note that a careful analysis of exception
     ** safety has not been done.  The interface was selected so as
     ** not to preclude exception safety.
     **
     ** During normal usage all client threads are expected to stay
     ** within maximumLength elements of the head.  If a consumer
     ** thread falls more than maximumLength elements behind the head,
     ** then the next call to push_back() will throw an overflow
     ** exception.  This behavior may change in the future.
     **/
    template <class Type>
    class DistributionQueue
      : public Monitor
    {
    public:

      /**
       ** Each thread that interacts with a particular DistributionQueue
       ** instance must create a ClientID instance, and then register
       ** with the DistributionQueue by calling the registerClient() member
       ** function of that DistributionQueue instance, passing the clientID
       ** instance as an argument.  The same ClientID instance must be
       ** passed as an argument in subsequent interactions with that
       ** DistributionQueue instance.  The ClientID instance identifies the
       ** thread to the DistributionQueue, and simplifies internal
       ** bookkeeping.
       **/
      typedef ClientID<Type> ClientID;


      /** 
       * This constructor specifies the maximum length of the FIFO.
       * 
       * @param maximumLength This argument specifies the maximum
       * length of the FIFO.  All consumers are expected to call
       * this->popBack() frequently enough to stay within this many
       * elements of the head of the FIFO.  If consumers fail to keep
       * up, then calls to pushFront(), bufferBack() or lockBack() may
       * throw exceptions.
       *
       * @param numberOfClients This argument can be used to specify
       * the number of threads that will interact with the DistributionQueue
       * instance.  If this argument is not provided, the DistributionQueue
       * will function normally, except that memory usage will likely
       * be higher, and execution time for some member function calls
       * may be increased.
       */
      DistributionQueue(size_t maximumLength, size_t numberOfClients=2);


      /** 
       * The copy constructor does a shallow copy.  Operating on the
       * copy has exactly the same effect as operating on the
       * original.
       * 
       * @param source The DistributionQueue instance to be copied.
       */
      DistributionQueue(const DistributionQueue<Type>& source);


      /** 
       * Destroys the DistributionQueue instance and releases any resources.
       */
      virtual
      ~DistributionQueue();


      /** 
       * The assignment operator does a shallow copy.  After assignment,
       * operating on the copy has exactly the same effect as operating
       * on the original.
       * 
       * @param source The DistributionQueue instance to be copied.
       * @return A reference to *this.
       */
      DistributionQueue<Type>&
      operator=(const DistributionQueue<Type>& source);


      /** 
       * This member function copies the tail element of the
       * DistributionQueue into an interal buffer, and then removes it
       * from the queue so that other threads can no longer access it.
       * Repeated calls to bufferBack() will copy and remove
       * subsequent elements from the queue.
       *
       * Each thread has its own buffer, so elements can be buffered
       * by two or more consumer threads with no conflict.
       *
       * If all items have been removed from the queue, bufferBack()
       * will block, waiting for a new element to be copied onto the
       * queue.
       *
       * @param clientID This argument identifies the calling thread
       * to the DistributionQueue.  It should have been previously
       * registered with the DistributionQueue by calling member
       * function registerClient().
       */
      void
      bufferBack(const ClientID& clientID);


      /** 
       * This member function copies the tail element of the
       * DistributionQueue into an interal buffer, and then removes it
       * from the queue so that other threads can no longer access it.
       * It is just like void bufferBack(const ClientID&), with the
       * exception that it will not block for longer than the
       * specified timeout.
       *
       * @param clientID This argument identifies the calling thread
       * to the DistributionQueue.  It should have been previously
       * registered with the DistributionQueue by calling member
       * function registerClient().
       *
       * @param timeout This argument specifies the total amount of
       * time that bufferBack is permitted to block while waiting for
       * access to the queue and/or waiting for a new element to
       * become available.  If timeout is less than or equal to zero
       * then the call will return immediately.
       */
      bool
      bufferBack(const ClientID& clientID, double timeout);


      /** 
       * This member function empties the queue.
       */
      void
      clear();
      

      /** 
       * This member function copies the buffered element (see member
       * function bufferBack() from the internal buffer into user
       * code.  Repeated calls to copyBuffer() will copy the same
       * element until a new call to bufferBack() is made.  If
       * copyBuffer() is called before any calls to bufferBack() are
       * made, a LogicException will be thrown.
       *
       * @param clientID This argument identifies the calling thread
       * to the DistributionQueue.  It should have been previously
       * registered with the DistributionQueue by calling member
       * function registerClient().
       * 
       * @param target This reference argument is used to return a
       * copy of the tail element in the queue.
       */
      void
      copyBuffer(const ClientID& clientID, Type& target);


      /** 
       * This member function returns the length at which the
       * DistributionQueue instance will stop accepting new elements.
       * 
       * @return The return value is the maximum number of elements
       * that consumer threads can fall behind without causing
       * problems.
       */
      size_t
      getMaximumLength() {return m_size;}
      

      /** 
       * This member function locks the DistributionQueue instance so
       * that items cannot be pushed or removed by other threads.  By
       * never using this member function, you will avoid many
       * headaches, so please use copyBack() instead if at all
       * possible.  Member function lockBack() is provided to handle a
       * few extreme circumstances.
       *
       * Member function lockBack() will block until the lock is acquired.
       * If the queue is empty, lockBack() will sleep until data is
       * available, and then block until the lock is acquired.
       *
       * An example of a circumstance in which it makes sense to use
       * lockBack() is when the distributionQueue elements contain
       * pointers or references to data, so that copyBuffer() gives
       * you a copy of a pointer, but doesn't actually copy the data.
       * Presumably, the producer threads are smart enough not to
       * de-allocate the data until the element drops of the end of
       * the queue, so you can be confident that you're getting a good
       * pointer at the time of the bufferBack() call, but there's
       * nothing to stop the associated data from being de-allocated)
       * after the call, when the consumer is still trying to access
       * it.  In this case, it makes sense to call lockBack() in the
       * following sequence.
       *
       *   myDistributionQueue.lockBack(myClientID);
       *
       *   myDistributionQueue.bufferBack(myClientID);
       *
       *   myDistributionQueue.copyBuffer(myClientID, myLocalElement);
       *
       *   // Code here does accesses to the referenced data.
       *
       *   myDistributionQueue.unlockBack(myClientID);
       *
       * Member function bufferBack() is smart enough not to deadlock
       * when copying an already-locked element.
       *
       * @param clientID This argument identifies the calling thread
       * to the DistributionQueue.  It should have been previously
       * registered with the DistributionQueue by calling member
       * function registerClient().
       */
      void
      lockBack(const ClientID& clientID);


      /** 
       * This member function locks the DistributionQueue instance so
       * that items cannot be pushed or removed by other threads.  By
       * never using this member function, you will avoid many
       * headaches.  It is just like void lockBack(const ClientID&),
       * with the exception that it will not block for longer than the
       * specified timeout.
       *
       * @param clientID This argument identifies the calling thread
       * to the DistributionQueue.  It should have been previously
       * registered with the DistributionQueue by calling member
       * function registerClient().
       * 
       * @param timeout This argument controls how long the call to
       * lockBack() will block waiting for the lock to become
       * available.  If timeout is less than or equal to zero, then
       * the call to will return immediately.
       *
       * @return The return value is true if a lock was established,
       * and false otherwise.
       */
      bool
      lockBack(const ClientID& clientID, double timeout);
      

      /** 
       * This member function is called by producer threads to add
       * elements to the head of the queue.  When the element is added
       * to the queue, it is copied, not referenced.
       * 
       * @param clientID This argument identifies the calling thread
       * to the DistributionQueue.  It should have been previously
       * registered with the DistributionQueue by calling member
       * function registerClient().
       * 
       * @param element This argument is the element to be copied onto
       * the head of the DistributionQueue instance.
       */
      void
      pushFront(const ClientID& clientID, const Type& element);


      /** 
       * This member function introduces a thread to a
       * DistributionQueue instance.  The thread should create a
       * ClientID instance, and then call registerClient() exactly
       * once, passing the ClientID instance as the first argument.
       * This same ClientID instance should be used by the calling
       * thread in subsequent interactions with the DistributionQueue
       * instance.  Using the same ClientID instance to interact with
       * another DistributionQueue instance (besides *this) is bad, and
       * will have undefined results.
       *
       * Note that the ClientID class is currently not copyable.  This
       * means the client thread should create the ClientID in an
       * outer scope and then pass it by reference into any inner
       * scopes that need access to the DistributionQueue.
       *
       * @param clientID This argument is the clientID that will be
       * used to identify the calling thread to the DistributionQueue
       * in future interactions.
       *
       * @param clientName This argument specifies a name for the
       * thread.  It is useful to specify a unique name for each
       * thread that interacts with a DistributionQueue instance
       * because this will make it easier to debug any exceptions
       * thrown by the DistributionQueue member functions.
       */
      void
      registerClient(ClientID& clientID,
                     const std::string& clientName="anonymous");


      /** 
       * This member function releases a lock acquired by
       * lockBack().
       * 
       * @param clientID This argument identifies the calling thread
       * to the DistributionQueue.  It should have been previously
       * registered with the DistributionQueue by calling member
       * function registerClient().
       */
      void
      unlockBack(const ClientID& clientID);
      
    private:

      inline void
      releaseLock();
      
      
      inline void
      waitForLock(Token& token);


      inline bool
      waitForLock(Token& token, double timeout);

      
        
      size_t* m_headIndexPtr;
      size_t m_size;
      size_t* m_tailIndexPtr;
      
      Type** m_bufferArrayPtr;
      std::map<size_t, std::string>* m_clientNameMapPtr;
      Type* m_dataArray;
      size_t* m_maxNumberOfClientsPtr;
      size_t* m_numberOfClientsPtr;

      Condition m_headChangedCondition;
      bool* m_isLockedPtr;
      Condition m_isLockedCondition;

      // We use a pointer to ReferenceCount, rather than a
      // ReferenceCount instance, so that we can manage construction,
      // destruction, etc. inside locked sections of code.
      ReferenceCount* m_referenceCountPtr;
    };

  } // namespace thread

} // namespace dlr


/* =============       Implementation follows        ============= */

#include <algorithm>
#include <iostream>
#include <sstream>
#include <dlrThread/private.h>


namespace dlr {

  namespace thread {

    /* ===========    Implementation of DistributionQueue      =========== */

    template <class Type>
    DistributionQueue<Type>::
    DistributionQueue(size_t maximumLength, size_t numberOfClients)
      : Monitor(),
        m_headIndexPtr(new size_t),
        m_size(maximumLength),
        m_tailIndexPtr(new size_t),
        m_bufferArrayPtr(new Type*),
        m_clientNameMapPtr(new std::map<size_t, std::string>),
        m_dataArray(new Type[maximumLength]),
        m_maxNumberOfClientsPtr(new size_t),
        m_numberOfClientsPtr(new size_t),
		m_headChangedCondition(Monitor::createCondition()),
        m_isLockedPtr(new bool),
		m_isLockedCondition(Monitor::createCondition()),
        m_referenceCountPtr(new ReferenceCount)
    {
      if(numberOfClients == 0) {
        DLR_THROW(ValueException, "DistributionQueue::DistributionQueue()",
                  "Argument numberOfClients must be greater than zero.");
      }

      // No elements have been added yet.
      *m_headIndexPtr = 0;
      *m_tailIndexPtr = 0;

      // But indicate that no clients have registered yet.
      *m_bufferArrayPtr = new Type[numberOfClients];
      *m_maxNumberOfClientsPtr = numberOfClients;
      *m_numberOfClientsPtr = 0;

      // And that nobody's called this->lockBack().
      *m_isLockedPtr = false;

      // Finally, allow copying.
      this->makeCopyable();
    }


    // The copy constructor does a shallow copy.
    template <class Type>
    DistributionQueue<Type>::
    DistributionQueue(const DistributionQueue<Type>& source)
      : Monitor(source),
        m_headIndexPtr(0),
        m_size(0),
        m_tailIndexPtr(0),
        m_bufferArrayPtr(0),
        m_clientNameMapPtr(0),
        m_dataArray(0),
        m_maxNumberOfClientsPtr(0),
        m_numberOfClientsPtr(0),
		m_headChangedCondition(Monitor::createCondition(false)),
        m_isLockedPtr(0),
		m_isLockedCondition(Monitor::createCondition(false)),
        m_referenceCountPtr(0)
    {
      Token token = this->getToken();
      // No need to wait for *m_isLockedPtr because we're not going to
      // read or change queue contents.
      
      m_headIndexPtr = source.m_headIndexPtr;
      m_size = source.m_size;
      m_tailIndexPtr = source.m_tailIndexPtr;
      m_bufferArrayPtr = source.m_bufferArrayPtr;
      m_clientNameMapPtr = source.m_clientNameMapPtr;
      m_dataArray = source.m_dataArray;
      m_maxNumberOfClientsPtr = source.m_maxNumberOfClientsPtr;
      m_numberOfClientsPtr = source.m_numberOfClientsPtr;
      m_headChangedCondition = source.m_headChangedCondition;
      m_isLockedPtr = source.m_isLockedPtr;
      m_isLockedCondition = source.m_isLockedCondition;
      m_referenceCountPtr = source.m_referenceCountPtr;
      ++(*m_referenceCountPtr);
    }

    
    // Destroys the DistributionQueue instance and releases any resources.
    template <class Type>
    DistributionQueue<Type>::
    ~DistributionQueue()
    {
      Token token = this->getToken();
      // No need to wait for *m_isLockedPtr because we're not going to
      // read or change queue contents unless we're the only instance
      // referencing this data.
      
      if(!(m_referenceCountPtr->isShared())) {
        delete m_headIndexPtr;
        delete m_tailIndexPtr;
        delete m_bufferArrayPtr;
        delete m_clientNameMapPtr;
        delete m_dataArray;
        delete m_maxNumberOfClientsPtr;
        delete m_numberOfClientsPtr;
        delete m_isLockedPtr;
        delete m_referenceCountPtr;
      } else {
        --(*m_referenceCountPtr);
      }
    }


    // The assignment operator does a shallow copy.
    template <class Type>
    DistributionQueue<Type>&
    DistributionQueue<Type>::
    operator=(const DistributionQueue<Type>& source)
    {
      if(&source != this) {
        Monitor::operator=(source);
        Token token = this->getToken();
        // No need to wait for *m_isLockedPtr because we're not going
        // to read or change queue contents unless we're the only
        // instance referencing the data.

        this->release();
        this->copyOther(source);
      }
    }      


    // This member function copies the tail element of the
    // DistributionQueue.
    template <class Type>
    void
    DistributionQueue<Type>::
    bufferBack(const ClientID& clientID)
    {
      // All locking, synchronizing, etc. handled inside lockBack().
      this->lockBack(clientID);
      
      try {
        (*m_bufferArrayPtr)[clientID.m_idNumber]
          = m_dataArray[*m_tailIndexPtr];
      } catch(...) {
        this->unlockBack(clientID);
        throw;
      }

      ++(*m_tailIndexPtr);
      if(*m_tailIndexPtr == m_size) {
        *m_tailIndexPtr = 0;
      }

      this->unlockBack(clientID);
    }


    // This member function copies the tail element of the
    // DistributionQueue.
    template <class Type>
    bool
    DistributionQueue<Type>::
    bufferBack(const ClientID& clientID, double timeout)
    {
      // All locking, synchronizing, etc. handled inside lockBack().
      if(this->lockBack(clientID, timeout) == false) {
        return false;
      }

      try {
        (*m_bufferArrayPtr)[clientID.m_idNumber]
          = m_dataArray[*m_tailIndexPtr];
      } catch(...) {
        this->unlockBack(clientID);
        throw;
      }

      ++(*m_tailIndexPtr);
      if(*m_tailIndexPtr == m_size) {
        *m_tailIndexPtr = 0;
      }

      this->unlockBack(clientID);
      return true;
    }


    // This member function empties the queue.
    template <class Type>
    void
    DistributionQueue<Type>::
    clear()
    {
      Token token = this->getToken();
      this->waitForLock();
      *m_tailIndexPtr = *m_headIndexPtr;
      this->releaseLock();
    }

    
    // This member function copies the buffered element (see member
    // function bufferBack() from the internal buffer into user
    // code.
    template <class Type>
    void
    DistributionQueue<Type>::
    copyBuffer(const ClientID& clientID, Type& target)
    {
      target = (*m_bufferArrayPtr)[clientID.m_idNumber];
    }


    // This member function locks the tail element of the DistributionQueue
    // instance so that it cannot fall off the back of the queue.
    template <class Type>
    void
    DistributionQueue<Type>::
    lockBack(const ClientID& clientID)
    {
      Token token = this->getToken();
      this->waitForLock(token);

      // Make sure there's data available to copy.  That is, make
      // sure we're not ahead of the queue, and wait for a producer
      // to add an element if we are.
      while(*m_tailIndexPtr == *m_headIndexPtr) {
        this->releaseLock();
        this->wait(m_headChangedCondition, token);
        this->waitForLock(token);
      }

      // Note that we still have the lock!
      return;
    }


    // This member function locks the tail element of the DistributionQueue
    // instance so that it cannot fall off the back of the queue.
    template <class Type>
    bool
    DistributionQueue<Type>::
    lockBack(const ClientID& clientID, double timeout)
    {
      // Get exclusive control of the distributionQueue.
      double deadline = getCurrentTime() + timeout;
      Token token = this->getToken(timeout);
      if(!token) {
        return false;
      }
      if(!this->waitForLock(token, deadline - getCurrentTime())) {
        return false;
      }

      // Make sure there's data available to copy.  That is, make
      // sure we're not ahead of the queue, and wait for a producer
      // to add an element if we are.
      while(*m_tailIndexPtr == *m_headIndexPtr) {
        this->releaseLock();
        if(!this->wait(m_headChangedCondition, token,
                       deadline - getCurrentTime())) {
          return false;
        }
        if(!this->waitForLock(token, deadline - getCurrentTime())) {
          return false;
        }
      }
        
      // Note that we still have the lock!
      return true;
    }


    template <class Type>
    void
    DistributionQueue<Type>::
    pushFront(const ClientID& clientID, const Type& element)
    {
      // This variable lets us avoid time-consuming formatting of
      // error messages during the time that we have *m_accessMutexPtr
      // locked.
      bool overflowFlag = false;
      {
        // Get exclusive control of the DistributionQueue.
        Token token = this->getToken();
        this->waitForLock(token);
        try {
          // Increment head, and wrap around if necessary.
          size_t newHeadIndex = *m_headIndexPtr + 1;
          if(newHeadIndex == m_size) {
            newHeadIndex = 0;
          }
          
          // Check for overflow.
          if(newHeadIndex == *m_tailIndexPtr) {
            overflowFlag = true;
          } else {
            // Proceed with the copying, etc.
            m_dataArray[*m_headIndexPtr] = element;
          }
          *m_headIndexPtr = newHeadIndex;
          
          // Wake up any threads that are waiting for data.
          this->signalAll(m_headChangedCondition);
        } catch(...) {
          this->releaseLock();
          throw;
        }
        this->releaseLock();
      }

      // Report any errors.
      if(overflowFlag) {
        std::ostringstream message;
        message << "Queue size exceeds the allowable maximum of "
                << m_size << " " << "on a function call from thread " 
                << (m_clientNameMapPtr->find(clientID.m_idNumber))->second
                << ".";
        DLR_THROW(OverflowException, "DistributionQueue::pushFront()",
                  message.str().c_str());
      }
    }


    // This member function introduces a thread to a
    // DistributionQueue instance.
    template <class Type>
    void 
    DistributionQueue<Type>::
    registerClient(ClientID& clientID, const std::string& clientName)
    {
      // Registration must be strictly serialized.  Only one thread
      // can register at once.
      Token token = this->getToken();
      this->waitForLock(token);

      try {
        m_clientNameMapPtr->insert(
          std::make_pair(*m_numberOfClientsPtr, clientName));
        clientID.m_count = 0;
        clientID.m_idNumber = (*m_numberOfClientsPtr)++;
        clientID.m_index = 0;

        // If we now have more than the expected number of clients, we
        // have to resize our internal storage.
        if(*m_numberOfClientsPtr > *m_maxNumberOfClientsPtr) {
          size_t oldMaxNumberOfClients = *m_maxNumberOfClientsPtr;
          Type* oldBufferArray = *m_bufferArrayPtr;
        
          *m_maxNumberOfClientsPtr *= 2;
          *m_bufferArrayPtr = new Type[*m_maxNumberOfClientsPtr];

          std::copy(oldBufferArray, oldBufferArray + oldMaxNumberOfClients,
                    *m_bufferArrayPtr);
          delete[] oldBufferArray;        
        }
      } catch(...) {
        this->releaseLock();
        throw;
      }
      this->releaseLock();
    }
    

    // This member function releases a lock acquired by
    // lockBack().
    template <class Type>
    void
    DistributionQueue<Type>::
    unlockBack(const ClientID& clientID)
    {
      Token token = this->getToken();
      this->releaseLock();
    }

    
    template <class Type>
    void
    DistributionQueue<Type>::
    releaseLock()
    {
      *m_isLockedPtr = false;
      this->signalOne(m_isLockedCondition);
    }

    
    template <class Type>
    void
    DistributionQueue<Type>::
    waitForLock(Token& token)
    {
      while(*m_isLockedPtr) {
        this->wait(this->m_isLockedCondition, token);
      }
      *m_isLockedPtr = true;
    }


    template <class Type>
    bool
    DistributionQueue<Type>::
    waitForLock(Token& token, double timeout)
    {
      double deadline = getCurrentTime() + timeout;
      while(*m_isLockedPtr) {
        if(this->wait(this->m_isLockedCondition, token,
                      deadline - getCurrentTime()) == false) {
          return false;
        }
      }
      *m_isLockedPtr = true;
      return true;
    }
    
    
  } // namespace thread

} // namespace dlr

#endif /* #ifndef _DLR_THREAD_DISTRIBUTIONQUEUE_H_ */
