/**
***************************************************************************
* @file dlrThread/clientID.h 
* Header file declaring ClientID class.
*
* Copyright (C) 2006 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 886 $
* $Date: 2007-05-04 01:59:27 -0400 (Fri, 04 May 2007) $
***************************************************************************
**/

#ifndef _DLR_THREAD_CLIENTID_H_
#define _DLR_THREAD_CLIENTID_H_

#include <dlrCommon/exception.h>

namespace dlr {

  namespace thread {

    // Forward declaration.
    template<class Type> class BroadcastQueue;
    template<class Type> class DistributionQueue;
    
    /** 
     ** This class is used to identify individual threads to
     ** individual BroadcastQueue instances.  Each thread that wishes to
     ** interact with a particular BroadcastQueue instance must create a
     ** ClientID instance, and then pass the ClientID to the
     ** registerClient() member function of that BroadcastQueue instance
     ** exactly once.  A particular ClientID must be used with only
     ** one BroadcastQueue instance.  ClientID's public interface is
     ** completely opaque: the thread code should to nothing with the
     ** ClientID except pass it to the member functions of the
     ** associated BroadcastQueue instance.
     **/
    template <class Type>
    class ClientID {
      friend class BroadcastQueue<Type>;
      friend class DistributionQueue<Type>;

    public:

      /** 
       * The default constructor creates a ClientID.
       */
      ClientID()
        : m_count(0), m_idNumber(0), m_index(0) {}

      
      /** 
       * The destructor destroys the ClientID instance and cleans up
       * any associated system resources.
       */
      ~ClientID() {}
      

    private:

      /** 
       * For now, copying ClientID instances is forbidden.  The copy
       * constructor will be made public only if it turns out that
       * there is good reason to do so.
       */
      ClientID(const ClientID&) {
        DLR_THROW(LogicException, "ClientID::ClientID()",
                  "Copying ClientID instances is disallowed.");
      }
                
      
      /** 
       * For now, copying ClientID instances is forbidden.  The
       * assignment operator will be made public only if it turns out
       * that there is good reason to do so.
       */
      ClientID&
      operator=(const ClientID& source) {
        DLR_THROW(LogicException, "ClientID::operator=()",
                  "Copying ClientID instances is disallowed.");
      }

      size_t m_count;
      size_t m_idNumber;
      size_t m_index;
    };

  } // namespace thread

} // namespace dlr

#endif /* #ifndef _DLR_THREAD_CLIENTID_H_ */
