/**
***************************************************************************
* @file gaussianDistributionTest.cpp
* Source file defining GaussianDistributionTest class.
*
* Copyright (C) 2006 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 906 $
* $Date: 2007-05-17 15:04:14 -0400 (Thu, 17 May 2007) $
***************************************************************************
**/

#include <stdlib.h>

#include <dlrStochastics/gaussianDistribution.h>
#include <dlrNumeric/utilities.h>

#include <dlrTest/testFixture.h>

namespace dlr {

  class GaussianDistributionTest
    : public TestFixture<GaussianDistributionTest>
  {
  public:

    GaussianDistributionTest();
    ~GaussianDistributionTest() {}

    void setUp(const std::string& testName) {}
    void tearDown(const std::string& testName) {}

    void testAddSample();

  private:

    template <class TYPE>
    bool equivalent(const Array1D<TYPE>& arg0,
                    const Array1D<TYPE>& arg1,
                    TYPE tolerance);

    template <class TYPE>
    bool equivalent(const Array2D<TYPE>& arg0,
                    const Array2D<TYPE>& arg1,
                    TYPE tolerance);

    Array1D<double>
    getRandomSample(size_t dimensionality);
    
    double m_defaultTolerance;
    
  }; // class GaussianDistributionTest


  /* ============== Member Function Definititions ============== */

  GaussianDistributionTest::
  GaussianDistributionTest()
    : TestFixture<GaussianDistributionTest>("GaussianDistributionTest"),
      m_defaultTolerance(1.0E-6)
  {
    // Register all tests.
    DLR_TEST_REGISTER_MEMBER(testAddSample);
  }


  void
  GaussianDistributionTest::
  testAddSample()
  {
    const size_t numberOfSamples = 10000;
    const size_t dimensionality = 3;

    // Create the distribution to be tested.
    GaussianDistribution< Array1D<double> > distribution;
    
    // Build samples and add them to the distribution.
    Array2D<double> sampleArray(numberOfSamples, dimensionality);
    for(size_t sampleIndex = 0; sampleIndex < numberOfSamples; ++sampleIndex) {
      Array1D<double> newSample = this->getRandomSample(dimensionality);
      sampleArray.row(sampleIndex).copy(newSample);
      distribution.addSample(newSample.copy());
    }

    // Compute mean and covariance.
    Array1D<double> referenceMean;
    Array2D<double> referenceCovariance;
    getMeanAndCovariance(sampleArray, referenceMean, referenceCovariance);

    // Verify result.
    DLR_TEST_ASSERT(
      this->equivalent(
        distribution.getMean(), referenceMean, m_defaultTolerance));
    DLR_TEST_ASSERT(
      this->equivalent(
        distribution.getCovariance(), referenceCovariance,
        m_defaultTolerance));
  }
  
  
  template <class TYPE>
  bool
  GaussianDistributionTest::
  equivalent(const Array1D<TYPE>& array0,
             const Array1D<TYPE>& array1,
             TYPE tolerance)
  {
    if(array0.size() != array1.size()) {
      return false;
    }
    return std::equal(array0.begin(), array0.end(), array1.begin(),
                      ApproximatelyEqualFunctor<TYPE>(tolerance));
  }


  template <class TYPE>
  bool
  GaussianDistributionTest::
  equivalent(const Array2D<TYPE>& array0,
             const Array2D<TYPE>& array1,
             TYPE tolerance)
  {
    if(array0.rows() != array1.rows()) {
      return false;
    }
    if(array0.columns() != array1.columns()) {
      return false;
    }
    return std::equal(array0.begin(), array0.end(), array1.begin(),
                      ApproximatelyEqualFunctor<TYPE>(tolerance));
  }


  Array1D<double>
  GaussianDistributionTest::
  getRandomSample(size_t dimensionality)
  {
    Array1D<double> returnValue(dimensionality);
    double randMax = static_cast<double>(RAND_MAX);
    for(size_t elementIndex = 0; elementIndex < dimensionality;
        ++elementIndex) {
      returnValue[elementIndex] = rand() / randMax;
    }
    return returnValue;
  }
  
} // namespace dlr


#if 0

int main(int argc, char** argv)
{
  dlr::GaussianDistributionTest currentTest;
  bool result = currentTest.run();
  return (result ? 0 : 1);
}

#else

namespace {

  dlr::GaussianDistributionTest currentTest;

}

#endif
