/**
***************************************************************************
* @file dlrPortability/timeUtilities.cpp
*
* Source file declaring some useful time-related routines.
*
* (C) Copyright 2004-2006 David LaRose, dlr@alumni.carnegiemellon.edu
* See accompanying LICENSE file for details.
*
* $Revision: 882 $
* $Date: 2007-05-04 00:40:09 -0400 (Fri, 04 May 2007) $
***************************************************************************
**/

#include <dlrPortability/timeUtilities.h>

/* ===================== Common includes ===================== */


/* ===================== End common includes ===================== */

#ifdef _WIN32

/* ===================== Windows includes ===================== */

#include <time.h>
#include <windows.h>
#include <sys/timeb.h>

/* ===================== End Windows includes ===================== */

#else /* #ifdef _WIN32 */

/* ===================== Linux includes ===================== */

#include <time.h>
#include <sys/time.h>
#include <sys/select.h>

/* ===================== End Linux includes ===================== */

#endif /* #ifdef _WIN32 */


/* ===================== Common code ===================== */

namespace dlr {

  namespace portability {

    void preciseSleep(int milliseconds)
    {
      portableSleep(static_cast<double>(milliseconds) / 1000.0);
    }
    
  } // namespace portability

} // namespace dlr

/* ===================== End common code ===================== */


#ifdef _WIN32

/* ===================== Windows code ===================== */

// Anonymous namespace for stuff local to this file.
namespace dlr {

  namespace portability {

    double
    getCurrentTime() {
      struct __timeb64 tp;
      _ftime64(&tp);
      return tp.time + (tp.millitm * 0.001);
    }
    
    
    void
    portableSleep(double seconds) {
      if(seconds > 0.0) {
        int milliseconds = static_cast<int>(1000.0 * seconds + 0.5);
        Sleep(milliseconds);
      }
    }
    
  } // namespace portability
  
} // namespace dlr

/* ===================== End Windows code ===================== */

#else /* #ifdef _WIN32 */

/* ===================== Linux code ===================== */

#include <cmath>

namespace dlr {

  namespace portability {

    double
    getCurrentTime() {
      struct timeval tp;
      gettimeofday(&tp, 0);
      return tp.tv_sec + (tp.tv_usec * 0.000001);
    }


    void
    portableSleep(double seconds) {
      if(seconds > 0.0) {
        double integerPart;
        double fractionalPart = std::modf(seconds, &integerPart);
        struct timeval timeout;
        timeout.tv_sec = static_cast<int>(integerPart);
        timeout.tv_usec =
          static_cast<int>(1000000.0 * fractionalPart + 0.5);
        if(timeout.tv_usec == 1000000) {
          ++timeout.tv_sec;
          timeout.tv_usec = 0;
        }
        select(0, NULL, NULL, NULL, &timeout);
      }
    }

  } // namespace portability

} // namespace dlr

/* ===================== End Linux code ===================== */

#endif /* #ifdef _WIN32 */
