/**
**********************************************************************
* @file dlrOptimization/gradientFunction.h
*
* Header file declaring GradientFunction class.
*
* Copyright (C) 2003-2007 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 881 $ 
* $Date: 2007-05-04 00:37:20 -0400 (Fri, 04 May 2007) $ 
**********************************************************************
**/

#ifndef _DLR_GRADIENTFUNCTION_H_
#define _DLR_GRADIENTFUNCTION_H_

#include <functional>

namespace dlr {

  namespace optimization {

    /**
     ** The GradientFunction class template is derived from
     ** std::unary_function, and adds one additional member function for
     ** computing the gradient of the function output with respect to
     ** the argument.
     **
     ** Template argument Functor is assumed to be a subclass of
     ** std::unary_function.  Functor::result_type is assumed to be a
     ** continuous scalar type.  Functor::argument_type is assumed to be
     ** a vector or 1D array type which supports the following
     ** interface:
     **
     **   argument_type(size_t N): construct an N-element vector.
     **   size_t size(): return the number of elements in the vector.
     **   element_type& operator[](size_t i): return a reference to the
     **                        (i)th element of the array.
     **   const element_type& operator[](size_t i) const: return a
     **                        const reference to the (i)th element of
     **                        the array.
     **
     ** It is further assumed that element type of argument_type is a
     ** continuous scalar, and can be implicitly cast to and from
     ** double.
     **/
    template <class Functor>
    class GradientFunction
      : public std::unary_function<typename Functor::argument_type,
                                   typename Functor::result_type>
    {
    public:
      /** 
       * This is the default constructor.
       *
       * @param functor This argument is the function object to be
       * adapted.
       *
       * @param epsilon If the gradient() method is not overridden in a
       * subclass, the gradient will be computed by using symmetric
       * divided differences with a total step size of 2 * epsilon.
       */
      GradientFunction(const Functor& functor, double epsilon=1.0e-6) :
        m_functor(functor), m_epsilon(epsilon) {
        if(epsilon == 0.0) {
          DLR_THROW3(ValueException, "GradientFunction::GradientFunction()",
                     "Invalid value (0.0) for argument epsilon.");
        }
      }

      /** 
       * Destructor.
       */
      virtual ~GradientFunction() {}

      /** 
       * This method numerically approximates the gradient of
       * this->operator() by divided differences.  This method should
       * often be overridden by a subclass.
       * 
       * @param theta The point around which to compute the gradient.
       * @return The computed gradient.
       */
      typename Functor::argument_type
      gradient(const typename Functor::argument_type& theta);

      /** 
       * This method evaluates the function value a the specified point.
       * 
       * @param theta The point at which to evaluate the function.
       * @return The function value at theta.
       */
      typename Functor::result_type
      operator()(const typename Functor::argument_type& theta) {
        return m_functor(theta);
      }

    private:
      Functor m_functor;
      double m_epsilon;

    }; // class GradientFunction

  } // namespace optimization

} // namespace dlr


/* ======= Declarations to maintain compatibility with legacy code. ======= */

namespace dlr {

  using optimization::GradientFunction;

} // namespace dlr


/*******************************************************************
 * Member function definitions follow.  This would be a .C file
 * if it weren't templated.
 *******************************************************************/

namespace dlr {

  namespace optimization {

    // Numerically approximate the gradient of this->operator() by
    // divided differences.
    template <class Functor>
    typename Functor::argument_type
    GradientFunction<Functor>::
    gradient(const typename Functor::argument_type& theta)
    {
      // Create some vectors to use as input to operator()().
      typename Functor::argument_type thetaMinus(theta.size());
      typename Functor::argument_type thetaPlus(theta.size());
      // Return value must be a vector, too, so use argument_type.
      typename Functor::argument_type result(theta.size());
      // Initialize arguments.
      for(size_t index = 0; index < theta.size(); ++index) {
        thetaMinus[index] = theta[index];
        thetaPlus[index] = theta[index];
      }
      // Now compute each partial derivative.
      for(size_t index = 0; index < theta.size(); ++index) {
        // Set up the difference.
        thetaMinus[index] = theta[index] - m_epsilon;
        thetaPlus[index] = theta[index] + m_epsilon;
        // Compute divided difference.
        typename Functor::result_type valueMinus =
          m_functor.operator()(thetaMinus);
        typename Functor::result_type valuePlus =
          m_functor.operator()(thetaPlus);
        result[index] = (valuePlus - valueMinus) / (2 * m_epsilon);
        // Reset argument values.
        thetaMinus[index] = theta[index];
        thetaPlus[index] = theta[index];
      }
      return result;
    }

  } // namespace optimization

} // namespace dlr

#endif // #ifndef _DLR_GRADIENTFUNCTION_H_
