/**
***************************************************************************
* @file vector3D.cpp
*
* Source file defining Vector3D class.
*
* Copyright (C) 2000-2007 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 880 $
* $Date: 2007-05-04 00:33:49 -0400 (Fri, 04 May 2007) $
***************************************************************************
**/

#include <dlrNumeric/vector3D.h>

namespace dlr {

  namespace numeric {
    
    Vector3D
    operator+(const Vector3D& vector0, const Vector3D& vector1)
    {
      return Vector3D(vector0.x() + vector1.x(),
                      vector0.y() + vector1.y(),
                      vector0.z() + vector1.z());
    }
  
    Vector3D
    operator-(const Vector3D& vector0, const Vector3D& vector1)
    {
      return Vector3D(vector0.x() - vector1.x(),
                      vector0.y() - vector1.y(),
                      vector0.z() - vector1.z());
    }
  
    Vector3D
    operator*(const Vector3D& vector0, const Vector3D& vector1)
    {
      return Vector3D(vector0.x() * vector1.x(),
                      vector0.y() * vector1.y(),
                      vector0.z() * vector1.z());
    }
  
    Vector3D
    operator/(const Vector3D& vector0, const Vector3D& vector1)
    {
      return Vector3D(vector0.x() / vector1.x(),
                      vector0.y() / vector1.y(),
                      vector0.z() / vector1.z());
    }
  
    Vector3D
    operator+(const Vector3D& vector0, double scalar)
    {
      return Vector3D(vector0.x() + scalar,
                      vector0.y() + scalar,
                      vector0.z() + scalar);
    }

    Vector3D
    operator-(const Vector3D& vector0, double scalar)
    {
      return Vector3D(vector0.x() - scalar,
                      vector0.y() - scalar,
                      vector0.z() - scalar);
    }

    Vector3D
    operator*(const Vector3D& vector0, double scalar)
    {
      return Vector3D(vector0.x() * scalar,
                      vector0.y() * scalar,
                      vector0.z() * scalar);
    }
  
    Vector3D
    operator/(const Vector3D& vector0, double scalar)
    {
      return Vector3D(vector0.x() / scalar,
                      vector0.y() / scalar,
                      vector0.z() / scalar);
    }

    bool
    operator==(const Vector3D& vector0, const Vector3D& vector1)
    {
      return((vector0.x() == vector1.x()) &&
             (vector0.y() == vector1.y()) &&
             (vector0.z() == vector1.z()));
    }

    bool
    operator!=(const Vector3D& vector0, const Vector3D& vector1)
    {
      return(!operator==(vector0, vector1));
    }

    std::ostream&
    operator<<(std::ostream& stream, const Vector3D& vector0)
    {
      stream << "Vector3D(" << vector0.x() << ", " << vector0.y() << ", "
             << vector0.z() << ")";
      return stream;
    }

    std::istream&
    operator>>(std::istream& stream, Vector3D& vector0)
    {
      const char intro[] = "Vector3D(";
      const char intermission[] = ",";
      const char outro[] = ")";
      double x, y, z;
      char inChar;
      size_t index;

      for(index = 0; index < strlen(intro); ++index) {
        inChar = 0;
        stream >> inChar;
        if(inChar != intro[index]) {
          // Hmm.  g++ diverges from the standard?
          // stream.clear(ios_base::failbit);
          stream.clear(std::ios::failbit);
          return stream;
        }
      }
      stream >> x;
      for(index = 0; index < strlen(intermission); ++index) {
        inChar = 0;
        stream >> inChar;
        if(inChar != intermission[index]) {
          // stream.clear(ios_base::failbit);
          stream.clear(std::ios::failbit);
          return stream;
        }
      }
      stream >> y;
      for(index = 0; index < strlen(intermission); ++index) {
        inChar = 0;
        stream >> inChar;
        if(inChar != intermission[index]) {
          // stream.clear(ios_base::failbit);
          stream.clear(std::ios::failbit);
          return stream;
        }
      }
      stream >> z;
      for(index = 0; index < strlen(outro); ++index) {
        inChar = 0;
        stream >> inChar;
        if(inChar != outro[index]) {
          // stream.clear(ios_base::failbit);
          stream.clear(std::ios::failbit);
          return stream;
        }
      }
      if(stream) {
        vector0.setValue(x, y, z);
      }
      return stream;
    }

  } // namespace numeric

} // namespace dlr
