/**
***************************************************************************
* @file utilitiesTest.cpp
*
* Source file defining UtilitiesTest class.
*
* Copyright (C) 2005 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 906 $
* $Date: 2007-05-17 15:04:14 -0400 (Thu, 17 May 2007) $
***************************************************************************
**/

#include <stdlib.h>
#include <dlrNumeric/utilities.h>

#include <dlrTest/testFixture.h>

namespace dlr {

  class UtilitiesTest : public TestFixture<UtilitiesTest> {

  public:

    UtilitiesTest();
    ~UtilitiesTest() {}

    void setUp(const std::string& testName) {}
    void tearDown(const std::string& testName) {}

    void testGetMeanAndCovariance();    
    void testSolveQuadratic();

  private:

    template <class TYPE>
    bool equivalent(const Array1D<TYPE>& arg0,
                    const Array1D<TYPE>& arg1,
                    TYPE tolerance);

    template <class TYPE>
    bool equivalent(const Array2D<TYPE>& arg0,
                    const Array2D<TYPE>& arg1,
                    TYPE tolerance);

    Array1D<double>
    getRandomSample(size_t dimensionality);
    
    double m_defaultTolerance;
    
  }; // class UtilitiesTest


  /* ============== Member Function Definititions ============== */

  UtilitiesTest::
  UtilitiesTest()
    : TestFixture<UtilitiesTest>("UtilitiesTest"),
      m_defaultTolerance(1.0E-6)
  {
    // Register all tests.
    DLR_TEST_REGISTER_MEMBER(testGetMeanAndCovariance);
    DLR_TEST_REGISTER_MEMBER(testSolveQuadratic);
  }


  void
  UtilitiesTest::
  testGetMeanAndCovariance()
  {
    const size_t numberOfSamples = 10000;
    const size_t dimensionality = 3;

    // Build samples.
    Array2D<double> sampleArray(numberOfSamples, dimensionality);
    for(size_t sampleIndex = 0; sampleIndex < numberOfSamples; ++sampleIndex) {
      Array1D<double> newSample = this->getRandomSample(dimensionality);
      sampleArray.row(sampleIndex).copy(newSample);
    }

    // Compute sample mean.
    Array1D<double> referenceMean(dimensionality);
    referenceMean = 0.0;
    for(size_t sampleIndex = 0; sampleIndex < numberOfSamples; ++sampleIndex) {
      referenceMean += sampleArray.row(sampleIndex);
    }
    referenceMean /= static_cast<double>(numberOfSamples);

    // Compute sample covariance estimate.
    Array2D<double> referenceCovariance(dimensionality, dimensionality);
    referenceCovariance = 0.0;
    for(size_t sampleIndex = 0; sampleIndex < numberOfSamples; ++sampleIndex) {
      Array1D<double> sampleOffset =
        sampleArray.row(sampleIndex) - referenceMean;
      referenceCovariance += outerProduct(sampleOffset, sampleOffset);      
    }
    referenceCovariance /= static_cast<double>(numberOfSamples - 1);

    // Run routine under test.
    Array1D<double> testMean;
    Array2D<double> testCovariance;
    getMeanAndCovariance(sampleArray, testMean, testCovariance);

    // Verify result.
    DLR_TEST_ASSERT(
      this->equivalent(
        testMean, referenceMean, m_defaultTolerance));
    DLR_TEST_ASSERT(
      this->equivalent(
        testCovariance, referenceCovariance, m_defaultTolerance));

    // Run routine under test for column-major samples.
    getMeanAndCovariance(sampleArray.transpose(), testMean, testCovariance, 1);

    // Verify result.
    DLR_TEST_ASSERT(
      this->equivalent(
        testMean, referenceMean, m_defaultTolerance));
    DLR_TEST_ASSERT(
      this->equivalent(
        testCovariance, referenceCovariance, m_defaultTolerance));
  }
  
  
  void
  UtilitiesTest::
  testSolveQuadratic()
  {
    // Test for (2x + 4)(-x + 7) = 0;
    double c0 = -2.0;
    double c1 = 10.0;
    double c2 = 28.0;

    double root0;
    double root1;
    bool valid;
    solveQuadratic(c0, c1, c2, root0, root1, valid);

    if(root0 > root1) {
      std::swap(root0, root1);
    }
    DLR_TEST_ASSERT(valid == true);
    DLR_TEST_ASSERT(approximatelyEqual(root0, -2.0, m_defaultTolerance));
    DLR_TEST_ASSERT(approximatelyEqual(root1, 7.0, m_defaultTolerance));


    // Test for (2x + 4)(-x - 7) = 0;
    c0 = -2.0;
    c1 = -18.0;
    c2 = -28.0;

    solveQuadratic(c0, c1, c2, root0, root1, valid);

    if(root0 > root1) {
      std::swap(root0, root1);
    }
    DLR_TEST_ASSERT(valid == true);
    DLR_TEST_ASSERT(approximatelyEqual(root0, -7.0, m_defaultTolerance));
    DLR_TEST_ASSERT(approximatelyEqual(root1, -2.0, m_defaultTolerance));
  }


  template <class TYPE>
  bool
  UtilitiesTest::
  equivalent(const Array1D<TYPE>& array0,
             const Array1D<TYPE>& array1,
             TYPE tolerance)
  {
    if(array0.size() != array1.size()) {
      return false;
    }
    return std::equal(array0.begin(), array0.end(), array1.begin(),
                      ApproximatelyEqualFunctor<TYPE>(tolerance));
  }


  template <class TYPE>
  bool
  UtilitiesTest::
  equivalent(const Array2D<TYPE>& array0,
             const Array2D<TYPE>& array1,
             TYPE tolerance)
  {
    if(array0.rows() != array1.rows()) {
      return false;
    }
    if(array0.columns() != array1.columns()) {
      return false;
    }
    return std::equal(array0.begin(), array0.end(), array1.begin(),
                      ApproximatelyEqualFunctor<TYPE>(tolerance));
  }


  Array1D<double>
  UtilitiesTest::
  getRandomSample(size_t dimensionality)
  {
    Array1D<double> returnValue(dimensionality);
    double randMax = static_cast<double>(RAND_MAX);
    for(size_t elementIndex = 0; elementIndex < dimensionality;
        ++elementIndex) {
      returnValue[elementIndex] = rand() / randMax;
    }
    return returnValue;
  }
  
} // namespace dlr


#if 0

int main(int argc, char** argv)
{
  dlr::UtilitiesTest currentTest;
  bool result = currentTest.run();
  return (result ? 0 : 1);
}

#else

namespace {

  dlr::UtilitiesTest currentTest;

}

#endif
