/**
***************************************************************************
* @file amanatidesWoo3DTest.cpp
* 
* Source file defining AmanatidesWoo3DTest class.
*
* Copyright (C) 2004 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 939 $
* $Date: 2007-06-09 13:08:18 -0400 (Sat, 09 Jun 2007) $
***************************************************************************
**/

#include <math.h>
#include <dlrNumeric/array3D.h>
#include <dlrNumeric/vector3D.h>
#include <dlrNumeric/utilities.h>
#include <dlrNumeric/amanatidesWoo3D.h>

#include <dlrTest/testFixture.h>

namespace dlr {

  class AmanatidesWoo3DTest : public TestFixture<AmanatidesWoo3DTest> {

  public:

    AmanatidesWoo3DTest();
    ~AmanatidesWoo3DTest() {}

    void setUp(const std::string& testName) {}
    void tearDown(const std::string& testName) {}

    void testConstructor();
    void testCopyConstructor();
    void testDestructor();
    void testBegin0();
    void testBegin1();
    void testBegin2();
    void testBegin3();
    void testBegin4();
    void testEnd();
    void testGetData();
    void testValidIntersection0();
    void testValidIntersection1();
    void testAssignmentOperator();

  private:
    void
    compareImages(const Array3D<double>& testImage,
                  const Array3D<double>& groundTruthImage);

    
    void
    traverseImage(AmanatidesWoo3D< Array3D<double> >& aw3D);

    
    Vector3D m_directionXYZ0;
    Vector3D m_directionXYZ1;
    Vector3D m_directionXYZ2;
    Array3D<double> m_downStreamImage;
    Array3D<double> m_fullImage;
    Transform3D m_pixelTworld;
    Vector3D m_startXYZ0;
    Vector3D m_startXYZ1;
    Vector3D m_startXYZ2;
    double m_testEpsilon;
    Array3D<double> m_zeroImage;
    
  }; // class AmanatidesWoo3DTest


  /* ============== Member Function Definititions ============== */

  AmanatidesWoo3DTest::
  AmanatidesWoo3DTest()
    : TestFixture<AmanatidesWoo3DTest>("AmanatidesWoo3DTest"),
      m_directionXYZ0(),
      m_directionXYZ1(),
      m_directionXYZ2(),
      m_downStreamImage(),
      m_fullImage(),
      m_pixelTworld(),
      m_startXYZ0(),
      m_startXYZ1(),
      m_startXYZ2(),
      m_testEpsilon(1.0e-8),
      m_zeroImage()
  {
    // Register all tests.
    DLR_TEST_REGISTER_MEMBER(testConstructor);
    DLR_TEST_REGISTER_MEMBER(testCopyConstructor);
    DLR_TEST_REGISTER_MEMBER(testDestructor);
    DLR_TEST_REGISTER_MEMBER(testBegin0);
    DLR_TEST_REGISTER_MEMBER(testBegin1);
    DLR_TEST_REGISTER_MEMBER(testBegin2);
    DLR_TEST_REGISTER_MEMBER(testBegin3);
    DLR_TEST_REGISTER_MEMBER(testBegin4);
    DLR_TEST_REGISTER_MEMBER(testEnd);
    DLR_TEST_REGISTER_MEMBER(testGetData);
    DLR_TEST_REGISTER_MEMBER(testValidIntersection0);
    DLR_TEST_REGISTER_MEMBER(testValidIntersection1);
    DLR_TEST_REGISTER_MEMBER(testAssignmentOperator);

    // Set up ground truth.  Each element of these arrays represents
    // ray traversal distance through the corresponding region of
    // space.  See the documentation file AmanatidesWoo3DTest.fig for
    // more details.
    m_downStreamImage = Array3D<double>(
      "[[[0.0,          0.0,          0.0,          0.0],"
      "  [0.0,          0.0,          0.0,          0.0],"
      "  [0.0,          0.0,          0.0,          0.0]],"
      " [[0.0,          0.0,          0.0,          0.0],"
      "  [0.0,          1.3017082793, 0.6508541397, 0.0],"
      "  [0.0,          0.0,          0.0,          0.0]],"
      " [[0.0,          0.0,          1.3017082793, 1.9525624189],"
      "  [0.0,          0.0,          0.6508541397, 0.0],"
      "  [0.0,          0.0,          0.0,          0.0]]]");

    m_fullImage = Array3D<double>(
      "[[[0.0,          0.0,          0.0,          0.0],"
      "  [0.6508541397, 0.0,          0.0,          0.0],"
      "  [1.3017082793, 0.0,          0.0,          0.0]],"
      " [[0.0,          0.0,          0.0,          0.0],"
      "  [0.6508541397, 2.6034165586, 0.6508541397, 0.0],"
      "  [0.0,          0.0,          0.0,          0.0]],"
      " [[0.0,          0.0,          1.3017082793, 1.9525624189],"
      "  [0.0,          0.0,          0.6508541397, 0.0],"
      "  [0.0,          0.0,          0.0,          0.0]]]");

    m_zeroImage = zeros(3, 3, 4, Double);

    // The "XYZ0" data members describe a ray which originates inside
    // the pixel array.
    m_startXYZ0 = Vector3D(11.0, 8.0, 2.0);
    Vector3D endXYZ0(17.0, 5.0, -2.0);
    m_directionXYZ0 = endXYZ0 - m_startXYZ0;
    m_directionXYZ0 /= magnitude(m_directionXYZ0);

    // The "XYZ1" data members describe a ray which originates outside
    // the pixel array, but _does_ intersect the pixel array.
    // m_directionXYZ1 points toward the pixel array.
    m_startXYZ1 = Vector3D(5.0, 11.0, 6.0);
    Vector3D endXYZ1(17.0, 5.0, -2.0);
    m_directionXYZ1 = endXYZ1 - m_startXYZ1;
    m_directionXYZ1 /= magnitude(m_directionXYZ1);

    // The "XYZ2" data members describe a ray which originates outside
    // the pixel array, but _does not_ intersect the pixel array.
    m_startXYZ2 = Vector3D(5, 11.0, 6.0);
    Vector3D endXYZ2(6.0, 6.0, 5.0);
    m_directionXYZ2 = endXYZ2 - m_startXYZ2;
    m_directionXYZ2 /= magnitude(m_directionXYZ2);
    
    // Coordinate transformation which converts world coordinates to pixel
    // coordinates.
    m_pixelTworld = Transform3D(0.5, 0.0, 0.0, -4.0,
                                0.0, 0.5, 0.0, -2.5,
                                0.0, 0.0, -0.5, 2.5,
                                0.0, 0.0, 0.0, 1.0);
  }


  void
  AmanatidesWoo3DTest::
  testConstructor()
  {
    // Passing testBegin*() methods is sufficient.
  }


  void
  AmanatidesWoo3DTest::
  testCopyConstructor()
  {
    // Set up the image to be traversed.
    Array3D<double> traceImage = m_zeroImage.copy();

    // Construct the AmanatidesWoo3D instance.  Start point is inside
    // of the pixel array.  DownStreamOnly flag is true.
    AmanatidesWoo3D< Array3D<double> >
      aw3D(traceImage, m_pixelTworld, m_startXYZ0, m_directionXYZ0, true);

    // Construct a second AmanatidesWoo3D instance using the copy
    // constructor.
    AmanatidesWoo3D< Array3D<double> > aw3DCopy(aw3D);

    // Actual voxel traversal and image comparison is done in this
    // since it's identical for all of the testBegin*() functions.
    this->traverseImage(aw3DCopy);
    this->compareImages(traceImage, m_downStreamImage);
  }


  void
  AmanatidesWoo3DTest::
  testDestructor()
  {
    // No independent test for destructor.
  }

  
  void
  AmanatidesWoo3DTest::
  testBegin0()
  {
    // Set up the image to be traversed.
    Array3D<double> traceImage = m_zeroImage.copy();

    // Construct the AmanatidesWoo3D instance.  Start point is inside
    // of the pixel array.  DownStreamOnly flag is true.
    AmanatidesWoo3D< Array3D<double> >
      aw3D(traceImage, m_pixelTworld, m_startXYZ0, m_directionXYZ0, true);

    // Actual voxel traversal and image comparison is done in this
    // since it's identical for all of the testBegin*() functions.
    this->traverseImage(aw3D);
    this->compareImages(traceImage, m_downStreamImage);
  }

  void
  AmanatidesWoo3DTest::
  testBegin1()
  {
    // Set up the image to be traversed.
    Array3D<double> traceImage = m_zeroImage.copy();

    // Construct the AmanatidesWoo3D instance.  Start point is inside
    // of the pixel array.  DownStreamOnly flag is false.
    AmanatidesWoo3D< Array3D<double> >
      aw3D(traceImage, m_pixelTworld, m_startXYZ0, m_directionXYZ0, false);

    // Actual voxel traversal and image comparison is done in this
    // since it's identical for all of the testBegin*() functions.
    this->traverseImage(aw3D);
    this->compareImages(traceImage, m_fullImage);
  }

  void
  AmanatidesWoo3DTest::
  testBegin2()
  {
    // Set up the image to be traversed.
    Array3D<double> traceImage = m_zeroImage.copy();

    // Construct the AmanatidesWoo3D instance.  Start point is outside
    // of the pixel array.  DownStreamOnly flag is true.
    AmanatidesWoo3D< Array3D<double> >
      aw3D(traceImage, m_pixelTworld, m_startXYZ1, m_directionXYZ1, true);

    // Actual voxel traversal and image comparison is done in this
    // since it's identical for all of the testBegin*() functions.
    this->traverseImage(aw3D);
    this->compareImages(traceImage, m_fullImage);
  }

    void
  AmanatidesWoo3DTest::
  testBegin3()
  {
    // Set up the image to be traversed.
    Array3D<double> traceImage = m_zeroImage.copy();

    // Construct the AmanatidesWoo3D instance.  Start point is outside
    // of the pixel array.  DownStreamOnly flag is true, and direction
    // is away from the pixel array.
    AmanatidesWoo3D< Array3D<double> >
      aw3D(traceImage, m_pixelTworld, m_startXYZ1, -1 * m_directionXYZ1, true);

    // Actual voxel traversal and image comparison is done in this
    // since it's identical for all of the testBegin*() functions.
    this->traverseImage(aw3D);
    this->compareImages(traceImage, m_zeroImage);
  }

    void
  AmanatidesWoo3DTest::
  testBegin4()
  {
    // Set up the image to be traversed.
    Array3D<double> traceImage = m_zeroImage.copy();

    // Construct the AmanatidesWoo3D instance.  Start point is outside
    // of the pixel array.  Ray does not intersect the pixel array.
    AmanatidesWoo3D< Array3D<double> >
      aw3D(traceImage, m_pixelTworld, m_startXYZ2, m_directionXYZ2, false);

    // Actual voxel traversal and image comparison is done in this
    // since it's identical for all of the testBegin*() functions.
    this->traverseImage(aw3D);
    this->compareImages(traceImage, m_zeroImage);
  }


  void
  AmanatidesWoo3DTest::
  testEnd()
  {
    // Passing testBegin*() methods is sufficient.
  }


  void
  AmanatidesWoo3DTest::
  testGetData()
  {
    // Set up the image to be traversed.
    Array3D<double> traceImage = m_zeroImage.copy();

    // Construct the AmanatidesWoo3D instance.  Start point is inside
    // of the pixel array.  DownStreamOnly flag is true.
    AmanatidesWoo3D< Array3D<double> >
      aw3D(traceImage, m_pixelTworld, m_startXYZ0, m_directionXYZ0, true);

    // Actual voxel traversal and image comparison is done here.
    this->traverseImage(aw3D);
    this->compareImages(aw3D.getData(), traceImage);
  }

  
  void
  AmanatidesWoo3DTest::
  testValidIntersection0()
  {
    // Set up the image to be traversed.
    Array3D<double> traceImage = m_zeroImage.copy();

    // Construct the AmanatidesWoo3D instance.  Start point is inside
    // of the pixel array.  Ray does intersect the pixel array.
    AmanatidesWoo3D< Array3D<double> >
      aw3D(traceImage, m_pixelTworld, m_startXYZ0, m_directionXYZ0, true);

    // The validIntersection() member function should return true.
    DLR_TEST_ASSERT(aw3D.validIntersection());
  }


  void
  AmanatidesWoo3DTest::
  testValidIntersection1()
  {
    // Set up the image to be traversed.
    Array3D<double> traceImage = m_zeroImage.copy();

    // Construct the AmanatidesWoo3D instance.  Start point is outside
    // of the pixel array.  DownStreamOnly flag is true, and direction
    // is away from the pixel array.
    AmanatidesWoo3D< Array3D<double> >
      aw3D(traceImage, m_pixelTworld, m_startXYZ1, -1 * m_directionXYZ1, true);

    // The validIntersection() member function should return false.
    DLR_TEST_ASSERT(!(aw3D.validIntersection()));
  }

  
  void
  AmanatidesWoo3DTest::
  testAssignmentOperator()
  {
    // Set up the images to be traversed.
    Array3D<double> traceImage = m_zeroImage.copy();
    Array3D<double> traceImageCopy = m_zeroImage.copy();

    // Construct the AmanatidesWoo3D instance.  Start point is inside
    // of the pixel array.  DownStreamOnly flag is true.
    AmanatidesWoo3D< Array3D<double> >
      aw3D(traceImage, m_pixelTworld, m_startXYZ0, m_directionXYZ0, true);

    // Construct a second AmanatidesWoo3D having different contents.
    AmanatidesWoo3D< Array3D<double> >
      aw3DCopy(traceImageCopy, m_pixelTworld, m_startXYZ2, m_directionXYZ2,
               false);
    
    // Copy using the assignment operator.
    aw3DCopy = aw3D;

    // Actual voxel traversal and image comparison is done in this
    // since it's identical for all of the testBegin*() functions.
    this->traverseImage(aw3DCopy);
    this->compareImages(traceImage, m_downStreamImage);
  }

  
  void
  AmanatidesWoo3DTest::
  compareImages(const Array3D<double>& testImage,
                const Array3D<double>& groundTruthImage)
  {
    // Verify result.
    DLR_TEST_ASSERT(testImage.shape0() == groundTruthImage.shape0());
    DLR_TEST_ASSERT(testImage.shape1() == groundTruthImage.shape1());
    DLR_TEST_ASSERT(testImage.shape2() == groundTruthImage.shape2());
    for(size_t index = 0; index < groundTruthImage.size(); ++index) {
      double residual = std::fabs(testImage(index) - groundTruthImage(index));
      DLR_TEST_ASSERT(residual < m_testEpsilon);
    }
  }


  void
  AmanatidesWoo3DTest::
  traverseImage(AmanatidesWoo3D< Array3D<double> >& aw3D)
  {
    // Do the voxel traversal.
    typedef AmanatidesWoo3D< Array3D<double> >::iterator aw3DIterator;
    aw3DIterator endIterator = aw3D.end();
    for(aw3DIterator iter = aw3D.begin(); iter != endIterator; ++iter) {
      double parametricDistance = iter.tExit() - iter.tEntry();
      *iter = parametricDistance;
    }
  }

} // namespace dlr


#if 0

int main(int argc, char** argv)
{
  dlr::AmanatidesWoo3DTest currentTest;
  bool result = currentTest.run();
  return (result ? 0 : 1);
}

#else

namespace {

  dlr::AmanatidesWoo3DTest currentTest;

}

#endif
