/**
***************************************************************************
* @file amanatidesWoo2DTest.cpp
* 
* Source file defining AmanatidesWoo2DTest class.
*
* Copyright (C) 2004 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 939 $
* $Date: 2007-06-09 13:08:18 -0400 (Sat, 09 Jun 2007) $
***************************************************************************
**/

#include <math.h>
#include <dlrNumeric/array2D.h>
#include <dlrNumeric/vector2D.h>
#include <dlrNumeric/utilities.h>
#include <dlrNumeric/amanatidesWoo2D.h>

#include <dlrTest/testFixture.h>

namespace dlr {

  class AmanatidesWoo2DTest : public TestFixture<AmanatidesWoo2DTest> {

  public:

    AmanatidesWoo2DTest();
    ~AmanatidesWoo2DTest() {}

    void setUp(const std::string& testName) {}
    void tearDown(const std::string& testName) {}

    void testConstructor();
    void testCopyConstructor();
    void testDestructor();
    void testBegin0();
    void testBegin1();
    void testBegin2();
    void testBegin3();
    void testBegin4();
    void testEnd();
    void testGetData();
    void testValidIntersection0();
    void testValidIntersection1();
    void testAssignmentOperator();

  private:
    void
    compareImages(const Array2D<double>& testImage,
                  const Array2D<double>& groundTruthImage);
    void
    traverseImage(AmanatidesWoo2D< Array2D<double> >& aw2D);

    
    Vector2D m_directionXY0;
    Vector2D m_directionXY1;
    Vector2D m_directionXY2;
    Array2D<double> m_downStreamImage;
    Array2D<double> m_fullImage;
    Transform2D m_pixelTworld;
    Vector2D m_startXY0;
    Vector2D m_startXY1;
    Vector2D m_startXY2;
    double m_testEpsilon;
    Array2D<double> m_zeroImage;
    
  }; // class AmanatidesWoo2DTest


  /* ============== Member Function Definititions ============== */

  AmanatidesWoo2DTest::
  AmanatidesWoo2DTest()
    : TestFixture<AmanatidesWoo2DTest>("AmanatidesWoo2DTest"),
      m_directionXY0(),
      m_directionXY1(),
      m_directionXY2(),
      m_downStreamImage(),
      m_fullImage(),
      m_pixelTworld(),
      m_startXY0(),
      m_startXY1(),
      m_startXY2(),
      m_testEpsilon(1.0e-8),
      m_zeroImage()
  {
    // Register all tests.
    DLR_TEST_REGISTER_MEMBER(testConstructor);
    DLR_TEST_REGISTER_MEMBER(testCopyConstructor);
    DLR_TEST_REGISTER_MEMBER(testDestructor);
    DLR_TEST_REGISTER_MEMBER(testBegin0);
    DLR_TEST_REGISTER_MEMBER(testBegin1);
    DLR_TEST_REGISTER_MEMBER(testBegin2);
    DLR_TEST_REGISTER_MEMBER(testBegin3);
    DLR_TEST_REGISTER_MEMBER(testBegin4);
    DLR_TEST_REGISTER_MEMBER(testEnd);
    DLR_TEST_REGISTER_MEMBER(testGetData);
    DLR_TEST_REGISTER_MEMBER(testValidIntersection0);
    DLR_TEST_REGISTER_MEMBER(testValidIntersection1);
    DLR_TEST_REGISTER_MEMBER(testAssignmentOperator);

    // Set up ground truth.  Each element of these arrays represents
    // ray traversal distance through the corresponding region of
    // space.  See the documentation file AmanatidesWoo2DTest.fig for
    // more details.
    m_downStreamImage = Array2D<double>(
      "[[0.0,          0.0,          1.1180339887, 2.2360679775],"
      " [0.0,          1.1180339887, 1.1180339887, 0.0],"
      " [0.0,          0.0,          0.0,          0.0]]");
    m_fullImage = Array2D<double>(
      "[[0.0,          0.0,          1.1180339887, 2.2360679775],"
      " [1.1180339887, 2.2360679775, 1.1180339887, 0.0],"
      " [1.1180339887, 0.0,          0.0,          0.0]]");
    m_zeroImage = zeros(3, 4, Double);

    // The "XY0" data members describe a ray which originates inside
    // the pixel array.
    m_startXY0 = Vector2D(11.0, 8.0);
    Vector2D endXY0(17.0, 5.0);
    m_directionXY0 = endXY0 - m_startXY0;
    m_directionXY0 /= magnitude(m_directionXY0);

    // The "XY1" data members describe a ray which originates outside
    // the pixel array, but _does_ intersect the pixel array.
    // m_directionXY1 points toward the pixel array.
    m_startXY1 = Vector2D(5.0, 11.0);
    Vector2D endXY1(17.0, 5.0);
    m_directionXY1 = endXY1 - m_startXY1;
    m_directionXY1 /= magnitude(m_directionXY1);

    // The "XY2" data members describe a ray which originates outside
    // the pixel array, but _does not_ intersect the pixel array.
    m_startXY2 = Vector2D(5.0, 11.0);
    Vector2D endXY2(6.0, 6.0);
    m_directionXY2 = endXY2 - m_startXY2;
    m_directionXY2 /= magnitude(m_directionXY2);
    
    // Coordinate transformation which converts world coordinates to pixel
    // coordinates.
    m_pixelTworld = Transform2D(0.5, 0.0, -4.0,
                                0.0, 0.5, -2.5,
                                0.0, 0.0, 1.0);
  }


  void
  AmanatidesWoo2DTest::
  testConstructor()
  {
    // Passing testBegin*() methods is sufficient.
  }


  void
  AmanatidesWoo2DTest::
  testCopyConstructor()
  {
    // Set up the image to be traversed.
    Array2D<double> traceImage = m_zeroImage.copy();

    // Construct the AmanatidesWoo2D instance.  Start point is inside
    // of the pixel array.  DownStreamOnly flag is true.
    AmanatidesWoo2D< Array2D<double> >
      aw2D(traceImage, m_pixelTworld, m_startXY0, m_directionXY0, true);

    // Construct a second AmanatidesWoo2D instance using the copy
    // constructor.
    AmanatidesWoo2D< Array2D<double> > aw2DCopy(aw2D);

    // Actual voxel traversal and image comparison is done in this
    // since it's identical for all of the testBegin*() functions.
    this->traverseImage(aw2DCopy);
    this->compareImages(traceImage, m_downStreamImage);
  }


  void
  AmanatidesWoo2DTest::
  testDestructor()
  {
    // No independent test for destructor.
  }

  
  void
  AmanatidesWoo2DTest::
  testBegin0()
  {
    // Set up the image to be traversed.
    Array2D<double> traceImage = m_zeroImage.copy();

    // Construct the AmanatidesWoo2D instance.  Start point is inside
    // of the pixel array.  DownStreamOnly flag is true.
    AmanatidesWoo2D< Array2D<double> >
      aw2D(traceImage, m_pixelTworld, m_startXY0, m_directionXY0, true);

    // Actual voxel traversal and image comparison is done in this
    // since it's identical for all of the testBegin*() functions.
    this->traverseImage(aw2D);
    this->compareImages(traceImage, m_downStreamImage);
  }

  void
  AmanatidesWoo2DTest::
  testBegin1()
  {
    // Set up the image to be traversed.
    Array2D<double> traceImage = m_zeroImage.copy();

    // Construct the AmanatidesWoo2D instance.  Start point is inside
    // of the pixel array.  DownStreamOnly flag is false.
    AmanatidesWoo2D< Array2D<double> >
      aw2D(traceImage, m_pixelTworld, m_startXY0, m_directionXY0, false);

    // Actual voxel traversal and image comparison is done in this
    // since it's identical for all of the testBegin*() functions.
    this->traverseImage(aw2D);
    this->compareImages(traceImage, m_fullImage);
  }

  void
  AmanatidesWoo2DTest::
  testBegin2()
  {
    // Set up the image to be traversed.
    Array2D<double> traceImage = m_zeroImage.copy();

    // Construct the AmanatidesWoo2D instance.  Start point is outside
    // of the pixel array.  DownStreamOnly flag is true.
    AmanatidesWoo2D< Array2D<double> >
      aw2D(traceImage, m_pixelTworld, m_startXY1, m_directionXY1, true);

    // Actual voxel traversal and image comparison is done in this
    // since it's identical for all of the testBegin*() functions.
    this->traverseImage(aw2D);
    this->compareImages(traceImage, m_fullImage);
  }

    void
  AmanatidesWoo2DTest::
  testBegin3()
  {
    // Set up the image to be traversed.
    Array2D<double> traceImage = m_zeroImage.copy();

    // Construct the AmanatidesWoo2D instance.  Start point is outside
    // of the pixel array.  DownStreamOnly flag is true, and direction
    // is away from the pixel array.
    AmanatidesWoo2D< Array2D<double> >
      aw2D(traceImage, m_pixelTworld, m_startXY1, -1 * m_directionXY1, true);

    // Actual voxel traversal and image comparison is done in this
    // since it's identical for all of the testBegin*() functions.
    this->traverseImage(aw2D);
    this->compareImages(traceImage, m_zeroImage);
  }

    void
  AmanatidesWoo2DTest::
  testBegin4()
  {
    // Set up the image to be traversed.
    Array2D<double> traceImage = m_zeroImage.copy();

    // Construct the AmanatidesWoo2D instance.  Start point is outside
    // of the pixel array.  Ray does not intersect the pixel array.
    AmanatidesWoo2D< Array2D<double> >
      aw2D(traceImage, m_pixelTworld, m_startXY2, m_directionXY2, false);

    // Actual voxel traversal and image comparison is done in this
    // since it's identical for all of the testBegin*() functions.
    this->traverseImage(aw2D);
    this->compareImages(traceImage, m_zeroImage);
  }


  void
  AmanatidesWoo2DTest::
  testEnd()
  {
    // Passing testBegin*() methods is sufficient.
  }


  void
  AmanatidesWoo2DTest::
  testGetData()
  {
    // Set up the image to be traversed.
    Array2D<double> traceImage = m_zeroImage.copy();

    // Construct the AmanatidesWoo2D instance.  Start point is inside
    // of the pixel array.  DownStreamOnly flag is true.
    AmanatidesWoo2D< Array2D<double> >
      aw2D(traceImage, m_pixelTworld, m_startXY0, m_directionXY0, true);

    // Actual voxel traversal and image comparison is done here.
    this->traverseImage(aw2D);
    this->compareImages(aw2D.getData(), traceImage);
  }

  
  void
  AmanatidesWoo2DTest::
  testValidIntersection0()
  {
    // Set up the image to be traversed.
    Array2D<double> traceImage = m_zeroImage.copy();

    // Construct the AmanatidesWoo2D instance.  Start point is inside
    // of the pixel array.  Ray does intersect the pixel array.
    AmanatidesWoo2D< Array2D<double> >
      aw2D(traceImage, m_pixelTworld, m_startXY0, m_directionXY0, true);

    // The validIntersection() member function should return true.
    DLR_TEST_ASSERT(aw2D.validIntersection());
  }


  void
  AmanatidesWoo2DTest::
  testValidIntersection1()
  {
    // Set up the image to be traversed.
    Array2D<double> traceImage = m_zeroImage.copy();

    // Construct the AmanatidesWoo2D instance.  Start point is outside
    // of the pixel array.  DownStreamOnly flag is true, and direction
    // is away from the pixel array.
    AmanatidesWoo2D< Array2D<double> >
      aw2D(traceImage, m_pixelTworld, m_startXY1, -1 * m_directionXY1, true);

    // The validIntersection() member function should return false.
    DLR_TEST_ASSERT(!(aw2D.validIntersection()));
  }

  
  void
  AmanatidesWoo2DTest::
  testAssignmentOperator()
  {
    // Set up the images to be traversed.
    Array2D<double> traceImage = m_zeroImage.copy();
    Array2D<double> traceImageCopy = m_zeroImage.copy();

    // Construct the AmanatidesWoo2D instance.  Start point is inside
    // of the pixel array.  DownStreamOnly flag is true.
    AmanatidesWoo2D< Array2D<double> >
      aw2D(traceImage, m_pixelTworld, m_startXY0, m_directionXY0, true);

    // Construct a second AmanatidesWoo2D having different contents.
    AmanatidesWoo2D< Array2D<double> >
      aw2DCopy(traceImageCopy, m_pixelTworld, m_startXY2, m_directionXY2,
               false);
    
    // Copy using the assignment operator.
    aw2DCopy = aw2D;

    // Actual voxel traversal and image comparison is done in this
    // since it's identical for all of the testBegin*() functions.
    this->traverseImage(aw2DCopy);
    this->compareImages(traceImage, m_downStreamImage);
  }

  
  void
  AmanatidesWoo2DTest::
  compareImages(const Array2D<double>& testImage,
                const Array2D<double>& groundTruthImage)
  {
    // Verify result.
    DLR_TEST_ASSERT(testImage.rows() == groundTruthImage.rows());
    DLR_TEST_ASSERT(testImage.columns() == groundTruthImage.columns());
    for(size_t index = 0; index < groundTruthImage.size(); ++index) {
      double residual = std::fabs(testImage(index) - groundTruthImage(index));
      DLR_TEST_ASSERT(residual < m_testEpsilon);
    }
  }


  void
  AmanatidesWoo2DTest::
  traverseImage(AmanatidesWoo2D< Array2D<double> >& aw2D)
  {
    // Do the voxel traversal.
    typedef AmanatidesWoo2D< Array2D<double> >::iterator aw2DIterator;
    aw2DIterator endIterator = aw2D.end();
    for(aw2DIterator iter = aw2D.begin(); iter != endIterator; ++iter) {
      double parametricDistance = iter.tExit() - iter.tEntry();
      *iter = parametricDistance;
    }
  }

} // namespace dlr


#if 0

int main(int argc, char** argv)
{
  dlr::AmanatidesWoo2DTest currentTest;
  bool result = currentTest.run();
  return (result ? 0 : 1);
}

#else

namespace {

  dlr::AmanatidesWoo2DTest currentTest;

}

#endif
