/**
***************************************************************************
* @file dlrNumeric/subArray1D.h
*
* Header file declaring SubArray1D class.
*
* Copyright (C) 2001-2007 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 880 $
* $Date: 2007-05-04 00:33:49 -0400 (Fri, 04 May 2007) $
***************************************************************************
**/


#ifndef _DLR_SUBARRAY1D_H_
#define _DLR_SUBARRAY1D_H_

#include <dlrNumeric/array1D.h>
#include <dlrNumeric/slice.h>

namespace dlr {

  namespace numeric {
    
    /**
     ** Header file defining a simple SubArray class to work with Array1D.h
     ** The goal here is simplicity.  This is not intended to be a full
     ** interface.  Just enough to let you select and copy subArrays.
     ** Use this class via the function subArray(), like this:
     **
     **  subArray(array0) = subArray(array1, Slice(0, 6, 2));
     **
     ** Note that this class has deep copy semantics.
     **/
    template <class Type>
    class SubArray1D {
    public:

      /** 
       * The single argument constructs a subarray referencing every
       * element of the source array.
       * 
       * @param source This argument specifies the Array1D into which to index.
       */
      SubArray1D(const Array1D<Type>& source);

    
      /** 
       * This constructor permits slicing of the source array.  The
       * resulting subarray references only those elements of the source
       * array indicated by slice0.
       * 
       * @param source This argument specifies the Array1D into which to
       * index.
       *
       * @param slice0 This argument specifies which elements to include
       * in the subArray.
       */
      SubArray1D(const Array1D<Type>& source, const Slice& slice0);

    
      /** 
       * This is the copy constructor.  After construction, *this will
       * reference the same elements (of the same source array) as the
       * copied SubArray1D instance.
       * 
       * @param other This argument specifies the SubArray1D instance to
       * be copied
       */
      SubArray1D(const SubArray1D<Type> &other);

    
      /** 
       * Destroys the SubArray1D instance.
       */
      virtual
      ~SubArray1D() {}

    
      /** 
       * This conversion operator generates an Array1D instance from a
       * SubArray1D.  The element values from the *this are copied into
       * the newly created Array1D instance.
       * 
       * @return An Array1D instance containing copies of the elements
       * referenced by *this.
       */
      operator Array1D<Type>() const;

    
      /** 
       * This member function returns the number of elements referenced by
       * *this.  For example, if a SubArray1D references every third element
       * of an 18 element Array1D, then its size() method will return 6.
       * 
       * @return The number of elements referenced by *this.
       */
      inline size_t
      size() const {return this->m_size;}

    
      /** 
       * This member function returns the index of the first element
       * referenced by *this.  For example, if a SubArray1D references
       * every third element of an 18 element Array1D, starting from
       * element 4, then its start() method will return 4.
       * 
       * @return The index of the first element referenced by *this.
       */
      inline size_t
      start() const {return this->m_start;}


      /** 
       * This member function returns the spacing of the elements
       * referenced by *this.  For example, if a SubArray1D references
       * every third element of an Array1D, then its stride() method
       * will return 3.
       * 
       * @return The spacing of the elements referenced by *this.
       */
      inline size_t
      stride() const {return this->m_stride;}

    
      /** 
       * This method returns an Array1D instance referencing the same
       * memory as the Array1D instance from which *this was
       * constructed.
       * 
       * @return An Array1D instance which references the same memory as
       * the Array1D instance from which *this was constructed.
       */
      inline Array1D<Type>
      getArray() const {return this->m_source;}
  

      /** 
       * This assignment operator performs a deep copy, copying each
       * element from other into *this.  Note that this modifies the
       * corresponding elements of the array from which *this was
       * created.
       * 
       * @param other This argument specifies the SubArray1D instance to
       * be copied
       *
       * @return A reference to *this.
       */
      SubArray1D<Type>&
      operator=(const SubArray1D<Type>& other);

    
      /** 
       * This assignment operator sets each element reference by *this
       * to the specified value.  Note that this modifies the
       * corresponding elements of the array from which *this was
       * created.
       * 
       * @param value This argument specifies the value to which the
       * array elements should be set.
       *
       * @return A reference to *this.
       */
      SubArray1D<Type>&
      operator=(Type value);

      /** 
       * This operator increments each element of *this by the
       * corresponding element of its argument.  Note that this modifies
       * the corresponding elements of the array from which *this was
       * created.
       * 
       * @param other This argument specifies a SubArray1D instance, the
       * elements of which will be added to the elements of *this.
       *
       * @return A reference to *this.
       */
      SubArray1D<Type>&
      operator+=(const SubArray1D<Type>& other);

      /** 
       * This operator decrements each element of *this by the
       * corresponding element of its argument.  Note that this modifies
       * the corresponding elements of the array from which *this was
       * created.
       * 
       * @param other This argument specifies a SubArray1D instance, the
       * elements of which will be subtracted from the elements of *this.
       *
       * @return A reference to *this.
       */
      SubArray1D<Type>&
      operator-=(const SubArray1D<Type>& other);

    private:
      /** 
       * MS VC++ seems to have trouble compiling code which uses
       * std::abs(int), so we provide a replacement here.
       * 
       * @param argument This argument 
       * 
       * @return The return value 
       */
      inline int
      abs(int argument) {return (argument >= 0) ? argument : -argument;}

      inline void
      checkArray1DSize(const Array1D<Type>& other) const;

      inline void
      checkSubArray1DSize(const SubArray1D<Type>& other) const;

      SubArray1D<Type>&
      copy(const SubArray1D<Type>& other);

      Array1D<Type> m_source;
      int m_start;
      int m_stop;
      int m_stride;
      size_t m_size;
    };


    /* Non-member functions */

    /** 
     * This is a convenience function for constructing SubArray1D
     * instances which reference every element of the source array.  Use
     * this function when you want to modify every element of an Array1D
     * instance.
     * 
     * @param source This argument specifies the Array1D into which to index.
     *
     * @return A SubArray1D instance referencing every element of the
     * source array.
     */
    template <class Type>
    inline SubArray1D<Type>
    subArray(const Array1D<Type>& source) {return SubArray1D<Type>(source);}

  
    /** 
     * This is a convenience function for constructing SubArray1D
     * instances which reference only selected elements of the source
     * array.  Use this function when you want to modify only some of
     * the elements of an Array1D instance.  For example, you might use
     * the following to copy the first, sixth, eleventh, and sixteenth
     * elements of array1 into the second, sixth, tenth, and fourteenth
     * elements of array0:
     *
     *  subArray(array0, Slice(1, 15, 4)) = subArray(array1, Slice(0, 16, 5));
     * 
     * @param source This argument specifies the Array1D into which to
     * index.
     *
     * @param rowSlice A slice instance indicating the target elements.
     *
     * @return A SubArray1D instance referencing the selected elements
     * of the source array.
     */
    template <class Type>
    inline SubArray1D<Type>
    subArray(const Array1D<Type>& source,
             const Slice& rowSlice) {
      return SubArray1D<Type>(source, rowSlice);
    }

  
    /** 
     * This stream output operator sends a text representation of the
     * SubArray1D instance to the supplied stream instance.
     * 
     * @param stream The stream to which data should be written.
     *
     * @param subArray0 The SubArray1D instance to be written.
     *
     * @return A reference to argument stream.
     */
    template <class Type>
    std::ostream&
    operator<<(std::ostream& stream, const SubArray1D<Type>& subArray0);

  } // namespace numeric

} // namespace dlr


/* ======= Declarations to maintain compatibility with legacy code. ======= */

namespace dlr {

  using numeric::SubArray1D;
  using numeric::subArray;

} // namespace dlr


/*******************************************************************
 * Function definitions follow.  This would be a .C file
 * if SubArray1D weren't templated.
 *******************************************************************/

#include <cmath>
#include <dlrCommon/stridedPointer.h>

namespace dlr {

  namespace numeric {
    
    template <class Type>
    SubArray1D<Type>::
    SubArray1D(const Array1D<Type>& source)
      : m_source(source),
        m_start(0),
        m_stop(source.size()),
        m_stride(1),
        m_size(source.size())
    {
      // Empty
    }

    template <class Type>
    SubArray1D<Type>::
    SubArray1D(const Array1D<Type>& source, const Slice& slice0)
      : m_source(source),
        m_start(slice0.start()),
        m_stop(slice0.stop()),
        m_stride(slice0.stride()),
        m_size(0)
    {
      // It's convenient to be able to specify "last element" somehow.
      // Setting the stop value to zero will wrap to source.size()
      // if appropriate.
      if((this->m_stop == 0) && (this->m_stride > 0)) {
        this->m_stop = source.size();
      }
    
      // Negative indexing is also super-convenient
      while(this->m_start < 0) {
        this->m_start += source.size();
      }
      while(this->m_stop < 0) {
        this->m_stop += source.size();
      }

      // Now compute sizes (yuck).
      this->m_size = ((this->m_stop - this->m_start)
                      / this->m_stride); // integer division
      if(this->m_size < 0) {
        this->m_size = 0; 
      } else {
        // Can't think of a better way to do this.
        if(this->abs(this->m_size * this->m_stride)
           < this->abs(this->m_stop - this->m_start)) {
          ++this->m_size;
        }
      }

      // Make sure we won't read/write outside of the array.
      this->checkArray1DSize(source);
    }

  
    template <class Type>
    SubArray1D<Type>::
    SubArray1D(const SubArray1D<Type> &other)
      : m_source(other.m_source),
        m_start(other.m_start),
        m_stop(other.m_stop),
        m_stride(other.m_stride),
        m_size(other.m_size)
    {
      // Empty
    }

  
    template <class Type>
    SubArray1D<Type>::operator Array1D<Type>() const
    {
      Array1D<Type> returnVal(this->size());
      subArray(returnVal) = *this;
      return returnVal;
    }

  
    template <class Type>
    SubArray1D<Type>& SubArray1D<Type>::
    operator=(const SubArray1D<Type>& other)
    {
      this->checkSubArray1DSize(other);
      return this->copy(other);
    }

  
    template <class Type>
    SubArray1D<Type>& SubArray1D<Type>::
    operator=(Type value)
    {
      StridedPointer<Type> outPtr0(this->m_source.data(m_start), this->m_stride);
      StridedPointer<Type> outPtr1 = outPtr0 + this->size();
      std::fill(outPtr0, outPtr1, value);
      return *this;
    }

  
    template <class Type>
    SubArray1D<Type>& SubArray1D<Type>::
    operator+=(const SubArray1D<Type>& other)
    {
      this->checkSubArray1DSize(other);
      StridedPointer<Type> thisDataPtr0(this->m_source.data(this->m_start),
                                        this->m_stride);
      StridedPointer<Type> thisDataPtr1 = thisDataPtr0 + this->size();
      StridedPointer<const Type>
        otherDataPtr0(other.m_source.data(other.m_start), other.m_stride);
      std::transform(thisDataPtr0, thisDataPtr1, otherDataPtr0, thisDataPtr0,
                     std::plus<Type>());
      return *this;
    }

  
    template <class Type>
    SubArray1D<Type>& SubArray1D<Type>::
    operator-=(const SubArray1D<Type>& other)
    {
      this->checkSubArray1DSize(other);
      StridedPointer<Type> thisDataPtr0(this->m_source.data(this->m_start),
                                        this->m_stride);
      StridedPointer<Type> thisDataPtr1 = thisDataPtr0 + this->size();
      StridedPointer<const Type>
        otherDataPtr0(other.m_source.data(other.m_start), other.m_stride);
      std::transform(thisDataPtr0, thisDataPtr1, otherDataPtr0, thisDataPtr0,
                     std::minus<Type>());
      return *this;
    }
  
    template <class Type>
    inline void SubArray1D<Type>::
    checkArray1DSize(const Array1D<Type>& other) const
    {
#ifdef _DLRNUMERIC_CHECKBOUNDS_
      if((m_start < 0) || (m_start >= static_cast<int>(other.size()))) {
        std::ostringstream message;
        message << "Invalid start index: " << m_start << std::endl;
        DLR_THROW(IndexException, "SubArray1D::checkArray1DSize()",
                  message.str().c_str());
      }
      if(m_stride > 0) {
        if(m_stop > static_cast<int>(other.size())) {
          std::ostringstream message;
          message << "Invalid stop index: " << m_stop << std::endl;
          DLR_THROW(IndexException, "SubArray1D::checkArray1DSize()",
                    message.str().c_str());
        }
      } else if(m_stride < 0) {
        if(m_stop < -1) {
          std::ostringstream message;
          message << "Invalid stop index: " << m_stop << std::endl;
          DLR_THROW(IndexException, "SubArray1D::checkArray1DSize()",
                    message.str().c_str());
        }
      } else {
        // m_stride == 0
        DLR_THROW(IndexException, "SubArray1D::checkArray1DSize()",
                  "Invalid stride: 0");
      }
#endif
    }

  
    template <class Type>
    inline void SubArray1D<Type>::
    checkSubArray1DSize(const SubArray1D<Type>& other) const
    {
#ifdef _DLRNUMERIC_CHECKBOUNDS_
      if(other.size() != this->size()) {
        std::ostringstream message;
        message << "Size mismatch: " << other.size() << " vs. "
                << this->size() << std::endl;
        DLR_THROW(IndexException, "SubArray1D::checkSubArray1DSize()",
                  message.str().c_str());
      }
#endif
    }

  
    template <class Type>
    SubArray1D<Type>& SubArray1D<Type>::
    copy(const SubArray1D<Type>& other)
    {
      StridedPointer<const Type>
        inPtr0(other.m_source.data(other.m_start), other.stride());
      StridedPointer<const Type> inPtr1 = inPtr0 + other.size();
      StridedPointer<Type>
        outPtr0(this->m_source.data(this->m_start), this->m_stride);
      std::copy(inPtr0, inPtr1, outPtr0);
      return *this;
    }

    /* Non-member functions */

    template <class Type>
    std::ostream& operator<<(std::ostream& stream,
                             const SubArray1D<Type>& subArray)
    {
      Array1D<Type> array(subArray.size());
      subArray1D(array) = subArray;
      stream << "SubArray1D([";
      for(int index0 = 0; index < array.size() - 1; ++index0) {
        stream << array(index0) << ", ";
      }
      stream << array(array.size() - 1) << "])";
      stream.flush();
      return stream;
    }

  } // namespace numeric

} // namespace dlr

#endif // #ifdef _DLR_SUBARRAY1D_H_

