/**
***************************************************************************
* @file dlrNumeric/slice.h
*
* Header file declaring Slice class.
*
* Copyright (C) 2002-2007 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 880 $
* $Date: 2007-05-04 00:33:49 -0400 (Fri, 04 May 2007) $
***************************************************************************
**/

#ifndef _DLR_SLICE_H_
#define _DLR_SLICE_H_

namespace dlr {

  namespace numeric {
    
    /**
     ** A simple Slice class to work with SubArrays.  This is modeled
     ** after std::slice, but has "stop" instead of "size," and permits
     ** negative indexing.
     **/
    class Slice {
    public:
      /** 
       * Default constructor initializes everything to zero.  Some
       * classes interpret this as meaning "every row" or "every
       * column."
       */
      Slice() : start_(0), stop_(0), stride_(1) {}

      /** 
       * This constructor sets start and stop as specified, and defaults
       * stride to 1.
       * 
       * @param start This argument specifies the desired first element.
       * @param stop This argument specifies an element one beyond the
       * desired last element.
       */
      Slice(int start, int stop) : start_(start), stop_(stop), stride_(1) {}

      /** 
       * This constructor lets the caller specify all three values.
       * 
       * @param start This argument specifies the desired first element.
       * @param stop This argument specifies an element one beyond the
       * desired last element.
       * @param stride This argument specifies how many elements to skip
       * with each step.
       */
      Slice(int start, int stop, int stride)
        : start_(start), stop_(stop), stride_(stride) {}

      /** 
       * Accessor function for the value of start.
       * 
       * @return The index of the first element in the slice.
       */
      inline int start() const {return start_;}

      /** 
       * Accessor function for the value of stop.
       * 
       * @return The index of an element one beyond the last element in
       * the slice.
       */
      inline int stop() const {return stop_;}
    
      /** 
       * Accessor function for the value of stride.
       * 
       * @return The value of stride, where the slice accesses every
       * stride'th element from start to just before stop.
       */
      inline int stride() const {return stride_;}

    private:
      int start_;
      int stop_;
      int stride_;
    };
  
  } // namespace numeric

} // namespace dlr


/* ======= Declarations to maintain compatibility with legacy code. ======= */

namespace dlr {

  using numeric::Slice;

} // namespace dlr

#endif // #ifdef _DLR_SLICE_H_

