/**
***************************************************************************
* @file dlrComputerVision/pixelRGB.h
*
* Header file declaring the PixelRGB class template.
*
* Copyright (C) 2005-2007 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 878 $
* $Date: 2007-05-04 00:17:41 -0400 (Fri, 04 May 2007) $
***************************************************************************
*/

#ifndef _DLRCOMPUTERVISION_PIXELRGB_H_
#define _DLRCOMPUTERVISION_PIXELRGB_H_

#include <dlrCommon/types.h>

namespace dlr {

  namespace computerVision {
    
    template<class TYPE>
    struct PixelRGB
    {
      /** 
       * This constructor makes no guarantees about the color of the
       * pixel.
       */
      PixelRGB()
        : red(), green(), blue() {}

    
      /** 
       * This constructor explicitly sets the pixel value.
       * 
       * @param red This argument specifies the red value for the new pixel.
       * 
       * @param green This argument specifies the green value for the new pixel.
       * 
       * @param blue This argument specifies the blue value for the new pixel.
       */
      PixelRGB(const TYPE& red, const TYPE& green, const TYPE& blue)
        : red(red), green(green), blue(blue) {}


      /** 
       * The destructor deletes cleans up for deletion.
       */
      ~PixelRGB() {}


      /** 
       * This member function copies the pixel component values, in
       * order, from consecutive iterator targets, incrementing the
       * iterator after each copy.
       * 
       * @param iter This argument is the iterator from which to copy the
       * pixel components.
       * 
       * @return The return value is a reference to iter after the
       * copying and incrementing is done.
       */
      template<class Iter>
      inline Iter&
      copyFromIterator(Iter& iter);


      /** 
       * This member function assigns the pixel component values, in
       * order, to consecutive iterator targets, incrementing the
       * iterator after each assignment.
       * 
       * @param iter This argument is the iterator to which to copy the
       * pixel components.
       * 
       * @return The return value is a reference to iter after the
       * copying and incrementing is done.
       */
      template<class Iter>
      inline Iter&
      copyToIterator(Iter& iter);


      /* ====== Public data members ====== */

      TYPE red;
      TYPE green;
      TYPE blue;


      /* ====== Static member functions ====== */

      /** 
       * This static member function indicates whether or not a pixel
       * instance is memory identical to a contiguous array of Component
       * type.
       * 
       * @return The return value is true if the pixel structure is not
       * padded by the compiler.
       */
      static inline bool
      isContiguous();

    };


    typedef PixelRGB<UnsignedInt8> PixelRGB8;
    typedef PixelRGB<UnsignedInt16> PixelRGB16;
    typedef PixelRGB<Int16> PixelRGBSigned16;
    typedef PixelRGB<Int32> PixelRGBSigned32;
    typedef PixelRGB<Float32> PixelRGBFloat32;
    typedef PixelRGB<Float64> PixelRGBFloat64;


    /** 
     * This operator subtracts the values of the individual color
     * components of its arguments.
     * 
     * @param pixel0 The color component values of pixel1 will be
     * subtracted from the color component values of this pixel.
     * 
     * @param pixel1 The color component values of this pixel will be
     * subtracted from the color component values of pixel1.
     * 
     * @return The return value is a pixel in which each color component
     * value is the difference of the corresponding values in the two
     * input pixels.
     */
    template<class TYPE>
    inline PixelRGB<TYPE>
    operator-(const PixelRGB<TYPE>& pixel0, const PixelRGB<TYPE>& pixel1);

  
    /** 
     * This operator returns true if the contents of the two argments
     * are identical, false otherwise.
     * 
     * @param pixel0 This argument is the first pixel value to be compared.
     * 
     * @param pixel1 This argument is the second pixel value to be compared.
     * 
     * @return The return value indicates whether the two pixels have
     * identical values.
     */
    template<class TYPE>
    inline bool
    operator==(const PixelRGB<TYPE>& pixel0, const PixelRGB<TYPE>& pixel1);

  } // namespace computerVision    

} // namespace dlr
  
/* ============ Definitions of inline & template functions ============ */


namespace dlr {

  namespace computerVision {
    
    // This member function copies the pixel component values, in
    // order, from consecutive iterator targets, incrementing the
    // iterator after each copy.
    template<class TYPE>
    template<class Iter>
    inline Iter&
    PixelRGB<TYPE>::
    copyFromIterator(Iter& iter)
    {
      red = *(iter++);
      green = *(iter++);
      blue = *(iter++);
      return iter;
    }


    // This member function assigns the pixel component values, in
    // order, to consecutive iterator targets, incrementing the
    // iterator after each assignment.
    template<class TYPE>
    template<class Iter>
    inline Iter&
    PixelRGB<TYPE>::
    copyToIterator(Iter& iter)
    {
      *(iter++) = red;
      *(iter++) = green;
      *(iter++) = blue;
      return iter;
    }


    // This static member function indicates whether or not a pixel
    // instance is memory identical to a contiguous array of Component
    // type.
    template<class TYPE>
    inline bool
    PixelRGB<TYPE>::
    isContiguous()
    {
      TYPE dummy;
      TYPE* typePtr = &dummy;
      PixelRGB<TYPE>* pixelPtr = reinterpret_cast<PixelRGB<TYPE>*>(typePtr);
      PixelRGB<TYPE>* pixelPtr2 = pixelPtr + 1;
      return
        ((reinterpret_cast<TYPE*>(&(pixelPtr2->red)) == &(typePtr[3]))
         && (reinterpret_cast<TYPE*>(&(pixelPtr2->green)) == &(typePtr[4]))
         && (reinterpret_cast<TYPE*>(&(pixelPtr2->blue)) == &(typePtr[5])));
    }


    // This operator subtracts the values of the individual color
    // components of its arguments.
    template<class TYPE>
    inline PixelRGB<TYPE>
    operator-(const PixelRGB<TYPE>& pixel0, const PixelRGB<TYPE>& pixel1)
    {
      return PixelRGB<TYPE>(
        pixel0.red - pixel1.red,
        pixel0.green - pixel1.green,
        pixel0.blue - pixel1.blue);
    }


    // This operator returns true if the contents of the two argments
    // are identical, false otherwise.
    template<class TYPE>
    inline bool
    operator==(const PixelRGB<TYPE>& pixel0, const PixelRGB<TYPE>& pixel1)
    {
      return (pixel0.red == pixel1.red
              && pixel0.green == pixel1.green
              && pixel0.blue == pixel1.blue);
    }

  } // namespace computerVision
  
} // namespace dlr

#endif /* #ifndef _DLRCOMPUTERVISION_PIXELRGB_H_ */
