/**
***************************************************************************
* @file histogramEqualize.cpp
*
* Source file defining histogram equalization routines.
*
* Copyright (C) 2005 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 878 $
* $Date: 2007-05-04 00:17:41 -0400 (Fri, 04 May 2007) $
***************************************************************************
*/

#include <limits>
#include <numeric>
#include <sstream>
#include <dlrComputerVision/histogramEqualize.h>

namespace dlr {

  namespace computerVision {
    
    // This function computes the histogram of an image.
    Array1D<unsigned int>
    getHistogram(const Image<GRAY8>& inputImage)
    {
      if(inputImage.size() > std::numeric_limits<unsigned int>::max()) {
        std::ostringstream message;
        message << "Currently, we can only equalize images with "
                << std::numeric_limits<int>::max() << " or fewer pixels.";
        DLR_THROW(ValueException, "histogramEqualize()", message.str().c_str());
      }

      Array1D<unsigned int> histogram(
        std::numeric_limits<Image<GRAY8>::PixelType>::max());
      histogram = 0;
      for(size_t pixelIndex = 0; pixelIndex < inputImage.size(); ++pixelIndex) {
        ++(histogram[inputImage[pixelIndex]]);
      }
      return histogram;
    }
  
  
    // This function remaps the pixel values of the input image in such
    // a way that output pixel value increases monotonically with input
    // pixel value, and the histogram of the output image is nearly
    // flat.
    Image<GRAY8>
    histogramEqualize(const Image<GRAY8>& inputImage)
    {
      // Compute the histogram and CDF.
      Array1D<unsigned int> histogram = getHistogram(inputImage);
      Array1D<unsigned int> cdf(histogram.size());
      std::partial_sum(histogram.begin(), histogram.end(), cdf.begin(),
                       std::plus<unsigned int>());

      // Rescale the image according to the CDF.
      Image<GRAY8> outputImage(inputImage.rows(), inputImage.columns());
      double scaleFactor = 256.0 / (inputImage.size() + 1);
      for(size_t pixelIndex = 0; pixelIndex < inputImage.size(); ++pixelIndex) {
        outputImage[pixelIndex] =
          static_cast<Int8>(scaleFactor * cdf[inputImage[pixelIndex]]);
      }
      return outputImage;
    }

  } // namespace computerVision    

} // namespace dlr
