/**
***************************************************************************
* @file dateTest.cpp
* Source file defining tests for date handling.
*
* Copyright (C) 2004 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 906 $
* $Date: 2007-05-17 15:04:14 -0400 (Thu, 17 May 2007) $
***************************************************************************
**/

#include <sstream>
#include <string>

#include <dlrUtilities/date.h>
#include <dlrTest/testFixture.h>

namespace dlr {

  class DateTest : public TestFixture<DateTest> {

  public:

    DateTest();
    ~DateTest() {}

    void setUp(const std::string& testName) {}
    void tearDown(const std::string& testName) {}

    // Tests of member functions.
    void testConstructor__void();
    void testConstructor__string();
    void testConstructor__size_t__size_t__size_t();
    void testConstructor__Date();
    void testDestructor();
    void testDay();
    void testMonth();
    void testSetDay();
    void testSetMonth();
    void testSetYear();
    void testYear();

    // Tests of non-member functions.
    void testOperatorEqualTo();
    void testOperatorGreaterThan();
    void testOperatorGreaterOrEqualTo();
    void testOperatorLessThan();
    void testOperatorLessOrEqualTo();
    void testOperatorNotEqualTo();
    void testOutputOperator();
    void testInputOperator();

  }; // class DateTest


  /* ============== Member Function Definititions ============== */

  DateTest::
  DateTest()
    : TestFixture<DateTest>("DateTest")
  {
    DLR_TEST_REGISTER_MEMBER(testConstructor__void);
    DLR_TEST_REGISTER_MEMBER(testConstructor__string);
    DLR_TEST_REGISTER_MEMBER(testConstructor__size_t__size_t__size_t);
    DLR_TEST_REGISTER_MEMBER(testConstructor__Date);
    DLR_TEST_REGISTER_MEMBER(testDestructor);
    DLR_TEST_REGISTER_MEMBER(testDay);
    DLR_TEST_REGISTER_MEMBER(testMonth);
    DLR_TEST_REGISTER_MEMBER(testSetDay);
    DLR_TEST_REGISTER_MEMBER(testSetMonth);
    DLR_TEST_REGISTER_MEMBER(testSetYear);
    DLR_TEST_REGISTER_MEMBER(testYear);

    // Tests of non-member functions.
    DLR_TEST_REGISTER_MEMBER(testOperatorEqualTo);
    DLR_TEST_REGISTER_MEMBER(testOperatorGreaterThan);
    DLR_TEST_REGISTER_MEMBER(testOperatorGreaterOrEqualTo);
    DLR_TEST_REGISTER_MEMBER(testOperatorLessThan);
    DLR_TEST_REGISTER_MEMBER(testOperatorLessOrEqualTo);
    DLR_TEST_REGISTER_MEMBER(testOperatorNotEqualTo);
    DLR_TEST_REGISTER_MEMBER(testOutputOperator);
    DLR_TEST_REGISTER_MEMBER(testInputOperator);
  }


  void
  DateTest::
  testConstructor__void()
  {
    Date date0;
    DLR_TEST_ASSERT(date0.day() == 0);
    DLR_TEST_ASSERT(date0.month() == 0);
    DLR_TEST_ASSERT(date0.year() == 0);
  }

  
  void
  DateTest::
  testConstructor__string()
  {
    // Try a well-behaved input.
    Date date0("2003-1-11");
    DLR_TEST_ASSERT(date0.day() == 11);
    DLR_TEST_ASSERT(date0.month() == 1);
    DLR_TEST_ASSERT(date0.year() == 2003);

    // Try a not-so-well-behaved input, which (counting the extra
    // days), happens to be the same day.
    Date date1("2002-11-72");
    DLR_TEST_ASSERT(date1.day() == date0.day());
    DLR_TEST_ASSERT(date1.month() == date0.month());
    DLR_TEST_ASSERT(date1.year() == date0.year());
  }

  
  void
  DateTest::
  testConstructor__size_t__size_t__size_t()
  {
    // Try a well-behaved input.
    Date date0(2002, 3, 01);
    DLR_TEST_ASSERT(date0.day() == 1);
    DLR_TEST_ASSERT(date0.month() == 3);
    DLR_TEST_ASSERT(date0.year() == 2002);

    // Try a not-so-well-behaved input which refers to the same day.
    // Note that 2002 is _not_ a leap year.
    Date date1(2002, 2, 29);
    DLR_TEST_ASSERT(date1.day() == date0.day());
    DLR_TEST_ASSERT(date1.month() == date0.month());
    DLR_TEST_ASSERT(date1.year() == date0.year());

    // Make sure leap-years get noticed (2000) is a leap year.
    Date date2(2000, 2, 29);
    DLR_TEST_ASSERT(date2.day() == 29);
    DLR_TEST_ASSERT(date2.month() == 2);
    DLR_TEST_ASSERT(date2.year() == 2000);
  }

  
  void
  DateTest::
  testConstructor__Date()
  {
    Date date0("2003-1-11");
    Date date1(date0);
    DLR_TEST_ASSERT(date1.day() == date0.day());
    DLR_TEST_ASSERT(date1.month() == date0.month());
    DLR_TEST_ASSERT(date1.year() == date0.year());
  }

  
  void
  DateTest::
  testDestructor()
  {
    // No explicit test.
  }

  
  void
  DateTest::
  testDay()
  {
    // No explicit test.
  }

  
  void
  DateTest::
  testMonth()
  {
    // No explicit test.
  }

  
  void
  DateTest::
  testSetDay()
  {
    // Straightforward test.  No need to test any wacky day values,
    // since we've already verified that sanitize() is working fine.
    Date date0(2001, 1, 1);
    date0.setDay(12);
    DLR_TEST_ASSERT(date0.day() == 12);
    DLR_TEST_ASSERT(date0.month() == 1);
    DLR_TEST_ASSERT(date0.year() == 2001);
  }

  
  void
  DateTest::
  testSetMonth()
  {
    // Straightforward test.  No need to test any wacky day values,
    // since we've already verified that sanitize() is working fine.
    Date date0(2001, 1, 1);
    date0.setMonth(12);
    DLR_TEST_ASSERT(date0.day() == 1);
    DLR_TEST_ASSERT(date0.month() == 12);
    DLR_TEST_ASSERT(date0.year() == 2001);
  }

  
  void
  DateTest::
  testSetYear()
  {
    // Straightforward test.
    Date date0(2001, 1, 1);
    date0.setYear(2002);
    DLR_TEST_ASSERT(date0.day() == 1);
    DLR_TEST_ASSERT(date0.month() == 1);
    DLR_TEST_ASSERT(date0.year() == 2002);
  }

  
  void
  DateTest::
  testYear()
  {
    // No explicit test.
  }

  
    // Tests of non-member functions.
  void
  DateTest::
  testOperatorEqualTo()
  {
    Date date0(2001, 2, 2);
    Date date1(2001, 2, 2);
    Date date2(2001, 2, 3);
    Date date3(2001, 3, 2);
    Date date4(2002, 2, 2);

    DLR_TEST_ASSERT(date0 == date0);
    DLR_TEST_ASSERT(date0 == date1);
    DLR_TEST_ASSERT(!(date0 == date2));
    DLR_TEST_ASSERT(!(date0 == date3));
    DLR_TEST_ASSERT(!(date0 == date4));
    DLR_TEST_ASSERT(!(date2 == date0));
    DLR_TEST_ASSERT(!(date3 == date0));
    DLR_TEST_ASSERT(!(date4 == date0));
  }

  
  void
  DateTest::
  testOperatorGreaterThan()
  {
    Date date0(2001, 2, 2);
    Date date1(2001, 2, 2);
    Date date2(2001, 2, 3);
    Date date3(2001, 3, 2);
    Date date4(2002, 2, 2);

    DLR_TEST_ASSERT(!(date0 > date0));
    DLR_TEST_ASSERT(!(date0 > date1));
    DLR_TEST_ASSERT(!(date0 > date2));
    DLR_TEST_ASSERT(!(date0 > date3));
    DLR_TEST_ASSERT(!(date0 > date4));
    DLR_TEST_ASSERT(date2 > date0);
    DLR_TEST_ASSERT(date3 > date0);
    DLR_TEST_ASSERT(date4 > date0);
  }

  
  void
  DateTest::
  testOperatorGreaterOrEqualTo()
  {
    Date date0(2001, 2, 2);
    Date date1(2001, 2, 2);
    Date date2(2001, 2, 3);
    Date date3(2001, 3, 2);
    Date date4(2002, 2, 2);

    DLR_TEST_ASSERT(date0 >= date0);
    DLR_TEST_ASSERT(date0 >= date1);
    DLR_TEST_ASSERT(!(date0 >= date2));
    DLR_TEST_ASSERT(!(date0 >= date3));
    DLR_TEST_ASSERT(!(date0 >= date4));
    DLR_TEST_ASSERT(date2 >= date0);
    DLR_TEST_ASSERT(date3 >= date0);
    DLR_TEST_ASSERT(date4 >= date0);
  }

  
  void
  DateTest::
  testOperatorLessThan()
  {
    Date date0(2001, 2, 2);
    Date date1(2001, 2, 2);
    Date date2(2001, 2, 3);
    Date date3(2001, 3, 2);
    Date date4(2002, 2, 2);

    DLR_TEST_ASSERT(!(date0 < date0));
    DLR_TEST_ASSERT(!(date0 < date1));
    DLR_TEST_ASSERT(date0 < date2);
    DLR_TEST_ASSERT(date0 < date3);
    DLR_TEST_ASSERT(date0 < date4);
    DLR_TEST_ASSERT(!(date2 < date0));
    DLR_TEST_ASSERT(!(date3 < date0));
    DLR_TEST_ASSERT(!(date4 < date0));
  }

  
  void
  DateTest::
  testOperatorLessOrEqualTo()
  {
    Date date0(2001, 2, 2);
    Date date1(2001, 2, 2);
    Date date2(2001, 2, 3);
    Date date3(2001, 3, 2);
    Date date4(2002, 2, 2);

    DLR_TEST_ASSERT(date0 <= date0);
    DLR_TEST_ASSERT(date0 <= date1);
    DLR_TEST_ASSERT(date0 <= date2);
    DLR_TEST_ASSERT(date0 <= date3);
    DLR_TEST_ASSERT(date0 <= date4);
    DLR_TEST_ASSERT(!(date2 <= date0));
    DLR_TEST_ASSERT(!(date3 <= date0));
    DLR_TEST_ASSERT(!(date4 <= date0));
  }

  
  void
  DateTest::
  testOperatorNotEqualTo()
  {
    Date date0(2001, 2, 2);
    Date date1(2001, 2, 2);
    Date date2(2001, 2, 3);
    Date date3(2001, 3, 2);
    Date date4(2002, 2, 2);

    DLR_TEST_ASSERT(!(date0 != date0));
    DLR_TEST_ASSERT(!(date0 != date1));
    DLR_TEST_ASSERT(date0 != date2);
    DLR_TEST_ASSERT(date0 != date3);
    DLR_TEST_ASSERT(date0 != date4);
    DLR_TEST_ASSERT(date2 != date0);
    DLR_TEST_ASSERT(date3 != date0);
    DLR_TEST_ASSERT(date4 != date0);
  }

  
  void
  DateTest::
  testOutputOperator()
  {
    Date date0(2001, 2, 2);
    Date date1;

    std::ostringstream outStream;
    outStream << date0;
    std::istringstream inStream(outStream.str());
    inStream >> date1;

    DLR_TEST_ASSERT(date0 == date1);
  }

  
  void
  DateTest::
  testInputOperator()
  {
    // No explicit test.
  }

} // namespace dlr



#ifdef DLR_TEST_NO_AUTOMATIC_REGISTRATION

int main(int argc, char** argv)
{
  dlr::DateTest currentTest;
  bool result = currentTest.run();
  return (result ? 0 : 1);
}

#else /* #ifdef DLR_TEST_NO_AUTOMATIC_REGISTRATION */

namespace {

  dlr::DateTest currentTest;

}

#endif /* #ifdef DLR_TEST_NO_AUTOMATIC_REGISTRATION */
