/**
***************************************************************************
* @file dlrUtilities/pythonIO.h
*
* Header file declaring routines for importing & exporting things in
* a format that's readable & writable by the python programming language.
*
* Copyright (C) 2005-2007, David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 946 $
* $Date: 2007-07-09 20:24:50 -0400 (Mon, 09 Jul 2007) $
***************************************************************************
**/

#ifndef _DLR_PYTHONIO_H_
#define _DLR_PYTHONIO_H_

#include <iostream>
#include <string>

namespace dlr {

  namespace utilities {
    
    /** 
     * This function formats a 1D array in a way that python can parse
     * into a built-in list.  The output will look something like this:
     *
     *   [1, 2, 3, 4]
     *
     * The provided array object should provide the following member
     * functions.
     *
     * - size() const: This member function should return the total
     *   number of elements in the array.
     *
     * - operator[](size_t) const: This member function should return by
     *   value one element of the array.
     *
     * @param array0 This argument is the array to be formatted.
     * 
     * @return The return value is a string containing the
     * python-friendly text.
     */
    template<class ArrayType>
    std::string
    toPythonList(const ArrayType& array0);

  
    /**
     * @deprecated {Replaced by toPythonNumeric2DArray().}
     *
     * This function is an alias for toPythonNumeric2DArray().  It is
     * deprecated, and kept around to support legacy code.
     * 
     * @param array0 This argument is the array to be formatted.
     * 
     * @return The return value is a string containing the
     * python-friendly text.
     */
    template<class ArrayType>
    std::string
    toPythonNumericArray(const ArrayType& array0) {
      return toPythonNumeric2DArray(array0);
    }


    /** 
     * This function formats a 1D Array object in a way that python can
     * parse into a Numeric array.  The output will look something like
     * this:
     *
     *   array([1, 2, 3, 4])
     *
     * The provided array object should provide an interface compatible
     * with the function toPythonList(), which is declared above.
     * 
     * @param array0 This argument is the array to be formatted.
     * 
     * @return The return value is a string containing the
     * python-friendly text.
     */
    template<class ArrayType>
    std::string
    toPythonNumeric1DArray(const ArrayType& array0);

  
    /** 
     * This function formats an 2D Array object in a way that python can
     * parse into a Numeric array.  The output will look something like
     * this:
     *
     *   array([[1, 2], [3, 4]])
     *
     * The provided array object should provide the following member
     * functions.
     *
     * - rows() const: This member function should return the total
     *   number of rows in the array.
     *
     * - row(size_t) const: This member function should return an
     *   array-like object which addresses a specific row of the array.
     *   This object must be compatible with the function toPythonList(),
     *   which is declared above.
     * 
     * @param array0 This argument is the array to be formatted.
     * 
     * @return The return value is a string containing the
     * python-friendly text.
     */
    template<class ArrayType>
    std::string
    toPythonNumeric2DArray(const ArrayType& array0);

  } // namespace utilities
  
} // namespace dlr


/* ======= Declarations to maintain compatibility with legacy code. ======= */

namespace dlr {
    
  using utilities::toPythonList;
  using utilities::toPythonNumericArray;
  using utilities::toPythonNumeric1DArray;
  using utilities::toPythonNumeric2DArray;
  
} // namespace dlr


/* ================ Function definitions follow ================ */

namespace dlr {

  namespace utilities {
    
    template<class ArrayType>
    std::string
    toPythonList(const ArrayType& array0)
    {
      // We'll use an ostringstream to format the string.
      std::ostringstream outputStream;

      // Python string syntax starts with a square-bracket.
      outputStream << "[";

      // We'll iterate over every element in the array.
      size_t count = 0;
      while(count < array0.size()) {
        // This variable will keep track of how many entries have been
        // written to each line, so that we can politely insert carriage
        // returns.
        size_t lineCount = 0;
        // Loop just enough to complete one line.
        while((count < array0.size()) & (lineCount < 20)) {
          // If it's not the first entry, we need separating punctuation.
          if(count != 0) {
            // Usually a comma is all that's necessary.
            outputStream << ", ";

            // If we just finished a line, we'll need a carriage return and
            // an indent.
            if(lineCount == 0) {
              outputStream << "\n\t\t";
            }
          }

          // Write the next element.
          outputStream << array0[count];

          // Update the counters.
          ++count;
          ++lineCount;
        }
      }

      // Add the closing square-bracket.
      outputStream << "]";

      // And return the formatted string.
      return outputStream.str();
    }


    template<class ArrayType>
    std::string
    toPythonNumeric1DArray(const ArrayType& array0)
    {
      // We'll use an ostringstream to format the string.
      std::ostringstream outputStream;

      // Start with the appropriate syntax for a Numeric array,
      // and the the opening square-bracket of the top-level list.
      outputStream << "array(";
      outputStream << toPythonList(array0);

      // Close the open syntax.
      outputStream << ")\n";

      // And return the formatted string.
      return outputStream.str();
    }


    template<class ArrayType>
    std::string
    toPythonNumeric2DArray(const ArrayType& array0)
    {
      // We'll use an ostringstream to format the string.
      std::ostringstream outputStream;

      // Start with the appropriate syntax for a Numeric array,
      // and the the opening square-bracket of the top-level list.
      outputStream << "array([";

      // Now iterate over each row.
      for(size_t index0 = 0; index0 < array0.rows(); ++index0) {
        if(index0 != 0) {
          outputStream << ",\n\t";
        }
        outputStream << toPythonList(array0.row(index0));
      }

      // Close the open syntax.
      outputStream << "])\n";

      // And return the formatted string.
      return outputStream.str();
    }  

  } // namespace utilities

} // namespace dlr

#endif // #ifndef _DLR_IMAGEIO_H_
