/**
***************************************************************************
* @file dlrThread/broadcastQueue.h 
* Header file declaring BroadcastQueue class.
*
* Copyright (C) 2005-2007 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 706 $
* $Date: 2006-08-04 19:41:11 -0400 (Fri, 04 Aug 2006) $
***************************************************************************
**/

#ifndef _DLR_THREAD_BROADCASTQUEUE_H_
#define _DLR_THREAD_BROADCASTQUEUE_H_

#include <map>
#include <boost/thread.hpp>
#include <dlrThread/clientID.h>
#include <dlrThread/exception.h>


namespace dlr {

  namespace thread {

    /** 
     ** This class implements FIFO-style communication between one or
     ** more producer threads and multiple consumer threads, so that
     ** each consumer thread receives a copy of each item pushed into
     ** the queue.  That is, each item on the queue is "consumed" once
     ** by each consumer thread.  If you want each queue item to be
     ** consumed by exactly one consumer thread, you should use a
     ** DistributionQueue instead of a BroadcastQueue.
     **
     ** Create the BroadcastQueue instance before instantiating the
     ** producer and consumer threads, and provide each thread with a
     ** copy of, pointer to, or reference to the BroadcastQueue.
     ** Producers can stuff the FIFO by the pushFront() member
     ** function.  Consumers can extract data from the other end using
     ** the copyBack() and popBack() member functions.
     **
     ** Normally, as elements are added to the head of the queue,
     ** elements that are more than (constructor argument)
     ** maximumLength items behind the head silently vanish.  During
     ** normal usage all client threads are expected to stay within
     ** maximumLength elements of the head, and this is not an issue.
     ** If a client falls more than maximumLength elements behind the
     ** head, so that it is trying to copy elements that are already
     ** gone, then its next call to copyBack() (or lockBack()) will
     ** throw an OverflowException.
     **/
    template <class Type>
    class BroadcastQueue
    {
    public:

      /**
       ** Each thread that interacts with a particular BroadcastQueue
       ** instance must create a ClientID instance, and then register
       ** with the BroadcastQueue by calling the registerClient() member
       ** function of that BroadcastQueue instance, passing the clientID
       ** instance as an argument.  The same ClientID instance must be
       ** passed as an argument in subsequent interactions with that
       ** BroadcastQueue instance.  The ClientID instance identifies the
       ** thread to the BroadcastQueue, and simplifies internal
       ** bookkeeping.
       **/
      typedef ClientID<Type> ClientID;


      /** 
       * This constructor specifies the maximum length of the FIFO.
       * 
       * @param maximumLength This argument specifies the maximum
       * length of the FIFO.  All consumers are expected to call
       * this->popBack() frequently enough to stay within this many
       * elements of the head of the FIFO.  If consumers fail to keep
       * up, then calls to copyBack() or lockBack() may throw
       * exceptions.
       *
       * @param numberOfClients This argument can be used to specify
       * the number of threads that will interact with the BroadcastQueue
       * instance.  If this argument is not provided, the BroadcastQueue
       * will function normally, except that memory usage will likely
       * be higher, and execution time for some member function calls
       * may be increased.
       */
      BroadcastQueue(size_t maximumLength, size_t numberOfClients=2);


      /** 
       * The copy constructor does a shallow copy.  Operating on the
       * copy has exactly the same effect as operating on the
       * original.
       * 
       * @param other The BroadcastQueue instance to be copied.
       */
      BroadcastQueue(const BroadcastQueue<Type>& other);


      /** 
       * Destroys the BroadcastQueue instance and releases any resources.
       */
      virtual
      ~BroadcastQueue();


      /** 
       * The assignment operator does a shallow copy.  After assignment,
       * operating on the copy has exactly the same effect as operating
       * on the original.
       * 
       * @param other The BroadcastQueue instance to be copied.
       * @return A reference to *this.
       */
      BroadcastQueue<Type>&
      operator=(const BroadcastQueue<Type>& other);


      /** 
       * This member function moves the calling thread to the head of
       * the BroadcastQueue.  Its effect is the same as calling popBack()
       * repeatedly until all elements have been popped from the
       * queue.  After calling catchUp(), an immediate call to
       * copyBack() is likely to block waiting for a new element to be
       * pushed onto the queue by a producer thread.
       * 
       * @param clientID This argument identifies the calling thread
       * to the BroadcastQueue.  It should have been previously registered
       * with the BroadcastQueue by calling member function
       * registerClient().
       */
      void
      catchUp(ClientID& clientID);
      

      /** 
       * This member function copies the tail element of the
       * BroadcastQueue.  Note that each consumer thread has its own idea
       * of which element is the tail of the BroadcastQueue.  Repeated
       * calls to copyBack() will copy the same element unless calls
       * to popBack() are made between them.  Each consumer is
       * responsible for calling popBack() frequently so that it
       * doesn't fall too far behind in the queue.
       *
       * If the client is all caught up in the queue, either because
       * of a recent call to member function catchUp() or because it's
       * been doing a good job of calling popBack(), copyBack() will
       * block, waiting for a new element to be copied onto the queue.
       *
       * If the client has fallen more than (constructor argument)
       * maximumLength elements behind the head of the queue, then
       * copyBack() will throw an OverflowException.  For more
       * information on this, please see the documentation in the
       * comment for the BroadcastQueue class.
       *
       * @param clientID This argument identifies the calling thread
       * to the BroadcastQueue.  It should have been previously registered
       * with the BroadcastQueue by calling member function
       * registerClient().
       * 
       * @param target This reference argument is used to return a
       * copy of the tail element in the queue.
       */
      void
      copyBack(const ClientID& clientID, Type& target);


      /** 
       * This member function copies the tail element of the
       * BroadcastQueue.  It is just like void copyBack(const ClientID&,
       * Type&), with the exception that it will not block for longer
       * than the specified timeout.
       *
       * @param clientID This argument identifies the calling thread
       * to the BroadcastQueue.  It should have been previously registered
       * with the BroadcastQueue by calling member function
       * registerClient().
       * 
       * @param target This reference argument is used to return a
       * copy of the tail element in the queue.
       *
       * @param timeout This argument specifies the total amount of
       * time that copyBack is permitted to block while waiting for
       * access to the queue and/or waiting for a new element to
       * become available.  If timeout is less than or equal to zero
       * then the call will return immediately.
       *
       * @return The return value is true if an element was copied,
       * and false otherwise.
       */
      bool
      copyBack(const ClientID& clientID, Type& target, double timeout);


      /** 
       * This member function returns the length at which the
       * BroadcastQueue instance will begin dropping elements off the tail.
       * 
       * @return The return value is the maximum number of elements a
       * client thread can fall behind without causing problems.
       */
      size_t
      getMaximumLength() {return m_size;}
      

      /** 
       * This member function locks the tail element of the BroadcastQueue
       * instance so that it cannot fall off the back of the queue.
       * By never using this member function, you will avoid many
       * headaches, so please use copyBack() instead if at all
       * possible.  Member function lockBack() is provided to handle a
       * few extreme circumstances.
       *
       * Member function lockBack() will block until the lock is
       * acquired.
       *
       * An example of a circumstance in which it makes sense to use
       * lockBack() is when the broadcastQueue elements contain pointers or
       * references to data, so that copyBack() gives you a copy of a
       * pointer, but doesn't actually copy the data.  Presumably, the
       * producer threads are smart enough not to de-allocate the data
       * until the element drops of the end of the queue, so you can
       * be confident that you're getting a good pointer at the time
       * of the copyBack() call, but there's nothing to stop the
       * element from falling off the queue (and the associated data
       * from being de-allocated) after the call, when the consumer is
       * still trying to access it.  In this case, it makes sense to
       * call lockBack() in the following sequence.
       *
       *   myBroadcastQueue.lockBack(myClientID);
       *
       *   myBroadcastQueue.copyBack(myClientID, myLocalElement);
       *
       *   // Code here does read-only access to the referenced data.
       *
       *   myBroadcastQueue.unlockBack(myClientID);
       *
       *   myBroadcastQueue.popBack(myClientID);
       *
       * Member function copyBack() is smart enough not to deadlock
       * when copying an already-locked element.
       *
       * Note that the example above calls for read-only access to the
       * allocated data.  This is because changing the values of
       * elements on the queue (or of data that is pointed to by
       * elements that are on the queue) makes the interactions
       * between threads much more difficult to think about and debug.
       *
       * @param clientID This argument identifies the calling thread
       * to the BroadcastQueue.  It should have been previously registered
       * with the BroadcastQueue by calling member function
       * registerClient().
       */
      void
      lockBack(const ClientID& clientID);


      /** 
       * This member function locks the tail element of the BroadcastQueue
       * instance so that it cannot fall off the back of the queue.
       * It is just like void lockBack(const ClientID&), with the
       * exception that it will not block for longer than the
       * specified timeout.
       *
       * @param clientID This argument identifies the calling thread
       * to the BroadcastQueue.  It should have been previously registered
       * with the BroadcastQueue by calling member function
       * registerClient().
       * 
       * @param timeout This argument controls how long the call to
       * lockBack() will block waiting for the lock to become
       * available.  If timeout is less than or equal to zero, then
       * the call to will return immediately.
       *
       * @return The return value is true if a lock was established,
       * and false otherwise.
       */
      bool
      lockBack(const ClientID& clientID, double timeout);
      

      /** 
       * This member function is called by consumer threads to
       * "discard" one element from the tail of the BroadcastQueue so that
       * subsequent calls to copyBack() will copy the next element in
       * the queue.  Note that each consumer thread has its own idea
       * of which element is the tail of the BroadcastQueue.  Each consumer
       * is responsible for calling popBack() frequently so that it
       * doesn't fall too far behind in the queue.
       * 
       * @param clientID This argument identifies the calling thread
       * to the BroadcastQueue.  It should have been previously registered
       * with the BroadcastQueue by calling member function
       * registerClient().
       */
      inline void
      popBack(ClientID& clientID);
      

      /** 
       * This member function is called by producer threads to add
       * elements to the head of the queue.  When the element is added
       * to the queue, it is copied, not referenced.  Note that there
       * is only one head of the queue, where as there are as many
       * tails as there are consumer threads.  Calls to
       * pushFront() advance the head for all producer threads.
       * Calls to popBack() advance the tail only for the calling
       * thread.
       * 
       * @param clientID This argument identifies the calling thread
       * to the BroadcastQueue.  It should have been previously registered
       * with the BroadcastQueue by calling member function
       * registerClient().
       * 
       * @param element This argument is the element to be copied onto
       * the head of the BroadcastQueue instance.
       */
      void
      pushFront(const ClientID& clientID, const Type& element);


      /** 
       * This member function introduces a thread to a BroadcastQueue
       * instance.  The thread should create a ClientID instance, and
       * then call registerClient() exactly once, passing the ClientID
       * instance as the first argument.  This same ClientID instance
       * should be used by the calling thread in subsequent
       * interactions with the BroadcastQueue instance.  Using the same
       * ClientID instance to interact with another BroadcastQueue instance
       * (besides *this) is bad, and will have undefined results.
       *
       * After a call to registerClient(), calling copyBack() will
       * copy the oldest element of the queue: the very first element
       * to have been pushed onto the queue.  If the BroadcastQueue
       * instance has been in use for a while, it is likely that this
       * element has already dropped of the end of the queue as
       * described in the comment for the BroadcastQueue class, leading to
       * an OverflowException when copyBack() is called.  To avoid
       * this problem, all consumer threads should register with the
       * BroadcastQueue instance before the producers start filling the
       * queue.  Alternatively, a consumer that registers late can
       * call catchUp() to move to the head of the queue.
       *
       * Note that the ClientID class is currently not copyable.  This
       * means the client thread should create the ClientID in an
       * outer scope and then pass it by reference into any inner
       * scopes that need access to the BroadcastQueue.
       *
       * @param clientID This argument is the clientID that will be
       * used to identify the calling thread to the BroadcastQueue in
       * future interactions.
       *
       * @param clientName This argument specifies a name for the
       * thread.  It is useful to specify a unique name for each
       * thread that interacts with a BroadcastQueue instance because this
       * will make it easier to debug any exceptions thrown by the
       * BroadcastQueue member functions.
       */
      void
      registerClient(ClientID& clientID,
                     const std::string& clientName="anonymous");


      /** 
       * This member function blocks until all clients of the
       * BroadcastQueue have called it, then moves the client to the head
       * of the queue.  That is, the first thread to call synchronize
       * will block, as will the second, etc. until each of the
       * registered threads has called synchronize().  When the final
       * thread calls synchronize(), all of the calling threads will
       * be unblocked and execution will continue.  Prior to the
       * return from synchronize(), each thread will be moved to the
       * head of the queue as described for member function catchUp().
       *
       * Once the final call to synchronize() has been completed and
       * all threads have been un-blocked, the internal count of
       * clients that have called synchronize() is reset.  Subsequent
       * calls to synchronize() will block again as if they were the
       * first, second, etc. until all of the registered clients have
       * called synchronize() a second time.  This cycle will repeat
       * indefinitely.
       * 
       * Calling synchronize() while another thread is blocking in
       * synchronizeQuorum() will throw a StateException.
       *
       * @param clientID This argument identifies the calling thread
       * to the BroadcastQueue.  It should have been previously registered
       * with the BroadcastQueue by calling member function
       * registerClient().
       */
      void
      synchronize(ClientID& clientID);


      /** 
       * This member function blocks until all clients of the
       * BroadcastQueue have called it, then moves the client to the
       * head of the queue.  It is just like synchronize(const
       * ClientID&), with the exception that it will not block for
       * longer than the specified timeout.
       *
       * Calling synchronize() while another thread is blocking in
       * synchronizeQuorum() will throw a StateException.
       *
       * @param clientID This argument identifies the calling thread
       * to the BroadcastQueue.  It should have been previously registered
       * with the BroadcastQueue by calling member function
       * registerClient().
       * 
       * @param timeout This argument controls how long the call to
       * synchronize() will block waiting for other threads.  If the
       * timeout expires before all of the other threads have called
       * synchronize(), then the return value will be false, and
       * execution will continue as if the calling thread had not
       * called synchronize().
       *
       * @return The return value is true if all threads were
       * synchronized, and false if the timeout expired.
       */
      bool
      synchronize(ClientID& clientID, double timeout);


      /** 
       * This member function blocks until the specified number of
       * BroadcastQueue clients have called it, then moves the client to
       * the head of the queue.  That is, the first thread to call
       * synchronize will block, as will the second, etc. until each
       * the specified number of threads have called
       * synchronizeQuorum(), at which time all of the blocking
       * threads will be unblocked.  Prior to the return from
       * synchronizeQuorum(), each thread will be moved to the head of
       * the queue as described for member function catchUp().
       * 
       * Calling synchronizeQuorum() while another thread is blocking
       * in synchronize() will throw a StateException.  Calling
       * synchronizeQuorum() with a different value for argument
       * quorumSize than a thread which is currently blocking in
       * synchronizeQuorum() will also throw a StateException.
       *
       * If synchronizeQuorum() is called with a quorumSize of zero,
       * then the quorum size will be set to the number of clients
       * currently registered with the BroadcastQueue instance.  Note
       * that this has a different effect than just using the
       * synchronize() methods: synchronize() methods will block
       * waiting for new threads to call synchronize() even if those
       * threads were added after the initial call to synchronize();
       * synchronizeQuorum() will wait only for the number of threads
       * that were registered at the time of the synchronizeQuorum()
       * call.  Be careful, though: if a thread calls
       * synchronizeQuorum() with quorumSize set to zero, a new thread
       * is registered with the BroadcastQueue, and then a second
       * thread calls synchronizeQuorum() with quorumSize set to zero,
       * the second call to synchronizeQuorum() will be trying to set
       * a different quorum size than the first, and will throw a
       * StateException as described in the previous paragraph.
       *
       * @param clientID This argument identifies the calling thread
       * to the BroadcastQueue.  It should have been previously registered
       * with the BroadcastQueue by calling member function
       * registerClient().
       *
       * @param quorumSize If this argument is nonzero, then the
       * function call will block only until the specified number of
       * clients have called synchronizeQuorum().  If quorumSize is
       * zero, it will be as if it were set to the number of clients
       * currently registered with the BroadcastQueue.
       */
      void
      synchronizeQuorum(ClientID& clientID, size_t quorumSize);


      /** 
       * This member function blocks until the specified number of
       * BroadcastQueue clients have called it, then moves the client to
       * the head of the queue.  It is just like
       * synchronizeQuorum(const ClientID&), with the exception that
       * it will not block for longer than the specified timeout.
       *
       * Calling synchronizeQuorum() while another thread is blocking
       * in synchronize() will throw a StateException.  Calling
       * synchronizeQuorum() with a different value for argument
       * quorumSize than a thread which is currently blocking in
       * synchronizeQuorum() will also throw a StateException.
       *
       * @param clientID This argument identifies the calling thread
       * to the BroadcastQueue.  It should have been previously registered
       * with the BroadcastQueue by calling member function
       * registerClient().
       * 
       * @param quorumSize If this argument is nonzero, then the
       * function call will block only until the specified number of
       * clients have called synchronizeQuorum().  If quorumSize is
       * zero, it will be as if it were set to the number of clients
       * currently registered with the BroadcastQueue.
       *
       * @param timeout This argument controls how long the call to
       * synchronizeQuorum() will block waiting for other threads.  If the
       * timeout expires before all of the other threads have called
       * synchronizeQuorum(), then the return value will be false, and
       * execution will continue as if the calling thread had not
       * called synchronizeQuorum().
       *
       * @return The return value is true if the specified number of
       * threads was synchronized, and false if the timeout expired.
       */
      bool
      synchronizeQuorum(ClientID& clientID, size_t quorumSize, double timeout);
      
      
      /** 
       * This member function releases a read lock acquired by
       * lockBack().
       * 
       * @param clientID This argument identifies the calling thread
       * to the BroadcastQueue.  It should have been previously registered
       * with the BroadcastQueue by calling member function
       * registerClient().
       */
      void
      unlockBack(const ClientID& clientID);
      
    private:

      /** 
       * This private member function is called by the copy
       * constructor and assignment operator to copy another
       * BroadcastQueue instance in a thread-safe way.
       * 
       * @param other This argument is the BroadcastQueue instance to
       * copy.
       */
      void
      copyOther(const BroadcastQueue<Type>& other);

      
      /** 
       * This private member function should only be called when
       * m_headIndexMutexPtr is locked.  It updates the BroadcastQueue
       * state to indicate that a client thread is read-locking its
       * particular tail element.
       * 
       * @param clientID This argument indicates which thread wants
       * the lock.
       * 
       * @param overflowFlag This reference argument should be set to
       * false at the time of the call.  If the client has passed the
       * head of the queue, it will be reset to true, indicating an
       * error.
       * 
       * @param underflowFlag This reference argument should be set to
       * false at the time of the call.  If the client has fallen off
       * the end of the queue, it will be set to true, indicating an
       * error.
       * 
       * @param headElementNumber If either overflowFlag or
       * underflowFlag is set to true after the call, then this
       * argument be set to the element number of the head of the
       * queue, otherwise it will not be touched.
       */
      inline void
      declareReadLock(const ClientID& clientID,
                      bool& overflowFlag,
                      size_t& headElementNumber);
      

      /** 
       * This private member function does the actual mechanics of
       * synchronizing threads.  It is called by member functios
       * synchronize() and synchronizeQuorum().
       * 
       * @param clientID This argument indicates which thread wants
       * the lock.
       * 
       * @param targetNumber This argument specifies how many threads
       * are required to complete the synchronization.
       *
       * @param syncLock This argument is a lock which must have been
       * acquired from *(this->m_syncMutexPtr).
       */
      void 
      handleSynchronize(ClientID& clientID,
                        size_t targetNumber,
                        boost::timed_mutex::scoped_lock& syncLock);
      

      /** 
       * This private member function does the actual mechanics of
       * synchronizing threads.  It is called by member functios
       * synchronize() and synchronizeQuorum().
       * 
       * @param clientID This argument indicates which thread wants
       * the lock.
       * 
       * @param targetNumber This argument specifies how many threads
       * are required to complete the synchronization.
       *
       * @param xtimeout This argument indicates at what time to give
       * up and return false.
       *
       * @param syncLock This argument is a lock which must have been
       * acquired from *(this->m_syncMutexPtr).
       *
       * @return The return value is true if all operations were
       * completed within the specified timeframe, false otherwise.
       * If the function returns false, it will restore the state of
       * the BroadcastQueue prior to its return so that it is as if
       * handleSynchronize() had not been called in the first place.
       */
      bool
      handleSynchronize(ClientID& clientID,
                        size_t targetNumber,
                        boost::xtime xtimeout,
                        boost::timed_mutex::scoped_timed_lock& syncLock);


      /** 
       * This private member function is called by the destructor
       * and assignment operator to decrement reference counts, etc.
       */
      void
      releaseResources();
      
      
      size_t* m_countPtr;
      size_t* m_headIndexPtr;
      size_t m_size;
      
      std::map<size_t, std::string>* m_clientNameMapPtr;
      Type* m_dataArray;
      size_t* m_maxNumberOfClientsPtr;
      size_t* m_numberOfClientsPtr;
      size_t* m_readLockCountArray;
      size_t** m_readLockingFlagsArray;
      size_t* m_syncClientCountPtr;
      size_t* m_syncElementCountPtr;
      size_t* m_syncIndexPtr;
      size_t* m_quorumSizePtr;

      boost::condition* m_headChangedConditionPtr;
      boost::timed_mutex* m_headIndexMutexPtr;
      // // This code left around in case we try to go back to
      // // individual mutexes for data elements.
      // boost::timed_mutex* m_mutexArray;
      boost::timed_mutex* m_numberOfClientsMutexPtr;

      boost::condition* m_syncConditionPtr;
      boost::timed_mutex* m_syncMutexPtr;

      // We don't use the pre-written ReferenceCount class becuase
      // it's not thread-safe.
      size_t* m_referenceCountPtr;
    };

  } // namespace thread

} // namespace dlr


/* =============       Implementation follows        ============= */

#include <algorithm>
#include <iostream>
#include <sstream>
#include <dlrThread/private.h>


namespace dlr {

  namespace thread {

    /* ===========    Implementation of BroadcastQueue      =========== */

    template <class Type>
    BroadcastQueue<Type>::
    BroadcastQueue(size_t maximumLength, size_t numberOfClients)
      : m_countPtr(new size_t),
        m_headIndexPtr(new size_t),
        m_size(maximumLength),
        m_clientNameMapPtr(new std::map<size_t, std::string>),
        m_dataArray(new Type[maximumLength]),
        m_maxNumberOfClientsPtr(new size_t),
        m_numberOfClientsPtr(new size_t),
        m_readLockCountArray(new size_t[maximumLength]),
        m_readLockingFlagsArray(new size_t*[maximumLength]),
        m_syncClientCountPtr(new size_t),
        m_syncElementCountPtr(new size_t),
        m_syncIndexPtr(new size_t),
        m_quorumSizePtr(new size_t),
        m_headChangedConditionPtr(new boost::condition),
        m_headIndexMutexPtr(new boost::timed_mutex),
        // // This code left around in case we try to go back to
        // // individual mutexes for data elements.
        // m_mutexArray(new boost::timed_mutex[maximumLength]),
        m_numberOfClientsMutexPtr(new boost::timed_mutex),
        m_syncConditionPtr(new boost::condition),
        m_syncMutexPtr(new boost::timed_mutex),
        m_referenceCountPtr(new size_t)
    {
      if(numberOfClients == 0) {
        DLR_THROW(ValueException, "BroadcastQueue::BroadcastQueue()",
                  "Argument numberOfClients must be greater than zero.");
      }

      // No elements have been added yet.
      *m_countPtr = 0;
      *m_headIndexPtr = 0;

      // Set client capacity as instructed.
      *m_maxNumberOfClientsPtr = numberOfClients;

      // But indicate that no clients have registered yet.
      *m_numberOfClientsPtr = 0;

      // For each element in the queue.
      for(size_t index0 = 0; index0 < m_size; ++index0) {
        // Allocate storage for the expected number of clients.
        m_readLockingFlagsArray[index0] =
          new size_t[*m_maxNumberOfClientsPtr];
        size_t* beginIterator = m_readLockingFlagsArray[index0];
        size_t* endIterator = beginIterator + (*m_maxNumberOfClientsPtr);
        std::fill(beginIterator, endIterator, 0);

        // Indicate that no client is locking this element.
        m_readLockCountArray[index0] = 0;
      }

      // Initially, no clients are waiting for sync events.
      *m_syncClientCountPtr = 0;
      *m_syncElementCountPtr = 0;
      *m_syncIndexPtr = 0;
      *m_quorumSizePtr = 0;
      *m_referenceCountPtr = 1;
    }


    // The copy constructor does a shallow copy.
    template <class Type>
    BroadcastQueue<Type>::
    BroadcastQueue(const BroadcastQueue<Type>& other)
      : m_countPtr(0),
        m_headIndexPtr(0),
        m_size(0),
        m_clientNameMapPtr(0),
        m_dataArray(0),
        m_maxNumberOfClientsPtr(0),
        m_numberOfClientsPtr(0),
        m_readLockCountArray(0),
        m_readLockingFlagsArray(0),
        m_syncClientCountPtr(0),
        m_syncElementCountPtr(0),
        m_syncIndexPtr(0),
        m_quorumSizePtr(0),
        m_headChangedConditionPtr(0),
        m_headIndexMutexPtr(0),
        m_numberOfClientsMutexPtr(0),
        m_syncConditionPtr(0),
        m_syncMutexPtr(0),
        m_referenceCountPtr(0)
    {
      this->copyOther(other);
    }

    
    // Destroys the BroadcastQueue instance and releases any resources.
    template <class Type>
    BroadcastQueue<Type>::
    ~BroadcastQueue()
    {
      this->releaseResources();
    }


    // The assignment operator does a shallow copy.
    template <class Type>
    BroadcastQueue<Type>&
    BroadcastQueue<Type>::
    operator=(const BroadcastQueue<Type>& other)
    {
      if(&other != this) {
	this->releaseResources();
	this->copyOther(other);
      }
    }      


    // This member function moves the calling thread to the head of
    // the BroadcastQueue.
    template <class Type>
    void
    BroadcastQueue<Type>::
    catchUp(ClientID& clientID)
    {
      boost::timed_mutex::scoped_lock headLock(*m_headIndexMutexPtr);
      clientID.m_count = *m_countPtr;
      clientID.m_index = *m_headIndexPtr;
    }

    
    // This member function copies the tail element of the
    // BroadcastQueue.
    template <class Type>
    void
    BroadcastQueue<Type>::
    copyBack(const ClientID& clientID, Type& target)
    {
      this->lockBack(clientID);

      try {
        target = m_dataArray[clientID.m_index];
      } catch(...) {
        this->unlockBack(clientID);
        throw;
      }

      this->unlockBack(clientID);
    }


    // This member function copies the tail element of the
    // BroadcastQueue.
    template <class Type>
    bool
    BroadcastQueue<Type>::
    copyBack(const ClientID& clientID, Type& target, double timeout)
    {
      if(this->lockBack(clientID, timeout) == false) {
        return false;
      }

      try {
        target = m_dataArray[clientID.m_index];
      } catch(...) {
        this->unlockBack(clientID);
        throw;
      }

      this->unlockBack(clientID);
      return true;
    }


    // This member function locks the tail element of the BroadcastQueue
    // instance so that it cannot fall off the back of the queue.
    template <class Type>
    void
    BroadcastQueue<Type>::
    lockBack(const ClientID& clientID)
    {
      // These variables let us avoid time-consuming formatting of
      // error messages during the time that we have
      // *m_headIndexMutexPtr locked.
      bool overflowFlag = false;
      size_t headElementNumber = 0;
      
      // Get exclusive control of the broadcastQueue, check that we're not
      // too far ahead or behind, and mark our current element as
      // being read.  The scoped_lock instance will be released when
      // it goes out of scope.
      {
        boost::timed_mutex::scoped_lock headLock(*m_headIndexMutexPtr);

        // // This code left around in case we try to go back to
        // // individual mutexes for data elements.
        // boost::timed_mutex::scoped_lock elementLock(
        //   m_mutexArray[clientID.m_index]);
        
        // Make sure there's data available to copy.  That is, make
        // sure we're not ahead of the queue, and wait for a producer
        // to add an element if we are.
        while(clientID.m_count >= (*m_countPtr)) {
          m_headChangedConditionPtr->wait(headLock);
        }

        // Mark the appropriate element as being read by the calling
        // thread.
        this->declareReadLock(clientID, overflowFlag, headElementNumber);
        
      } // Release the lock.

      if(overflowFlag) {
        std::ostringstream message;
        message << "Thread "
                << (m_clientNameMapPtr->find(clientID.m_idNumber))->second
                << " "
                << "has fallen irretrievably far behind the front of the "
                << "queue.  Accessed element is number "
                << clientID.m_count << ", head element is number "
                << headElementNumber << ".";
        DLR_THROW(OverflowException, "BroadcastQueue::lockBack()",
                  message.str().c_str());
      }
    }


    // This member function locks the tail element of the BroadcastQueue
    // instance so that it cannot fall off the back of the queue.
    template <class Type>
    bool
    BroadcastQueue<Type>::
    lockBack(const ClientID& clientID, double timeout)
    {
      // Argument checking.
      if(timeout < 0.0) {
        timeout = 0.0;
      }
      boost::xtime xtimeout;
      if(getXTime(timeout, xtimeout) == false) {
        // Note(xxx): should we throw an exception here?
        return false;
      }
      
      // These variables let us avoid time-consuming formatting of
      // error messages during the time that we have
      // *m_headIndexMutexPtr locked.
      bool overflowFlag = false;
      size_t headElementNumber = 0;

      // Get exclusive control of the broadcastQueue, check that we're not
      // too far ahead or behind, and mark our current element as
      // being read.  The scoped_lock instance will be released when
      // it goes out of scope.
      try {
        // This call throws a boost::lock_error if the timeout expires.
        boost::timed_mutex::scoped_timed_lock headLock(
          *m_headIndexMutexPtr, xtimeout);

        // Make sure there's data available to copy.  That is, make
        // sure we're not ahead of the queue, and wait for a producer
        // to add an element if we are.
        while(clientID.m_count >= (*m_countPtr)) {
          if(m_headChangedConditionPtr->timed_wait(headLock, xtimeout)
             == false) {
            // No new data before timeout expires.
            return false;
          }
        }

        // Mark the appropriate element as being read by the calling
        // thread.
        this->declareReadLock(clientID, overflowFlag, headElementNumber);
        
      } catch(const boost::lock_error&) {
        // The timeout expired before we could get a lock.
        return false;
      }

      if(overflowFlag) {
        std::ostringstream message;
        message << "Thread "
                << (m_clientNameMapPtr->find(clientID.m_idNumber))->second
                << " "
                << "has fallen irretrievably far behind the front of the "
                << "queue.  Accessed element is number "
                << clientID.m_count << ", head element is number "
                << headElementNumber << ".";
        DLR_THROW(OverflowException, "BroadcastQueue::lockBack()",
                  message.str().c_str());
      }
      return true;
    }


    template <class Type>
    inline void
    BroadcastQueue<Type>::
    popBack(ClientID& clientID)
    {
      ++clientID.m_count;
      ++clientID.m_index;
      if(clientID.m_index >= m_size) {
        clientID.m_index = 0;
      }
      if(clientID.m_count == std::numeric_limits<size_t>::max()) {
        std::ostringstream message;
        message << "Thread "
                << (m_clientNameMapPtr->find(clientID.m_idNumber))->second
                << " "
                << "has exceeded the maximum allowable queue position of "
                << std::numeric_limits<size_t>::max() - 1 << ".";
        DLR_THROW(LogicException, "BroadcastQueue::popBack()",
                  message.str().c_str());
      }
    }


    template <class Type>
    void
    BroadcastQueue<Type>::
    pushFront(const ClientID& clientID, const Type& element)
    {
      // These variables let us avoid time-consuming formatting of
      // error messages during the time that we have
      // *m_headIndexMutexPtr locked.
      bool readLockedFlag = false;
      bool countOverflowFlag = false;
      size_t threadIDNumber = 0;

      // Get exclusive control of the BroadcastQueue, check that noone's
      // read-locked an element that's so far behind the head that the
      // head has wrapped around and is about to stomp on the locked
      // data, copy the new element into it, and advance the head.
      {
        // This lock will be automatically released when it goes out
        // of scope.
        boost::timed_mutex::scoped_lock headLock(*m_headIndexMutexPtr);

        if(m_readLockCountArray[*m_headIndexPtr] != 0) {
          // Ack! This element is read-locked.  Figure out which
          // thread is the culprit.
          readLockedFlag = true;
          size_t* flagsPtr = m_readLockingFlagsArray[*m_headIndexPtr];
          while(threadIDNumber < (*m_numberOfClientsPtr)) {
            if(flagsPtr[threadIDNumber] != 0) {
              break;
            }
            ++threadIDNumber;
          }
        } else {
          // Not read-locked, proceed with the copying, etc.
          m_dataArray[*m_headIndexPtr] = element;
          ++(*m_headIndexPtr);
          ++(*m_countPtr);
          if((*m_headIndexPtr) >= m_size) {
            (*m_headIndexPtr) = 0;
          }
          if(*m_countPtr == std::numeric_limits<size_t>::max()) {
            countOverflowFlag = true;
          }

          // Wake up any threads that are waiting for data.
          m_headChangedConditionPtr->notify_all();
        }
      }

      // Report any errors.
      if(readLockedFlag) {
        // Sanity check.
        if(threadIDNumber == *m_numberOfClientsPtr) {
          DLR_THROW(LogicException, "BroadcastQueue::pushFront()",
                    "Read lock accounting is in an inconsistent state.");
        }
        std::ostringstream message;
        message << "Thread "
                << (m_clientNameMapPtr->find(clientID.m_idNumber))->second
                << " "
                << "can't advance the queue because thread "
                << (m_clientNameMapPtr->find(threadIDNumber))->second
                << " "
                << "is still read-locking the tail element.";
        DLR_THROW(OverflowException, "BroadcastQueue::pushFront()",
                  message.str().c_str());
      }
      if(countOverflowFlag) {
        std::ostringstream message;
        message << "Queue size exceeds the allowable maximum of "
                << std::numeric_limits<size_t>::max() - 1 << " "
                << "on a function call from thread " 
                << (m_clientNameMapPtr->find(clientID.m_idNumber))->second
                << ".";
        DLR_THROW(OverflowException, "BroadcastQueue::pushFront()",
                  message.str().c_str());
      }
    }


    // This member function returns a ClientID instance that should
    // be used by the calling thread in all subsequent interactions
    // with the BroadcastQueue instance.
    template <class Type>
    void 
    BroadcastQueue<Type>::
    registerClient(ClientID& clientID, const std::string& clientName)
    {
      // Registration must be strictly serialized.  Only one thread
      // can register at once.  Also, producer threads must be locked
      // out so they don't throw an exception when they can't get
      // locks on m_mutexArray elements.
      boost::timed_mutex::scoped_lock headLock(*m_headIndexMutexPtr);

      // If adding another client will overflow our bookkeeping space,
      // then allocate a bigger bookkeeping space and copy the old
      // data into it.
      if(*m_numberOfClientsPtr >= (*m_maxNumberOfClientsPtr)) {
        size_t newMaxNumberOfClients = (*m_maxNumberOfClientsPtr) << 1;
        for(size_t index0 = 0; index0 < m_size; ++index0) {
          {
            // // This code left around in case we try to go back to
            // // individual mutexes for data elements.
            // boost::timed_mutex::scoped_lock elementLock(
            //   m_mutexArray[index0]);
          
            size_t* newFlagsArray = new size_t[newMaxNumberOfClients];
            size_t* beginIterator = m_readLockingFlagsArray[index0];
            size_t* endIterator = beginIterator + (*m_maxNumberOfClientsPtr);
            std::copy(beginIterator, endIterator, newFlagsArray);
            
            beginIterator = newFlagsArray;
            endIterator = beginIterator + newMaxNumberOfClients;
            beginIterator += (*m_maxNumberOfClientsPtr);
            std::fill(beginIterator, endIterator, 0);
            
            delete[] m_readLockingFlagsArray[index0];
            m_readLockingFlagsArray[index0] = newFlagsArray;
          }
        }
        *m_maxNumberOfClientsPtr = newMaxNumberOfClients;
      }
      m_clientNameMapPtr->insert(
        std::make_pair(*m_numberOfClientsPtr, clientName));
      clientID.m_count = 0;
      clientID.m_idNumber = (*m_numberOfClientsPtr)++;
      clientID.m_index = 0;
    }
    

    // This member function blocks until all clients of the
    // BroadcastQueue have called it, then moves the client to the head
    // of the queue.
    template <class Type>
    void 
    BroadcastQueue<Type>::
    synchronize(ClientID& clientID)
    {
      boost::timed_mutex::scoped_lock syncLock(*m_syncMutexPtr);
      
      if((*m_syncClientCountPtr) != 0) {
        if((*m_quorumSizePtr) != 0) {
          std::ostringstream message;
          message << "Can't synchronize with all registered clients "
                  << "while another thread is waiting for a quorum of size "
                  << (*m_quorumSizePtr) << "."; 
          DLR_THROW(StateException, "BroadcastQueue::synchronize()",
                    message.str().c_str());
        }
      }
      (*m_quorumSizePtr) = 0;

      this->handleSynchronize(clientID, (*m_numberOfClientsPtr), syncLock);
    }


    // This member function blocks until all clients of the
    // BroadcastQueue have called it, then moves the client to the head
    // of the queue.
    template <class Type>
    bool
    BroadcastQueue<Type>::
    synchronize(ClientID& clientID, double timeout)
    {
      // Argument checking.
      if(timeout < 0.0) {
        timeout = 0.0;
      }
      boost::xtime xtimeout;
      if(getXTime(timeout, xtimeout) == false) {
        // Note(xxx): should we throw an exception here?
        return false;
      }

      bool returnValue = false;
      try {
        // This call throws a boost::lock_error if the timeout expires.
        boost::timed_mutex::scoped_timed_lock syncLock(
          *m_syncMutexPtr, xtimeout);

        // State checking.
        if((*m_syncClientCountPtr) != 0) {
          if((*m_quorumSizePtr) != 0) {
            std::ostringstream message;
            message << "Can't synchronize with all registered clients "
                    << "while another thread is waiting for a quorum of size "
                    << (*m_quorumSizePtr) << "."; 
            DLR_THROW(StateException, "BroadcastQueue::synchronize()",
                      message.str().c_str());
          }
        }
        (*m_quorumSizePtr) = 0;
        
        returnValue =
          this->handleSynchronize(
            clientID, (*m_numberOfClientsPtr), xtimeout, syncLock);
      } catch(const boost::lock_error&) {
        // The timeout expired before we could get a lock.
      }
      return returnValue;
    }


    // This member function blocks until the specified number of
    // BroadcastQueue clients have called it, then moves the client to
    // the head of the queue.
    template <class Type>
    void
    BroadcastQueue<Type>::
    synchronizeQuorum(ClientID& clientID, size_t quorumSize)
    {
      boost::timed_mutex::scoped_lock syncLock(*m_syncMutexPtr);

      if(quorumSize == 0) {
        quorumSize = (*m_numberOfClientsPtr);
      }
      
      if((*m_syncClientCountPtr) != 0) {
        if((*m_quorumSizePtr) == 0) {
          DLR_THROW(StateException, "BroadcastQueue::synchronizeQuorum()",
                    "Can't start a quorum while another thread is blocking "
                    "in member function synchronize()");         
        } else if((*m_quorumSizePtr) != quorumSize) {
          std::ostringstream message;
          message << "Can't start a quorum of size " << quorumSize << " "
                  << "while another thread is waiting for a quorum of size "
                  << (*m_quorumSizePtr) << ".";
          DLR_THROW(StateException, "BroadcastQueue::synchronizeQuorum()",
                    message.str().c_str());
        }
      }
      (*m_quorumSizePtr) = quorumSize;

      this->handleSynchronize(clientID, (*m_quorumSizePtr), syncLock);
    }


    // This member function blocks until the specified number of
    // BroadcastQueue clients have called it, then moves the client to
    // the head of the queue.
    template <class Type>
    bool
    BroadcastQueue<Type>::
    synchronizeQuorum(ClientID& clientID, size_t quorumSize, double timeout)
    {
      // Argument checking.
      if(timeout < 0.0) {
        timeout = 0.0;
      }
      boost::xtime xtimeout;
      if(getXTime(timeout, xtimeout) == false) {
        // Note(xxx): should we throw an exception here?
        return false;
      }
      
      bool returnValue = false;
      try {
        // This call throws a boost::lock_error if the timeout expires.
        boost::timed_mutex::scoped_timed_lock syncLock(
          *m_syncMutexPtr, xtimeout);

        // State checking.
        if(quorumSize == 0) {
          quorumSize = (*m_numberOfClientsPtr);
        }
      
        if((*m_syncClientCountPtr) != 0) {
          if((*m_quorumSizePtr) == 0) {
            DLR_THROW(StateException, "BroadcastQueue::synchronizeQuorum()",
                      "Can't start a quorum while another thread is blocking "
                      "in member function synchronize()");         
          } else if((*m_quorumSizePtr) != quorumSize) {
            std::ostringstream message;
            message << "Can't start a quorum of size " << quorumSize << " "
                    << "while another thread is waiting for a quorum of size "
                    << (*m_quorumSizePtr) << ".";
            DLR_THROW(StateException, "BroadcastQueue::synchronizeQuorum()",
                      message.str().c_str());
          }
        }
        (*m_quorumSizePtr) = quorumSize;
        
        returnValue =
          this->handleSynchronize(
            clientID, (*m_quorumSizePtr), xtimeout, syncLock);
      } catch(const boost::lock_error&) {
        // The timeout expired before we could get a lock.
      }
      return returnValue;
    }

    
    // This member function releases a read lock acquired by
    // lockBack().
    template <class Type>
    void
    BroadcastQueue<Type>::
    unlockBack(const ClientID& clientID)
    {
      // Get exclusive control of the broadcastQueue, and undo the read
      // lock.  The scoped_lock instance will be released when it goes
      // out of scope.
      {
        boost::timed_mutex::scoped_lock headLock(*m_headIndexMutexPtr);

        // m_readLockingFlagsArray keeps track of which threads are
        // read-locking which elements.  Note that lockedFlag is a
        // reference.
        size_t& lockedFlag =
          m_readLockingFlagsArray[clientID.m_index][clientID.m_idNumber];

        // We only need to release the lock if this client is actually
        // locking this element.
        if(lockedFlag != 0) {
          --lockedFlag;
          --(m_readLockCountArray[clientID.m_index]);
        }
      } // Release the lock.
    }
    
    
    /* =========== Private member functions of BroadcastQueue =========== */

    // This private member function is called by the copy
    // constructor and assignment operator to copy another
    // BroadcastQueue instance in a thread-safe way.
    template <class Type>
    void
    BroadcastQueue<Type>::
    copyOther(const BroadcastQueue<Type>& other)
    {
      {
	boost::timed_mutex::scoped_lock
	  headLock(*(other.m_headIndexMutexPtr));

	m_countPtr = other.m_countPtr;
	m_headIndexPtr = other.m_headIndexPtr;
	m_size = other.m_size;
	m_clientNameMapPtr = other.m_clientNameMapPtr;
	m_dataArray = other.m_dataArray;
	m_maxNumberOfClientsPtr = other.m_maxNumberOfClientsPtr;
	m_numberOfClientsPtr = other.m_numberOfClientsPtr;
	m_readLockCountArray = other.m_readLockCountArray;
	m_readLockingFlagsArray = other.m_readLockingFlagsArray;
	m_syncClientCountPtr = other.m_syncClientCountPtr;
	m_syncElementCountPtr = other.m_syncElementCountPtr;
	m_syncIndexPtr = other.m_syncIndexPtr;
	m_quorumSizePtr = other.m_quorumSizePtr;
	m_headIndexMutexPtr = other.m_headIndexMutexPtr;
	m_headChangedConditionPtr = other.m_headChangedConditionPtr;
	// // This code left around in case we try to go back to
	// // individual mutexes for data elements.
	// m_mutexArray = other.m_mutexArray;
	m_numberOfClientsMutexPtr = other.m_numberOfClientsMutexPtr;
	m_syncConditionPtr = other.m_syncConditionPtr;
	m_syncMutexPtr = other.m_syncMutexPtr;
	m_referenceCountPtr = other.m_referenceCountPtr;
	++(*m_referenceCountPtr);
      }
    }

    
    // This private member function should only be called when
    // m_headIndexMutexPtr is locked.
    template <class Type>
    inline void
    BroadcastQueue<Type>::
    declareReadLock(const ClientID& clientID,
                    bool& overflowFlag,
                    size_t& headElementNumber)
    {
      // Check that we're not too behind the head of the queue, and
      // mark our current element as being read.
      if((clientID.m_count + m_size) < (*m_countPtr)) {
        overflowFlag = true;
        headElementNumber = *m_countPtr;
      } else {
        // No overflow.
        ++(m_readLockingFlagsArray[clientID.m_index][clientID.m_idNumber]);
        ++(m_readLockCountArray[clientID.m_index]);
      }
    }
    

    // This private member function does the actual mechanics of
    // synchronizing threads.
    template <class Type>
    void 
    BroadcastQueue<Type>::
    handleSynchronize(ClientID& clientID, size_t targetNumber,
                      boost::timed_mutex::scoped_lock& syncLock)
    {
      // We know (because one of our arguments is a lock object) that
      // it's safe to mess with the m_sync* data members.
      ++(*m_syncClientCountPtr);
      if((*m_syncClientCountPtr) >= targetNumber) {
        {
          // We need to lock headlock before accessing m_countPtr and
          // m_headIndexPtr so they don't change out from under us.
          boost::timed_mutex::scoped_lock headLock(*m_headIndexMutexPtr);
          *m_syncElementCountPtr = *m_countPtr;
          *m_syncIndexPtr = *m_headIndexPtr;
        }
        // Wake up any threads that are waiting for sync.
        (*m_syncClientCountPtr) = 0;
        m_syncConditionPtr->notify_all();
      } else {
        m_syncConditionPtr->wait(syncLock);
      }
      clientID.m_count = *m_syncElementCountPtr;
      clientID.m_index = *m_syncIndexPtr;
    }
    

    // This private member function does the actual mechanics of
    // synchronizing threads.
    template <class Type>
    bool
    BroadcastQueue<Type>::
    handleSynchronize(ClientID& clientID,
                      size_t targetNumber,
                      boost::xtime xtimeout,
                      boost::timed_mutex::scoped_timed_lock& syncLock)
    {
      // We know (because one of our arguments is a lock object) that
      // it's safe to mess with the m_sync* data members.
      ++(*m_syncClientCountPtr);
      if((*m_syncClientCountPtr) >= targetNumber) {
        try {
          // We need to lock headlock before accessing m_countPtr and
          // m_headIndexPtr so they don't change out from under us.
          boost::timed_mutex::scoped_timed_lock
            headLock(*m_headIndexMutexPtr, xtimeout);
          *m_syncElementCountPtr = *m_countPtr;
          *m_syncIndexPtr = *m_headIndexPtr;
        } catch(const boost::lock_error&) {
          --(*m_syncClientCountPtr);
          return false;
        }
        // Wake up any threads that are waiting for sync.
        (*m_syncClientCountPtr) = 0;
        m_syncConditionPtr->notify_all();
      } else {
        if(m_syncConditionPtr->timed_wait(syncLock, xtimeout) == false) {
          // Sync event didn't happen.  Bail out.
          --(*m_syncClientCountPtr);
          return false;
        }
      }
      clientID.m_count = *m_syncElementCountPtr;
      clientID.m_index = *m_syncIndexPtr;
      return true;
    }


    // This protected member function is called by the destructor
    // and copy constructor to decrement reference counts, etc.
    template <class Type>
    void
    BroadcastQueue<Type>::    
    releaseResources()
    {
      bool isShared = true;
      {
        boost::timed_mutex::scoped_lock headLock(*m_headIndexMutexPtr);
        if(--(*m_referenceCountPtr) == 0) {
          isShared = false;
        }
      }
        
      if(!isShared) {
        delete m_countPtr;
        delete m_headIndexPtr;
        delete m_clientNameMapPtr;
        delete[] m_dataArray;
        delete m_maxNumberOfClientsPtr;
        delete m_numberOfClientsPtr;
        delete[] m_readLockCountArray;
        for(size_t index0 = 0; index0 < m_size; ++index0) {
          delete[] m_readLockingFlagsArray[index0];
        }
        delete[] m_readLockingFlagsArray;
        delete m_syncClientCountPtr;
        delete m_syncElementCountPtr;
        delete m_syncIndexPtr;
        delete m_quorumSizePtr;
        delete m_headChangedConditionPtr;
        delete m_headIndexMutexPtr;
        // // This code left around in case we try to go back to
        // // individual mutexes for data elements.
        // delete[] m_mutexArray;
        delete m_numberOfClientsMutexPtr;
        delete m_syncConditionPtr;
        delete m_syncMutexPtr;
        delete m_referenceCountPtr;
      }
    }
    
  } // namespace thread

} // namespace dlr



#endif /* #ifndef _DLR_THREAD_BROADCASTQUEUE_H_ */
