/**
***************************************************************************
* @file array2DTest.cpp
* 
* Source file defining Array2DTest class.
*
* Copyright (C) 2004 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 939 $
* $Date: 2007-06-09 13:08:18 -0400 (Sat, 09 Jun 2007) $
***************************************************************************
**/

#include <math.h>
#include <iomanip>
#include <sstream>
#include <dlrCommon/functional.h>
#include <dlrNumeric/utilities.h>
#include <dlrNumeric/array2D.h>
#include <dlrNumeric/test/arrayNDTest.h>

namespace dlr {

  template <class Type>
  class Array2DTest
    : public ArrayNDTest< Array2DTest<Type>, Array2D<Type>, Array2D<bool> > {

  public:

    typedef Array2DTest<Type> TestFixtureType;


    Array2DTest(const std::string& typeName);
    virtual ~Array2DTest();

    // Inherited methods from TestFixture
    void setUp(const std::string& testName) {}
    void tearDown(const std::string& testName) {}

    // Inherited methods from ArrayNDTest
    virtual void
    checkShapeEquality(const Array2D<Type>& array0,
                       const Array2D<Type>& array1);

    
    virtual void
    checkShapeEquality(const Array2D<Type>& array0,
                       const Array2D<bool>& array1);

    
    virtual void
    checkValueEquality(const Array2D<Type>& array0,
                       const Array2D<Type>& array1,
                       typename Array2D<Type>::value_type tolerance);

    
    virtual typename Array2D<Type>::value_type
    getComparisonOperatorThreshold() {
      return m_fibonacciCArray[m_defaultArraySize / 2];
    }

    
    virtual typename Array2D<Type>::value_type
    getEqualityOperatorTarget() {
      return m_squaresCArray[m_defaultArraySize / 2];
    }

    
    virtual Array2D<Type>
    getFibonacciArray() {return Array2D<Type>(m_fibonacciString);}

    
    virtual typename Array2D<Type>::value_type
    getIncrementOperatorArgument() {return static_cast<Type>(4);}

    
    virtual typename Array2D<Type>::value_type
    getMultiplicationOperatorArgument() {return static_cast<Type>(2);}

    
    virtual Array2D<Type>
    getSquaresArray() {return Array2D<Type>(m_squaresString);}

    
    // Tests of member functions.
    void testConstructor__void();
    void testConstructor__size_t__size_t();
    void testConstructor__string();
    void testConstructor__Array2D();
    void testConstructor__size_t__size_t__TypePtr();
    void testDestructor();
    void testBegin();
    void testBeginConst();
    void testClear();
    void testColumns();
    void testCopy();
    void testCopy__Array2D();
    void testCopy__Type2Ptr();
    void testData();
    void testDataConst();
    void testData__size_t();
    void testDataConst__size_t();
    void testData__size_t__size_t();
    void testDataConst__size_t__size_t();
    void testEnd();
    void testEndConst();
    void testRavel();
    void testRavelConst();
    void testReadFromStream();
    void testReinit();
    void testReshape();
    void testRow();
    void testRowConst();
    void testRows();
    void testShape();
    void testShape__size_t();
    void testSize();
    void testTranspose();
    void testAssignmentOperator__Array2D();
    void testAssignmentOperator__Type();
    void testApplicationOperator__size_t();
    void testApplicationOperatorConst__size_t();
    void testApplicationOperator__size_t__size_t();
    void testApplicationOperatorConst__size_t__size_t();
    void testIndexOperator();
    void testIndexOperatorConst();

    // Tests of non-member functions.
    void testSquareRoot__Array2D();
    void testSqrt__Array2D();


  private:
    
    size_t m_defaultArrayColumns;
    size_t m_defaultArrayRows;
    size_t m_defaultArraySize;
    size_t m_defaultArrayValue;
    size_t m_defaultIncrement;
    size_t m_defaultMultiplier;
    Type* m_fibonacciCArray;
    std::string m_fibonacciString;
    std::string m_illegalString;
    Type* m_squaresCArray;
    std::string m_squaresString;
    double m_testEpsilon;
    
  }; // class Array2DTest


  /* ============== Member Function Definititions ============== */

  template <class Type>
  Array2DTest<Type>::
  Array2DTest(const std::string& typeName)
    : ArrayNDTest< Array2DTest<Type>, Array2D<Type>, Array2D<bool> >(
        std::string("Array2DTest<") + typeName + ">"),
      m_defaultArrayColumns(4),
      m_defaultArrayRows(3),
      m_defaultArraySize(12),
      m_defaultArrayValue(9),
      m_defaultIncrement(4),
      m_defaultMultiplier(2),
      m_fibonacciCArray(0),
      m_fibonacciString(""),
      m_illegalString(),      
      m_squaresCArray(0),
      m_squaresString(""),
      m_testEpsilon(1.0e-8)
  {
    // // Register all tests.
    // Tests of member functions.
    DLR_TEST_REGISTER_MEMBER(testConstructor__void);
    DLR_TEST_REGISTER_MEMBER(testConstructor__size_t__size_t);
    DLR_TEST_REGISTER_MEMBER(testConstructor__string);
    DLR_TEST_REGISTER_MEMBER(testConstructor__Array2D);
    DLR_TEST_REGISTER_MEMBER(testConstructor__size_t__size_t__TypePtr);
    DLR_TEST_REGISTER_MEMBER(testDestructor);
    DLR_TEST_REGISTER_MEMBER(testBegin);
    DLR_TEST_REGISTER_MEMBER(testBeginConst);
    DLR_TEST_REGISTER_MEMBER(testClear);
    DLR_TEST_REGISTER_MEMBER(testColumns);
    DLR_TEST_REGISTER_MEMBER(testCopy);
    DLR_TEST_REGISTER_MEMBER(testCopy__Array2D);
    DLR_TEST_REGISTER_MEMBER(testCopy__Type2Ptr);
    DLR_TEST_REGISTER_MEMBER(testData);
    DLR_TEST_REGISTER_MEMBER(testDataConst);
    DLR_TEST_REGISTER_MEMBER(testData__size_t);
    DLR_TEST_REGISTER_MEMBER(testDataConst__size_t);
    DLR_TEST_REGISTER_MEMBER(testData__size_t__size_t);
    DLR_TEST_REGISTER_MEMBER(testDataConst__size_t__size_t);
    DLR_TEST_REGISTER_MEMBER(testEnd);
    DLR_TEST_REGISTER_MEMBER(testEndConst);
    DLR_TEST_REGISTER_MEMBER(testRavel);
    DLR_TEST_REGISTER_MEMBER(testRavelConst);
    DLR_TEST_REGISTER_MEMBER(testReadFromStream);
    DLR_TEST_REGISTER_MEMBER(testReinit);
    DLR_TEST_REGISTER_MEMBER(testReshape);
    DLR_TEST_REGISTER_MEMBER(testRow);
    DLR_TEST_REGISTER_MEMBER(testRowConst);
    DLR_TEST_REGISTER_MEMBER(testRows);
    DLR_TEST_REGISTER_MEMBER(testShape);
    DLR_TEST_REGISTER_MEMBER(testShape__size_t);
    DLR_TEST_REGISTER_MEMBER(testSize);
    DLR_TEST_REGISTER_MEMBER(testTranspose);
    DLR_TEST_REGISTER_MEMBER(testAssignmentOperator__Array2D);
    DLR_TEST_REGISTER_MEMBER(testAssignmentOperator__Type);
    DLR_TEST_REGISTER_MEMBER(testApplicationOperator__size_t);
    DLR_TEST_REGISTER_MEMBER(testApplicationOperatorConst__size_t);
    DLR_TEST_REGISTER_MEMBER(testApplicationOperator__size_t__size_t);
    DLR_TEST_REGISTER_MEMBER(testApplicationOperatorConst__size_t__size_t);
    DLR_TEST_REGISTER_MEMBER(testIndexOperator);
    DLR_TEST_REGISTER_MEMBER(testIndexOperatorConst);

    // Tests of non-member functions.
    DLR_TEST_REGISTER_MEMBER(testSquareRoot__Array2D);
    DLR_TEST_REGISTER_MEMBER(testSqrt__Array2D);

    
    // Set up fibonacci data for tests.
    m_fibonacciCArray = new Type[m_defaultArraySize];
    m_fibonacciCArray[0] = static_cast<Type>(1);
    m_fibonacciCArray[1] = static_cast<Type>(2);
    for(size_t index = 2; index < m_defaultArraySize; ++index) {
      m_fibonacciCArray[index] = (m_fibonacciCArray[index - 1]
                                  + m_fibonacciCArray[index - 2]);
    }

    // Set up squares data for tests.
    m_squaresCArray = new Type[m_defaultArraySize];
    for(size_t index = 0; index < m_defaultArraySize; ++index) {
      m_squaresCArray[index] = static_cast<Type>(index * index);
    }

    // Set up input strings for tests.
    std::ostringstream fibonacciBuffer;
    std::ostringstream squaresBuffer;
    size_t index0 = 0;
    fibonacciBuffer << std::fixed << "[";
    squaresBuffer << std::fixed << "[";
    for(size_t rowIndex = 0; rowIndex < m_defaultArrayRows;
        ++rowIndex) {
      fibonacciBuffer << "[" << static_cast<Type>(m_fibonacciCArray[index0]);
      squaresBuffer << "[" << static_cast<Type>(m_squaresCArray[index0]);
      ++index0;
      for(size_t column = 1; column < m_defaultArrayColumns; ++column) {
        fibonacciBuffer << ", "
                        << static_cast<Type>(m_fibonacciCArray[index0]);
        squaresBuffer << ", "
                      << static_cast<Type>(m_squaresCArray[index0]);
        ++index0;
      }
      fibonacciBuffer << "]";
      squaresBuffer << "]";
      if(rowIndex != m_defaultArrayRows - 1) {
        fibonacciBuffer << ",\n";
        squaresBuffer << ",\n";
      }
    }
    fibonacciBuffer << "]";
    squaresBuffer << "]";
    m_fibonacciString = fibonacciBuffer.str();
    m_squaresString = squaresBuffer.str();

    // Set up malformed input string for tests.
    m_illegalString = "[[1, 2, 3 4, 5, 6]\n[1, 2, 3, 4, 5, 6]]";
  }


  template <class Type>
  Array2DTest<Type>::
  ~Array2DTest()
  {
    delete[] m_fibonacciCArray;
    delete[] m_squaresCArray;
  }


  template <class Type>
  void
  Array2DTest<Type>::
  checkShapeEquality(const Array2D<Type>& array0,
                     const Array2D<Type>& array1)
  {
    DLR_TEST_ASSERT(array0.rows() == array1.rows());
    DLR_TEST_ASSERT(array0.columns() == array1.columns());
    DLR_TEST_ASSERT(array0.size() == array1.size());
    DLR_TEST_ASSERT(array0.data() != array1.data());
  }


  template <class Type>
  void
  Array2DTest<Type>::
  checkShapeEquality(const Array2D<Type>& array0,
                     const Array2D<bool>& array1)
  {
    DLR_TEST_ASSERT(array0.rows() == array1.rows());
    DLR_TEST_ASSERT(array0.columns() == array1.columns());
    DLR_TEST_ASSERT(array0.size() == array1.size());
  }


  template <class Type>
  void
  Array2DTest<Type>::
  checkValueEquality(const Array2D<Type>& array0,
                     const Array2D<Type>& array1,
                     typename Array2D<Type>::value_type tolerance)
  {
    this->checkShapeEquality(array0, array1);
    DLR_TEST_ASSERT(std::equal(array0.begin(), array0.end(), array1.begin(),
                               ApproximatelyEqualFunctor<Type>(tolerance)));
  }

  
  template <class Type>
  void
  Array2DTest<Type>::
  testConstructor__void()
  {
    // Default constructor should initialize to zero size.
    Array2D<Type> array0;
    DLR_TEST_ASSERT(array0.rows() == 0);
    DLR_TEST_ASSERT(array0.columns() == 0);
    DLR_TEST_ASSERT(array0.size() == 0);
  }
  

  template <class Type>
  void
  Array2DTest<Type>::
  testConstructor__size_t__size_t()
  {
    // This constructor should initialize to the specified size.
    Array2D<Type> array0(m_defaultArrayRows, m_defaultArrayColumns);
    DLR_TEST_ASSERT(array0.rows() == m_defaultArrayRows);
    DLR_TEST_ASSERT(array0.columns() == m_defaultArrayColumns);
    DLR_TEST_ASSERT(array0.size() == m_defaultArraySize);
  }
  

  template <class Type>
  void
  Array2DTest<Type>::
  testConstructor__string()
  {
    // This constructor should initialize according to the specified
    // string, which describes an array of m_defaultArraySize elements
    // having values matching those in m_fibonacciCArray.
    Array2D<Type> array0(m_fibonacciString);
    DLR_TEST_ASSERT(array0.rows() == m_defaultArrayRows);
    DLR_TEST_ASSERT(array0.columns() == m_defaultArrayColumns);
    DLR_TEST_ASSERT(array0.size() == m_defaultArraySize);
    DLR_TEST_ASSERT(std::equal(array0.begin(), array0.end(),
                               m_fibonacciCArray));

    // Construct using a malformed string.
    DLR_TEST_ASSERT_EXCEPTION(ValueException,
                              Array2D<Type> array0(m_illegalString));

  }
  

  template <class Type>
  void
  Array2DTest<Type>::
  testConstructor__Array2D()
  {
    // This constructor should make an array which references the same
    // data as its argument.
    Array2D<Type>* array0Ptr = new Array2D<Type>(m_fibonacciString);
    Array2D<Type> array1(*array0Ptr);
    DLR_TEST_ASSERT(array1.data() == array0Ptr->data());
    DLR_TEST_ASSERT(array1.rows() == array0Ptr->rows());
    DLR_TEST_ASSERT(array1.columns() == array0Ptr->columns());
    DLR_TEST_ASSERT(array1.size() == array0Ptr->size());
    delete array0Ptr;

    // Normally, deleting array0Ptr would invalidate the data it
    // points to.  If reference counting is working, however, the data
    // should remain valid until after array1 goes out of scope.
    DLR_TEST_ASSERT(std::equal(array1.begin(), array1.end(),
                               m_fibonacciCArray));
  }
  

  template <class Type>
  void
  Array2DTest<Type>::
  testConstructor__size_t__size_t__TypePtr()
  {
    // This constructor initializes the array to point to the
    // specified data.
    // Start by making an array we can play with.
    Type* cArray = new Type[m_defaultArraySize];
    std::copy(m_fibonacciCArray, m_fibonacciCArray + m_defaultArraySize,
              cArray);
    
    // Create an array using the constructor under test.
    Array2D<Type>* array0Ptr = new Array2D<Type>(
      m_defaultArrayRows, m_defaultArrayColumns, cArray);
    DLR_TEST_ASSERT(array0Ptr->data() == cArray);
    DLR_TEST_ASSERT(array0Ptr->rows() == m_defaultArrayRows);
    DLR_TEST_ASSERT(array0Ptr->columns() == m_defaultArrayColumns);
    DLR_TEST_ASSERT(array0Ptr->size() == m_defaultArraySize);
    delete array0Ptr;

    // In this case, no reference counting should be done, and the
    // data should not be deleted on deletion of array0Ptr.
    DLR_TEST_ASSERT(std::equal(cArray, cArray + m_defaultArraySize,
                               m_fibonacciCArray));

    // Clean up.
    delete[] cArray;
  }
  

//   testConstructor__size_t__TypePtr__size_tPtr()
//   {
//     // This constructor should reference the provided test data _and_
//     // maintain the reference count.

//     // First set up some data to reference.
//     Array2D<Type> array0(m_fibonacciString);

//     // Create a size_t to hold the reference count.
//     size_t refCount = 1;

//     // Create the array.  It should now reference the same data as the
//     // first array, and the reference count should be incremented.
//     Array2D<Type>* array2Ptr = new Array2D<Type>(
//       array0.size(), array0.data(), &refCount);
//     DLR_TEST_ASSERT(array2Ptr->data() == array0.data());
//     DLR_TEST_ASSERT(array2Ptr->size() == array0.size());
//     DLR_TEST_ASSERT(refCount == 2);
//     DLR_TEST_ASSERT(std::equal(array2Ptr->begin(), array2Ptr->end(),
//                                m_fibonacciCArray));

//     // Following deletion of the new array, the reference count should
//     // be back down to 1.
//     delete array2Ptr;
//     DLR_TEST_ASSERT(refCount == 1);
//   }


  
  template <class Type>
  void
  Array2DTest<Type>::
  testDestructor()
  {
    // No independent test for destructor.
  }


  template <class Type>
  void
  Array2DTest<Type>::
  testBegin()
  {
    // Member function begin() should return an iterator pointing to
    // the first element of the array.
    Array2D<Type> array0(
      m_defaultArrayRows, m_defaultArrayColumns, m_fibonacciCArray);
    DLR_TEST_ASSERT(&(*(array0.begin())) == m_fibonacciCArray);
  }


  template <class Type>
  void
  Array2DTest<Type>::
  testBeginConst()
  {
    // Member function begin() should return an iterator pointing to
    // the first element of the array.
    const Array2D<Type> array0(
      m_defaultArrayRows, m_defaultArrayColumns, m_fibonacciCArray);
    DLR_TEST_ASSERT(&(*(array0.begin())) == m_fibonacciCArray);
  }


  template <class Type>
  void
  Array2DTest<Type>::
  testClear()
  {
    // Member function clear should reset the array to zero size,
    // abandoning all contents.
    // First set up some data to reference.
    Array2D<Type> array0(m_fibonacciString);

    // Clearing the array should release the reference.
    array0.clear();
    DLR_TEST_ASSERT(array0.size() == 0);

//     // Member function clear should reset the array to zero size,
//     // abandoning all contents.
//     // First set up some data to reference.
//     Array2D<Type> array0(m_fibonacciString);

//     // Create a size_t to hold a reference count.
//     size_t refCount = 1;

//     // Create the test array.  It should now reference the same data
//     // as the first array, and the reference count should be
//     // incremented.
//     Array2D<Type>array1(array0.size(), array0.data(), &refCount);
//     DLR_TEST_ASSERT(refCount == 2);

//     // Clearing the array should release the reference.
//     array1.clear();
//     DLR_TEST_ASSERT(array1.size() == 0);
//     DLR_TEST_ASSERT(refCount == 1);
  }


  template <class Type>
  void
  Array2DTest<Type>::
  testColumns()
  {
    // No explicit test.
  }


  template <class Type>
  void
  Array2DTest<Type>::
  testCopy()
  {
    // Member function copy() should allocate a new array and deep
    // copy the contents of *this.
    Array2D<Type> array0(m_fibonacciString);
    Array2D<Type> array1 = array0.copy();
    DLR_TEST_ASSERT(array1.size() == array0.size());
    DLR_TEST_ASSERT(array1.rows() == array0.rows());
    DLR_TEST_ASSERT(array1.columns() == array0.columns());
    DLR_TEST_ASSERT(std::equal(array1.begin(), array1.end(), array0.begin()));
    DLR_TEST_ASSERT(array1.data() != array0.data());    
  }


  template <class Type>
  void
  Array2DTest<Type>::
  testCopy__Array2D()
  {
    // This member function deep copies the contents of source.  It is
    // an error if source does not have the same size as *this.
    Array2D<Type> array0(m_fibonacciString);
    Array2D<Type> array1(array0.rows(), array0.columns());
    Array2D<Type> array2(array0.rows(), array0.columns() + 1);
    Array2D<Type> array3(array0.rows() + 1, array0.columns());

    // Test that the copy goes OK.
    array1.copy(array0);
    DLR_TEST_ASSERT(array1.rows() == array0.rows());
    DLR_TEST_ASSERT(array1.columns() == array0.columns());
    DLR_TEST_ASSERT(array1.size() == array0.size());
    DLR_TEST_ASSERT(std::equal(array1.begin(), array1.end(), array0.begin()));
    DLR_TEST_ASSERT(array1.data() != array0.data());

    // Test that size is checked.
    DLR_TEST_ASSERT_EXCEPTION(ValueException, array2.copy(array0));
    DLR_TEST_ASSERT_EXCEPTION(ValueException, array3.copy(array0));
  }


  template <class Type>
  void
  Array2DTest<Type>::
  testCopy__Type2Ptr()
  {
    // This method simply copies elements from its argument.
    // Begin by making an appropriately sized array with known contents.
    Array2D<Type> array0(m_defaultArrayRows, m_defaultArrayColumns);
    array0 = 0;

    array0.copy(m_fibonacciCArray);
    DLR_TEST_ASSERT(array0.rows() == m_defaultArrayRows);
    DLR_TEST_ASSERT(array0.columns() == m_defaultArrayColumns);
    DLR_TEST_ASSERT(array0.size() == m_defaultArraySize);
    DLR_TEST_ASSERT(std::equal(array0.begin(), array0.end(),
                               m_fibonacciCArray));
    DLR_TEST_ASSERT(array0.data() != m_fibonacciCArray);
  }


  template <class Type>
  void
  Array2DTest<Type>::
  testData()
  {
    // This method returns a pointer to the internal data store.
    Array2D<Type> array0(
      m_defaultArrayRows, m_defaultArrayColumns, m_fibonacciCArray);
    DLR_TEST_ASSERT(array0.data() == m_fibonacciCArray);
  }


  template <class Type>
  void
  Array2DTest<Type>::
  testDataConst()
  {
    // This method returns a pointer to the internal data store.
    const Array2D<Type> array0(
      m_defaultArrayRows, m_defaultArrayColumns, m_fibonacciCArray);
    DLR_TEST_ASSERT(array0.data() == m_fibonacciCArray);
  }


  template <class Type>
  void
  Array2DTest<Type>::
  testData__size_t()
  {
    // This method returns a pointer to the index-th element of the
    // internal data store.
    Array2D<Type> array0(
      m_defaultArrayRows, m_defaultArrayColumns, m_fibonacciCArray);
    DLR_TEST_ASSERT(array0.data(m_defaultArraySize - 1)
                    == (m_fibonacciCArray + m_defaultArraySize - 1));
  }


  template <class Type>
  void
  Array2DTest<Type>::
  testDataConst__size_t()
  {
    // This method returns a pointer to the index-th element of the
    // internal data store.
    Array2D<Type> array0(
      m_defaultArrayRows, m_defaultArrayColumns, m_fibonacciCArray);
    DLR_TEST_ASSERT(array0.data(m_defaultArraySize - 1)
                    == (m_fibonacciCArray + m_defaultArraySize - 1));
  }


  template <class Type>
  void
  Array2DTest<Type>::
  testData__size_t__size_t()
  {
    // This method returns a pointer to the index-th element of the
    // internal data store.
    Array2D<Type> array0(
      m_defaultArrayRows, m_defaultArrayColumns, m_fibonacciCArray);
    DLR_TEST_ASSERT(array0.data(0, 0) == m_fibonacciCArray);
    DLR_TEST_ASSERT(array0.data(0, 1) == (m_fibonacciCArray + 1));
    DLR_TEST_ASSERT(array0.data(1, 0) 
                    == (m_fibonacciCArray + m_defaultArrayColumns));
    DLR_TEST_ASSERT(array0.data(m_defaultArrayRows - 1,
                                m_defaultArrayColumns - 1)
                    == (m_fibonacciCArray + m_defaultArraySize - 1));
  }


  template <class Type>
  void
  Array2DTest<Type>::
  testDataConst__size_t__size_t()
  {
    // This method returns a pointer to the index-th element of the
    // internal data store.
    const Array2D<Type> array0(
      m_defaultArrayRows, m_defaultArrayColumns, m_fibonacciCArray);
    DLR_TEST_ASSERT(array0.data(0, 0) == m_fibonacciCArray);
    DLR_TEST_ASSERT(array0.data(0, 1) == (m_fibonacciCArray + 1));
    DLR_TEST_ASSERT(array0.data(1, 0) 
                    == (m_fibonacciCArray + m_defaultArrayColumns));
    DLR_TEST_ASSERT(array0.data(m_defaultArrayRows - 1,
                                m_defaultArrayColumns - 1)
                    == (m_fibonacciCArray + m_defaultArraySize - 1));
  }


  template <class Type>
  void
  Array2DTest<Type>::
  testEnd()
  {
    // Member function end() should return an iterator pointing to
    // the last element of the array.
    Array2D<Type> array0(
      m_defaultArrayRows, m_defaultArrayColumns, m_fibonacciCArray);
    Type* finalElementPtr = m_fibonacciCArray + m_defaultArraySize;
    DLR_TEST_ASSERT(&(*(array0.end())) == finalElementPtr);
  }
    

  template <class Type>
  void
  Array2DTest<Type>::
  testEndConst()
  {
    // Member function end() should return an iterator pointing to
    // the last element of the array.
    const Array2D<Type> array0(
      m_defaultArrayRows, m_defaultArrayColumns, m_fibonacciCArray);
    Type* finalElementPtr = m_fibonacciCArray + m_defaultArraySize;
    DLR_TEST_ASSERT(&(*(array0.end())) == finalElementPtr);
  }


  template <class Type>
  void
  Array2DTest<Type>::
  testRavel()
  {
    // Member function ravel() should return an Array1D instance referencing
    // the same data.
    Array2D<Type>* array0Ptr = new Array2D<Type>(
      m_defaultArrayRows, m_defaultArrayColumns, m_fibonacciCArray);
    Array1D<Type> array1 = array0Ptr->ravel();

    // Check for correct value.
    DLR_TEST_ASSERT(array1.size() == array0Ptr->size());
    DLR_TEST_ASSERT(array1.data() == array0Ptr->data());
    DLR_TEST_ASSERT(std::equal(array1.begin(), array1.end(),
                               m_fibonacciCArray));

    // Delete array0Ptr and hope that array1 is still valid.
    delete array0Ptr;
    DLR_TEST_ASSERT(std::equal(array1.begin(), array1.end(),
                               m_fibonacciCArray));
  }
  

  template <class Type>
  void
  Array2DTest<Type>::
  testRavelConst()
  {
    // Member function ravel() should return an Array1D instance referencing
    // the same data.
    const Array2D<Type>* array0Ptr = new Array2D<Type>(
      m_defaultArrayRows, m_defaultArrayColumns, m_fibonacciCArray);
    const Array1D<Type> array1 = array0Ptr->ravel();
    
    // Check for correct value.
    DLR_TEST_ASSERT(array1.size() == array0Ptr->size());
    DLR_TEST_ASSERT(array1.data() == array0Ptr->data());
    DLR_TEST_ASSERT(std::equal(array1.begin(), array1.end(),
                               m_fibonacciCArray));

    // Delete array0Ptr and hope that array1 is still valid.
    delete array0Ptr;
    DLR_TEST_ASSERT(std::equal(array1.begin(), array1.end(),
                               m_fibonacciCArray));
  }
  

  template <class Type>
  void
  Array2DTest<Type>::
  testReadFromStream()
  {
    // This member function sets the value of the array from an input
    // stream.
    // Set up input streams
    std::istringstream inputStream0(m_fibonacciString);
    std::istringstream inputStream1(m_illegalString);

    // Try to parse the good stream.
    Array2D<Type> array0;
    array0.readFromStream(inputStream0);
    DLR_TEST_ASSERT(inputStream0);
    DLR_TEST_ASSERT(array0.rows() == m_defaultArrayRows);
    DLR_TEST_ASSERT(array0.columns() == m_defaultArrayColumns);
    DLR_TEST_ASSERT(array0.size() == m_defaultArraySize);
    DLR_TEST_ASSERT(std::equal(array0.begin(), array0.end(),
                               m_fibonacciCArray));

    // Try to parse the bad stream.  This should not affect array0 in
    // any way, but it should change the stream state.
    array0.readFromStream(inputStream1);
    DLR_TEST_ASSERT(!inputStream1);
    DLR_TEST_ASSERT(array0.rows() == m_defaultArrayRows);
    DLR_TEST_ASSERT(array0.columns() == m_defaultArrayColumns);
    DLR_TEST_ASSERT(array0.size() == m_defaultArraySize);
    DLR_TEST_ASSERT(std::equal(array0.begin(), array0.end(),
                               m_fibonacciCArray));
  }


  template <class Type>
  void
  Array2DTest<Type>::
  testReinit()
  {
    // This method changes the shape of the array and reallocates
    // storage.
    Array2D<Type> array0(
      m_defaultArrayRows, m_defaultArrayColumns, m_fibonacciCArray);
    array0.reinit(m_defaultArrayRows + 1, m_defaultArrayColumns + 2);
    DLR_TEST_ASSERT(array0.rows() == m_defaultArrayRows + 1);
    DLR_TEST_ASSERT(array0.columns() == m_defaultArrayColumns + 2);
    DLR_TEST_ASSERT(array0.size() ==
                    (m_defaultArrayRows + 1) * (m_defaultArrayColumns + 2));
    DLR_TEST_ASSERT(array0.data() != m_fibonacciCArray);
  }

  
  template <class Type>
  void
  Array2DTest<Type>::
  testReshape()
  {
    // This method changes the shape of the array withot reallocating
    // storage.
    Array2D<Type> array0(
      m_defaultArrayRows, m_defaultArrayColumns, m_fibonacciCArray);

    // Test that size is checked.
    DLR_TEST_ASSERT_EXCEPTION(
      ValueException, 
      array0.reshape(static_cast<int>(m_defaultArrayRows) + 1, 
			         static_cast<int>(m_defaultArrayColumns) + 2));

    // Check that correctly sized reshapes work.
    array0.reshape(1, static_cast<int>(m_defaultArraySize));
    DLR_TEST_ASSERT(array0.rows() == 1);
    DLR_TEST_ASSERT(array0.columns() == m_defaultArraySize);
    DLR_TEST_ASSERT(array0.size() == m_defaultArraySize);
    DLR_TEST_ASSERT(array0.data() == m_fibonacciCArray);
  }


  template <class Type>
  void
  Array2DTest<Type>::
  testRow()
  {
    Array2D<Type> array0(m_fibonacciString);
    Type* dataPtr = array0.data();
    Array1D<Type> array1 = array0.row(1);

    DLR_TEST_ASSERT(array0.data() == dataPtr);
    DLR_TEST_ASSERT(array1.size() == m_defaultArrayColumns);
    DLR_TEST_ASSERT(array1.data() == dataPtr + m_defaultArrayColumns);
    for(size_t index0 = 0; index0 < m_defaultArrayColumns; ++index0) {
      DLR_TEST_ASSERT(array1[index0] ==
                      m_fibonacciCArray[m_defaultArrayColumns + index0]);
    }
  }

  
  template <class Type>
  void
  Array2DTest<Type>::
  testRowConst()
  {
    const Array2D<Type> array0(m_fibonacciString);
    const Type* dataPtr = array0.data();
    const Array1D<Type> array1 = array0.row(1);

    DLR_TEST_ASSERT(array0.data() == dataPtr);
    DLR_TEST_ASSERT(array1.size() == m_defaultArrayColumns);
    DLR_TEST_ASSERT(array1.data() == dataPtr + m_defaultArrayColumns);
    for(size_t index0 = 0; index0 < m_defaultArrayColumns; ++index0) {
      DLR_TEST_ASSERT(array1[index0] ==
                      m_fibonacciCArray[m_defaultArrayColumns + index0]);
    }
  }

  
  template <class Type>
  void
  Array2DTest<Type>::
  testRows()
  {
    // No explicit test.
  }


  template <class Type>
  void
  Array2DTest<Type>::
  testShape()
  {
    Array2D<Type> array0(
      m_defaultArrayRows, m_defaultArrayColumns, m_fibonacciCArray);
    Array1D<size_t> shapeArray = array0.shape();
    DLR_TEST_ASSERT(shapeArray.size() == 2);
    DLR_TEST_ASSERT(shapeArray(0) == m_defaultArrayRows);
    DLR_TEST_ASSERT(shapeArray(1) == m_defaultArrayColumns);
  }


  template <class Type>
  void
  Array2DTest<Type>::
  testShape__size_t()
  {
    Array2D<Type> array0(
      m_defaultArrayRows, m_defaultArrayColumns, m_fibonacciCArray);
    DLR_TEST_ASSERT(array0.shape(0) == m_defaultArrayRows);
    DLR_TEST_ASSERT(array0.shape(1) == m_defaultArrayColumns);
  }


  template <class Type>
  void
  Array2DTest<Type>::
  testSize()
  {
    // No explicit test.
  }


  template <class Type>
  void
  Array2DTest<Type>::
  testTranspose()
  {
    const Array2D<Type> array0(m_fibonacciString);
    const Array2D<Type> array1 = array0.transpose();

    DLR_TEST_ASSERT(array0.rows() == m_defaultArrayRows);
    DLR_TEST_ASSERT(array0.columns() == m_defaultArrayColumns);
    DLR_TEST_ASSERT(array0.size() == m_defaultArraySize);    
    DLR_TEST_ASSERT(array1.rows() == m_defaultArrayColumns);
    DLR_TEST_ASSERT(array1.columns() == m_defaultArrayRows);
    DLR_TEST_ASSERT(array1.size() == m_defaultArraySize);    
    size_t index0 = 0;
    for(size_t row = 0; row < m_defaultArrayRows;
        ++row) {
      for(size_t column = 0; column < m_defaultArrayColumns; ++column) {
        DLR_TEST_ASSERT(array0(row, column) == m_fibonacciCArray[index0]);
        DLR_TEST_ASSERT(array1(column, row) == m_fibonacciCArray[index0]);
        ++index0;
      }
    }
  }
  

  template <class Type>
  void
  Array2DTest<Type>::
  testAssignmentOperator__Array2D()
  {
    // The assignment operator shallow copies the contents of source.
    Array2D<Type> array0(m_fibonacciString);
    Array2D<Type> array1;
    array1 = array0;
    DLR_TEST_ASSERT(array0.rows() == m_defaultArrayRows);
    DLR_TEST_ASSERT(array0.columns() == m_defaultArrayColumns);
    DLR_TEST_ASSERT(array0.size() == m_defaultArraySize);
    DLR_TEST_ASSERT(array1.rows() == m_defaultArrayRows);
    DLR_TEST_ASSERT(array1.columns() == m_defaultArrayColumns);
    DLR_TEST_ASSERT(array1.size() == array0.size());
    DLR_TEST_ASSERT(array1.data() == array0.data());

    // Make sure the data wasn't changed by the operation.
    DLR_TEST_ASSERT(std::equal(array1.begin(), array1.end(),
                               m_fibonacciCArray));
  }

  
  template <class Type>
  void
  Array2DTest<Type>::
  testAssignmentOperator__Type()
  {
    // This member function assigns the same value to every element in
    // the array.
    Array2D<Type> array0(m_defaultArrayRows, m_defaultArrayColumns);
    Type* dataPtr = array0.data();
    array0 = static_cast<Type>(m_defaultArrayValue);

    // Check that the size and location of the array wasn't changed.
    DLR_TEST_ASSERT(array0.rows() == m_defaultArrayRows);
    DLR_TEST_ASSERT(array0.columns() == m_defaultArrayColumns);
    DLR_TEST_ASSERT(array0.size() == m_defaultArraySize);
    DLR_TEST_ASSERT(array0.data() == dataPtr);

    // Check that all values were set by getting a pointer to the
    // first one that's not equal to m_defaultArrayValue
    typename Array2D<Type>::iterator firstRenegade =
      std::find_if(array0.begin(), array0.end(),
                   std::bind2nd(std::not_equal_to<Type>(),
                                static_cast<Type>(m_defaultArrayValue)));
    DLR_TEST_ASSERT(firstRenegade == array0.end());
  }

  
  template <class Type>
  void
  Array2DTest<Type>::
  testApplicationOperator__size_t()
  {
    // This member function returns the (index)th element of the array
    // by reference.
    Array2D<Type> array0(m_fibonacciString);
    for(size_t index = 0; index < array0.size(); ++index) {
      DLR_TEST_ASSERT(array0(index) == m_fibonacciCArray[index]);
    }
  }


  template <class Type>
  void
  Array2DTest<Type>::
  testApplicationOperatorConst__size_t()
  {
    // This member function returns the (index)th element of the array
    // by reference.
    const Array2D<Type> array0(m_fibonacciString);
    for(size_t index = 0; index < array0.size(); ++index) {
      DLR_TEST_ASSERT(array0(index) == m_fibonacciCArray[index]);
    }
  }


  template <class Type>
  void
  Array2DTest<Type>::
  testApplicationOperator__size_t__size_t()
  {
    Array2D<Type> array0(m_fibonacciString);

    size_t index0 = 0;
    for(size_t row = 0; row < m_defaultArrayRows;
        ++row) {
      for(size_t column = 0; column < m_defaultArrayColumns; ++column) {
        DLR_TEST_ASSERT(array0(row, column) == m_fibonacciCArray[index0]);
        ++index0;
      }
    }
  }


  template <class Type>
  void
  Array2DTest<Type>::
  testApplicationOperatorConst__size_t__size_t()
  {
    const Array2D<Type> array0(m_fibonacciString);

    size_t index0 = 0;
    for(size_t row = 0; row < m_defaultArrayRows; ++row) {
      for(size_t column = 0; column < m_defaultArrayColumns; ++column) {
        DLR_TEST_ASSERT(array0(row, column) == m_fibonacciCArray[index0]);
        ++index0;
      }
    }
  }


  template <class Type>
  void
  Array2DTest<Type>::
  testIndexOperator()
  {
    // This member function returns the (index)th element of the array
    // by reference.
    Array2D<Type> array0(m_fibonacciString);
    for(size_t index = 0; index < array0.size(); ++index) {
      DLR_TEST_ASSERT(array0[index] == m_fibonacciCArray[index]);
    }
  }


  template <class Type>
  void
  Array2DTest<Type>::
  testIndexOperatorConst()
  {
    // This member function returns the (index)th element of the array
    // by reference.
    const Array2D<Type> array0(m_fibonacciString);
    for(size_t index = 0; index < array0.size(); ++index) {
      DLR_TEST_ASSERT(array0[index] == m_fibonacciCArray[index]);
    }
  }


  // General test of square root is not performed, since square root
  // only makes sense for certain types.
  template <class Type>
  void
  Array2DTest<Type>::
  testSquareRoot__Array2D()
  {
    // Empty.
  }


  // Specialization tests square root for Array2D<double>.
  template<>
  void
  Array2DTest<double>::
  testSquareRoot__Array2D()
  {
    const Array2D<double> array0(m_fibonacciString);
    const Array2D<double> array1 = squareRoot(array0);

    DLR_TEST_ASSERT(array1.rows() == m_defaultArrayRows);
    DLR_TEST_ASSERT(array1.columns() == m_defaultArrayColumns);
    DLR_TEST_ASSERT(array1.size() == m_defaultArraySize);
    for(size_t index0 = 0; index0 < array1.size(); ++index0) {
      double targetValue = std::sqrt(
        static_cast<double>(m_fibonacciCArray[index0]));
      // Grr. std::sqrt(x) return value appears to vary slightly from
      // call to call.  Possibly this is due to pentium register
      // precision confusion.
      //
      // DLR_TEST_ASSERT(array1(index0) == targetValue);
      DLR_TEST_ASSERT(
        approximatelyEqual(array1(index0), targetValue, 1.0E-14));
    }
  }
  


  // General test of square root is not performed, since square root
  // only makes sense for certain types.
  template <class Type>
  void
  Array2DTest<Type>::
  testSqrt__Array2D()
  {
    // Empty.
  }


  // Specialization tests square root for Array2D<double>.
  template<>
  void
  Array2DTest<double>::
  testSqrt__Array2D()
  {
    const Array2D<double> array0(m_fibonacciString);
    const Array2D<double> array1 = sqrt(array0);

    DLR_TEST_ASSERT(array1.rows() == m_defaultArrayRows);
    DLR_TEST_ASSERT(array1.columns() == m_defaultArrayColumns);
    DLR_TEST_ASSERT(array1.size() == m_defaultArraySize);
    for(size_t index0 = 0; index0 < array1.size(); ++index0) {
      double targetValue = std::sqrt(
        static_cast<double>(m_fibonacciCArray[index0]));
      // Grr. std::sqrt(x) return value appears to vary slightly from
      // call to call.  Possibly this is due to pentium register
      // precision confusion.
      //
      // DLR_TEST_ASSERT(array1(index0) == targetValue);
      DLR_TEST_ASSERT(
        approximatelyEqual(array1(index0), targetValue, 1.0E-14));
    }
  }

  
} // namespace dlr


#if 0

int main(int argc, char** argv)
{
  dlr::Array2DTest<double> currentTest0("double");
  dlr::Array2DTest<float> currentTest1("float");
  dlr::Array2DTest<int> currentTest2("int");
  dlr::Array2DTest<size_t> currentTest3("size_t");
  bool result = (currentTest0.run()
                 && currentTest1.run()
                 && currentTest2.run()
                 && currentTest3.run());
  return (result ? 0 : 1);
}

#else

namespace {

  dlr::Array2DTest<double> currentTest0("double");
  dlr::Array2DTest<float> currentTest1("float");
  dlr::Array2DTest<int> currentTest2("int");
  // dlr::Array2DTest<size_t> currentTest3("size_t");

}

#endif

