/**
***************************************************************************
* @file dlrNumeric/sampledFunctions.h
*
* Header file declaring code which return sampled versions of common
* functions, such as gaussians, etc.
*
* Copyright (C) 2006 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: $
* $Date: $
***************************************************************************
*/

#ifndef _DLRNUMERIC_SAMPLEDFUNCTIONS_H_
#define _DLRNUMERIC_SAMPLEDFUNCTIONS_H_

#include <dlrNumeric/array1D.h>

namespace dlr {

  namespace numeric {

    /** 
     * This function returns an array in which the elements are
     * sampled from a 1D Gaussian.  The maximum of the Gaussian is at
     * the center of the returned array, and the distance between
     * adjacent samples (elements of the array) is assumed to be
     * exactly 1.0.
     * 
     * @param size This argument specifies how many elements should be
     * in the returned array.
     * 
     * @param sigma This argument specifies the desired standard
     * deviation of the Gaussian.
     * 
     * @param normalize This argument specifies whether, after the
     * elements of the array have been computed, the array should be
     * rescaled so that its elements sum to 1.0.
     * 
     * @return The return value is an array containing the sampled
     * values.
     */
    template <class TYPE>
    Array1D<TYPE>
    getGaussian1D(size_t size,
                  double sigma = 0.0,
                  bool normalize = false);

  } // namespace numeric
  
} // namespace dlr


/* =============================================================== */
/* Implementation follows.                                         */
/* =============================================================== */


#include <cmath>

namespace dlr {

  namespace numeric {

    // This function returns an array in which the elements are
    // sampled from a 1D Gaussian.
    Array1D<TYPE>
    getGaussian1D(size_t size, double sigma, bool normalize)
    {
      Array1D<TYPE> result(size);
      Gaussian1DFunctor functor(sigma);
      double x = (1.0 - static_cast<double>(size))/2.0;
      for(size_t index0 = 0; index0 < size; ++index0) {
        result[index0] = functor(x);
        x += 1.0;
      }
      if(normalize) {
        result /= sum(result);
      }
      return result;
    }
    
  } // namespace numeric
  
} // namespace dlr

#endif /* #ifndef _DLRNUMERIC_SAMPLEDFUNCTIONS_H_ */
