/**
***************************************************************************
* @file dlrNumeric/polynomial.h
*
* Header file declaring a class for representing simple polynomials.
*
* Copyright (C) 2006 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: $
* $Date: $
***************************************************************************
*/

#ifndef _DLRNUMERIC_POLYNOMIAL_H_
#define _DLRNUMERIC_POLYNOMIAL_H_

#include <dlrNumeric/array1D.h>

namespace dlr {

  namespace numeric {


    /**
     ** This class represents polynomials of the form
     **
     **   p(x) = k0 + (k1 * x) + (k2 * x^2) + (k3 * x^3) ...
     **
     ** It provides operators for evaluating the polynomial, as well
     ** as for combining the polynomial with other polynomials.
     **/
    template <class Type>
    class Polynomial {
    public:

      /** 
       * The default constructor makes a constant polynomial: p(x) = 1.
       */
      Polynomial();


      /** 
       * This constructor makes a constant polynomial:
       *
       *   p(x) = coefficient0.
       * 
       * @param coefficient0 This argument specifies the (constant)
       * value of the polynomial.
       */
      explicit
      Polynomial(Type coefficient0);


      /** 
       * This constructor makes a first order polynomial:
       *
       *   p(x) = coefficient0 + (coefficient1 * x)
       * 
       * @param coefficient1 This argument specifies one of the
       * coefficients of the polynomial.
       * 
       * @param coefficient0 This argument specifies one of the
       * coefficients of the polynomial.
       */
      Polynomial(Type coefficient1, Type coefficient0);


      /** 
       * This constructor makes a second order polynomial:
       *
       *   p(x) = coefficient0 + (coefficient1 * x) + (coefficient2 * x^2)
       * 
       * @param coefficient2 This argument specifies one of the
       * coefficients of the polynomial.
       *
       * @param coefficient1 This argument specifies one of the
       * coefficients of the polynomial.
       * 
       * @param coefficient0 This argument specifies one of the
       * coefficients of the polynomial.
       */
      Polynomial(Type coefficient2, Type coefficient1, Type coefficient0);


      /** 
       * This constructor makes a polynomial of arbitrary order.
       *
       *   p(x) = coefficients[0] + (coefficients[1] * x)
       *          + (coefficients[2] * x2) + ...
       *          + (coefficients[N - 1] * x^(N-1)
       *
       * where N = coefficients.size()
       * 
       * @param coefficients This argument is an array of coefficients
       * for the polynomial, as described above.
       */
      explicit
      Polynomial(const Array1D<Type> coefficients);


      /** 
       * The copy constructor does a deep copy.
       * 
       * @param other This argument is the Polynomial instance to copy.
       */
      Polynomial(const Polynomial<Type>& other);


      /** 
       * This member function returns the coefficients of the
       * polynomial.  The array values are arranged in the same way as
       * described in the documentation for constructor
       * Polynomial(const Array1D<Type>&).
       * 
       * @return The return value is an array of coefficients.
       */
      Array1D<Type>
      getCoefficientArray() const;


      /** 
       * This member function returns the order of the polynomial: 0
       * for constant; 1 for linear; 2 for quadratic; and so on.
       * 
       * @return The return value is the order of the polynomial.
       */
      size_t
      getOrder() const {return m_coefficientArray.size() - 1;}
      

      /** 
       * The assigment operator does a deep copy.
       * 
       * @param other This argument is the Polynomial instance to copy.
       *
       * @return The return value is a reference to *this.
       */
      Polynomial<Type>&
      operator=(const Polynomial<Type> other);
      

      /** 
       * This operator evaluates the polynomial.
       *
       * @param xValue This argument specifies the value of x at which
       * to evaluate the polynomial.
       *
       * @return The return value is the result of the evaluation.
       */
      Type
      operator()(Type xValue) const;


      /** 
       * This operator multiplies the polynomial by another
       * polynomial.  After the multipliation, the result of calling
       * this->operator()(xValue) will be equal to the previous
       * (before the call to this->operator*=()) result of
       * this->operator()(xValue) multiplied by the result of
       * other.operator()(xValue).
       * 
       * @param other This argument is the the polynomial by which to
       * multiply *this.
       * 
       * @return The return value is a reference to *this.
       */
      Polynomial&
      operator*=(const Polynomial& other);


      /** 
       * This operator adds another polynomial to *this.  After the
       * addition, the result of calling this->operator()(xValue)
       * will be equal to the previous (before the call to
       * this->operator+=()) result of this->operator()(xValue) plus
       * the result of other.operator()(xValue).
       * 
       * @param other This argument is the the polynomial to be added
       * to *this.
       * 
       * @return The return value is a reference to *this.
       */
      Polynomial&
      operator+=(const Polynomial& other);


      /** 
       * This operator subtracts another polynomial from *this.  After
       * the subtraction, the result of calling
       * this->operator()(xValue) will be equal to the previous
       * (before the call to this->operator+=()) result of
       * this->operator()(xValue) minus the result of
       * other.operator()(xValue).
       * 
       * @param other This argument is the the polynomial to be subtracted
       * from *this.
       * 
       * @return The return value is a reference to *this.
       */
      Polynomial&
      operator-=(const Polynomial& other);
      
    private:

      Array1D<Type> m_coefficientArray;
      
    };

    
    /* ============ Non-member function declarations ============ */

  
    /** 
     * This operator multiplies two Polynomial instances.  Values
     * returned by operator()(xValue) of the result of this
     * multiplication will be equal to the product of the values
     * returned by the operator()(xValue) of arg0 and arg1.
     * 
     * @param arg0 This argument is one of the Polynomial instances to
     * be multiplied.
     * 
     * @param arg1 This argument is one of the Polynomial instances to
     * be multiplied.
     * 
     * @return The return value is the result of the multiplication.
     */
    template<class Type>
    Polynomial<Type>
    operator*(const Polynomial<Type>& arg0, const Polynomial<Type>& arg1);


    /** 
     * This operator adds two Polynomial instances.  Values returned by
     * operator()(xValue) of the result of this multiplication will be
     * equal to the sum of the values returned by the operator()(xValue)
     * of arg0 and arg1.
     * 
     * @param arg0 This argument is one of the Polynomial instances to
     * be added.
     * 
     * @param arg1 This argument is one of the Polynomial instances to
     * be added.
     * 
     * @return The return value is the result of the addition.
     */
    template<class Type>
    Polynomial<Type>
    operator+(const Polynomial<Type>& arg0, const Polynomial<Type>& arg1);
  

    /** 
     * This operator subtracts two Polynomial instances.  Values
     * returned by operator()(xValue) of the result of this
     * multiplication will be equal to arg0.operator(xValue) -
     * arg1.operator()(xValue).
     * 
     * @param arg0 This argument is the Polynomial instances from which
     * arg1 is to be subtracted.
     * 
     * @param arg1 This argument is the Polynomial instances to be
     * subtracted from arg0.
     * 
     * @return The return value is the result of the subtraction.
     */
    template<class Type>
    Polynomial<Type>
    operator-(const Polynomial<Type>& arg0, const Polynomial<Type>& arg1);


  } // namespace numeric

} // namespace dlr


/* =============================================================== */
/* Implementation follows.                                         */
/* =============================================================== */



namespace dlr {

  namespace numeric {


    // The default constructor makes a constant polynomial: p(x) = 1.
    template <class Type>
    Polynomial<Type>::
    Polynomial()
      : m_coefficientArray(1)
    {
      m_coefficientArray[0] = Type(1.0);
    }


    // This constructor makes a constant polynomial:
    template <class Type>
    Polynomial<Type>::
    Polynomial(Type coefficient0)
      : m_coefficientArray(1)
    {
      m_coefficientArray[0] = coefficient0;
    }


    // This constructor makes a first order polynomial:
    template <class Type>
    Polynomial<Type>::
    Polynomial(Type coefficient1, Type coefficient0)
      : m_coefficientArray(2)
    {
      m_coefficientArray[0] = coefficient0;
      m_coefficientArray[1] = coefficient1;
    }


    // This constructor makes a first order polynomial:
    template <class Type>
    Polynomial<Type>::
    Polynomial(Type coefficient2, Type coefficient1, Type coefficient0)
      : m_coefficientArray(3)
    {
      m_coefficientArray[0] = coefficient0;
      m_coefficientArray[1] = coefficient1;
      m_coefficientArray[2] = coefficient2;
    }


    // This constructor makes a polynomial of arbitrary order.
    template <class Type>
    Polynomial<Type>::
    Polynomial(const Array1D<Type> coefficients)
      : m_coefficientArray(coefficients.copy())
    {
      // Empty.
    }


    // The copy constructor does a deep copy.
    template <class Type>
    Polynomial<Type>::
    Polynomial(const Polynomial<Type>& other)
      : m_coefficientArray(other.m_coefficientArray.copy())
    {
      // Empty.
    }

    
    // This member function returns the coefficients of the
    // polynomial.
    template <class Type>
    Array1D<Type>
    Polynomial<Type>::
    getCoefficientArray() const
    {
      return m_coefficientArray.copy();
    }
      

    // The assigment operator does a deep copy.
    template <class Type>
    Polynomial<Type>&
    Polynomial<Type>::
    operator=(const Polynomial<Type> other)
    {
      if(&other != this) {
        m_coefficientArray = other.m_coefficientArray.copy();
      }
      return *this;
    }

    
    // This operator evaluates the polynomial.
    template <class Type>
    Type
    Polynomial<Type>::
    operator()(Type xValue) const
    {
      Type result = m_coefficientArray[0];
      if(m_coefficientArray.size() > 1) {
        result += xValue * m_coefficientArray[1];
      }
      Type xToTheN = xValue;
      for(size_t index0 = 2; index0 < m_coefficientArray.size(); ++index0) {
        xToTheN *= xValue;
        result += xToTheN * m_coefficientArray[index0];
      }
      return result;
    }


    // This operator multiplies the polynomial by another polynomial.
    template <class Type>
    Polynomial<Type>&
    Polynomial<Type>::
    operator*=(const Polynomial& other)
    {
      size_t oldOrder = this->getOrder();
      size_t otherOrder = other.getOrder();
      size_t newOrder = oldOrder + otherOrder;
      Array1D<Type> newCoefficientArray(newOrder + 1);
      newCoefficientArray = Type(0);

      // We're going to use oldOrder and otherOrder as bounds for the
      // loops which calculate the new coefficients.  Increment them
      // now to avoid writing "oldOrder + 1" in the loop termination
      // condition.
      ++oldOrder;
      ++otherOrder;
      for(size_t index0 = 0; index0 < oldOrder; ++index0) {
        for(size_t index1 = 0; index1 < otherOrder; ++index1) {
          newCoefficientArray[index0 + index1] +=
            m_coefficientArray[index0] * other.m_coefficientArray[index1];
        }
      }
      m_coefficientArray = newCoefficientArray;
      return *this;
    }


    // This operator adds another polynomial to *this.
    template <class Type>
    Polynomial<Type>&
    Polynomial<Type>::
    operator+=(const Polynomial& other)
    {
      size_t largerSize = m_coefficientArray.size();
      size_t smallerSize = other.m_coefficientArray.size();
      if(smallerSize > largerSize) {
        std::swap(smallerSize, largerSize);
      }
      Array1D<Type> newCoefficientArray(largerSize);

      size_t index0 = 0;
      while(index0 < smallerSize) {
        newCoefficientArray[index0] =
          m_coefficientArray[index0] + (other.m_coefficientArray)[index0];
        ++index0;
      }
      while(index0 < m_coefficientArray.size()) {
        newCoefficientArray[index0] = m_coefficientArray[index0];
        ++index0;
      }
      while(index0 < other.m_coefficientArray.size()) {
        newCoefficientArray[index0] = other.m_coefficientArray[index0];
        ++index0;
      }
      m_coefficientArray = newCoefficientArray;
      return *this;
    }


    // This operator subtracts another polynomial from *this.
    template <class Type>
    Polynomial<Type>&
    Polynomial<Type>::
    operator-=(const Polynomial& other)
    {
      size_t largerSize = m_coefficientArray.size();
      size_t smallerSize = other.m_coefficientArray.size();
      if(smallerSize > largerSize) {
        std::swap(smallerSize, largerSize);
      }
      Array1D<Type> newCoefficientArray(largerSize);

      size_t index0 = 0;
      while(index0 < smallerSize) {
        newCoefficientArray[index0] =
          m_coefficientArray[index0] - (other.m_coefficientArray)[index0];
        ++index0;
      }
      while(index0 < m_coefficientArray.size()) {
        newCoefficientArray[index0] = m_coefficientArray[index0];
        ++index0;
      }
      while(index0 < other.m_coefficientArray.size()) {
        newCoefficientArray[index0] = -(other.m_coefficientArray[index0]);
        ++index0;
      }
      m_coefficientArray = newCoefficientArray;
      return *this;
    }
      


    /* ============ Non-member function definitions ============ */

    // This operator multiplies two Polynomial instances.
    template<class Type>
    Polynomial<Type>
    operator*(const Polynomial<Type>& arg0, const Polynomial<Type>& arg1)
    {
      Polynomial<Type> result = arg0;
      result *= arg1;
      return result;
    }


    // This operator adds two Polynomial instances.
    template<class Type>
    Polynomial<Type>
    operator+(const Polynomial<Type>& arg0, const Polynomial<Type>& arg1)
    {
      Polynomial<Type> result = arg0;
      result += arg1;
      return result;
    }
  

    // This operator subtracts two Polynomial instances.
    template<class Type>
    Polynomial<Type>
    operator-(const Polynomial<Type>& arg0, const Polynomial<Type>& arg1)
    {
      Polynomial<Type> result = arg0;
      result -= arg1;
      return result;
    }
    
    
  } // namespace numeric
  
} // namespace dlr

#endif /* #ifndef _DLRNUMERIC_POLYNOMIAL_H_ */
