/**
***************************************************************************
* @file dlrNumeric/amanatidesWoo3DIterator.h
*
* Header file declaring AmanatidesWoo3DIterator class.
*
* Copyright (C) 2004-2007 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 915 $
* $Date: 2007-05-18 01:27:47 -0400 (Fri, 18 May 2007) $
***************************************************************************
**/

#ifndef _DLR_NUMERIC_AMANATIDESWOO3DITERATOR_H_
#define _DLR_NUMERIC_AMANATIDESWOO3DITERATOR_H_

#include <iostream>
#include <dlrCommon/exception.h>

namespace dlr {

  namespace numeric {
    
    /**
     ** This class provides access to the elements of a data array along
     ** a straight path, and does the actual work of Amanatides and
     ** Woo's fast voxel traversal algorithm.  Typically, an
     ** AmanatidesWoo3DIterator instance will be created by an
     ** AmanatidesWoo3D object in order to access a line of voxels
     ** specified through the AmanatidesWoo3D class interface.  The user
     ** will probably never need to directly construct an
     ** AmanatidesWoo3DIterator.  For more information on the fast voxel
     ** traversal algorithm of Amanatides and Woo, please refer to
     ** [ref].
     **/
    template <class ARRAY3D>
    class AmanatidesWoo3DIterator
      : public std::iterator<std::forward_iterator_tag,
                             typename ARRAY3D::value_type>
    {
    public:
      /** 
       * The class constructor is initialized with all of the internal
       * variables of the voxel traversal algorithm.
       * 
       * @param data This parameter is a reference to the 3D data over
       * which to iterate.
       * @param U This parameter specifies the starting U coordinate
       * (column) in the voxel data.  Its value must lie in the range
       * [0..N), where N is the number of columns in parameter 'data'.
       * @param V This parameter specifies the starting V coordinate
       * (row) in the voxel data.  Its value must lie in the range
       * [0..M), where M is the number of rows in parameter 'data'.
       * @param W This parameter specifies the starting W coordinate
       * (row) in the voxel data.  Its value must lie in the range
       * [0..M), where M is the number of rows in parameter 'data'.
       * @param stepU This parameter specifies the increment by which
       * the U coordinate changes as we move along the direction of the
       * ray.  It must be either 1 or -1.
       * @param stepV This parameter specifies the increment by which
       * the V coordinate changes as we move along the direction of the
       * ray.  It must be either 1 or -1.
       * @param stepW This parameter specifies the increment by which
       * the W coordinate changes as we move along the direction of the
       * ray.  It must be either 1 or -1.
       * @param tMaxU This parameter specifies the value of ray
       * parameter 't' at which the ray passes from the current column
       * into the next column.  Parameter 't' is described in the
       * documentation for class AmanatidesWoo3D.
       * @param tMaxV This parameter specifies the value of ray
       * parameter 't' at which the ray passes from the current row into
       * the next row.  Parameter 't' is described in the documentation
       * for class AmanatidesWoo3D.
       * @param tMaxW This parameter specifies the value of ray
       * parameter 't' at which the ray passes from the current slice into
       * the next slice.  Parameter 't' is described in the documentation
       * for class AmanatidesWoo3D.
       * @param tDeltaU This parameter specifies the increment to ray
       * parameter 't' which moves one exactly one column width to the
       * left or right, where left and right describe the directions of
       * the negative and positive U axis, respectively.  Parameter 't'
       * is described in the documentation for class AmanatidesWoo3D.
       * @param tDeltaV This parameter specifies the increment to ray
       * parameter 't' which moves one exactly one row width up or down,
       * where up and down describe the directions of the negative and
       * positive V axis, respectively.  Parameter 't' is described in
       * the documentation for class AmanatidesWoo3D.
       * @param tDeltaW This parameter specifies the increment to ray
       * parameter 't' which moves one exactly one slice width up or down,
       * where up and down describe the directions of the negative and
       * positive Z axis, respectively.  Parameter 't' is described in
       * the documentation for class AmanatidesWoo3D.
       * @param tStart This parameter specifies the value of ray
       * parameter 't' at the very beginning point of the iteration.
       */
      AmanatidesWoo3DIterator(ARRAY3D& data,
                              int U, int V, int W,
                              int stepU, int stepV, int stepW,
                              double tMaxU, double tMaxV, double tMaxW,
                              double tDeltaU, double tDeltaV, double tDeltaW,
                              double tStart);

      /** 
       * Copy constructor.
       * 
       * @param source This argument specifies the AmanatidesWoo3D
       * instance to be copied.
       */
      AmanatidesWoo3DIterator(const AmanatidesWoo3DIterator& source);

      /** 
       * Destructor.
       */
      ~AmanatidesWoo3DIterator() {};

      /** 
       * This method returns the ray parameter t at which the ray being
       * followed passes into the current voxel.  In other words, the
       * value t such that (rayOrigin + t * rayDirection) is the point
       * of entry into the current voxel.
       * 
       * @return The return value is the value of t at which the ray
       * passes into the current voxel.
       */
      double
      tEntry() {return m_tEntry;}

      /** 
       * This method returns the ray parameter t at which the ray being
       * followed passes out of the current voxel.  In other words, the
       * value t such that (rayOrigin + t * rayDirection) is the point
       * of exit from the current voxel.  Invoking this method carries a
       * computational cost of 1 double precision float comparison.
       * 
       * @return The return value is the value of t at which the ray
       * passes out of the current voxel.
       */
      double
      tExit() {return std::min(m_tMaxU, std::min(m_tMaxV, m_tMaxW));}

      /** 
       * This method returns the U coordinate of the current voxel.  The
       * return value is int rather than size_t so that negative (out of
       * bounds) coordinates can be returned.
       * 
       * @return The return value is the U coordinate of the current
       * voxel.
       */
      int
      U() {return m_U;}

      /** 
       * This method returns the V coordinate of the current voxel.  The
       * return value is int rather than size_t so that negative (out of
       * bounds) coordinates can be returned.
       * 
       * @return The return value is the V coordinate of the current
       * voxel.
       */
      int
      V() {return m_V;}

      /** 
       * This method returns the W coordinate of the current voxel.  The
       * return value is int rather than size_t so that negative (out of
       * bounds) coordinates can be returned.
       * 
       * @return The return value is the W coordinate of the current
       * voxel.
       */
      int
      W() {return m_W;}

      /** 
       * This operator returns a reference to the Array3D element at the
       * current voxel.  With each increment of the
       * AmanatidesWoo3DIterator instance, this operator will return a
       * reference to the next voxel along the ray.
       * 
       * @return The return value is a reference the the relevant
       * Array3D element.
       */
      inline typename ARRAY3D::value_type& // element_type?
      operator*();

      /** 
       * This operator returns a pointer to the Array3D element at the
       * current voxel.  With each increment of the
       * AmanatidesWoo3DIterator instance, this operator will return a
       * pointer to the next voxel along the ray.
       * 
       * @return The return value is a pointer the the relevant
       * Array3D element.
       */
      inline typename ARRAY3D::value_type* // element_type?
      operator->();

      /** 
       * The pre-increment operator increments the iterator so that it
       * points to the next voxel along the path.
       * 
       * @return The return value is a reference to *this.
       */
      inline AmanatidesWoo3DIterator&
      operator++();	             // prefix

      /** 
       * The post-increment operator increments the iterator so that it
       * points to the next voxel along the path.  It differs from the
       * pre-increment operator in its return value.  Traditionally,
       * post-increment is a little slower than pre-increment.
       * 
       * @param dummy This parameter is a dummy which indicates to the
       * compiler that this operation is post-increment (rather than
       * pre-increment).
       *
       * @return The return value is a copy of *this which was generated
       * before the increment.
       */
      inline AmanatidesWoo3DIterator
      operator++(int dummy);                 // postfix

      /** 
       * This is the assignment operator.  It copies the value of its
       * argument into *this.
       * 
       * @param source This argument specifies the
       * AmanatidesWoo3DIterator instance to be copied.
       * @return The return value is a reference to *this.
       */
      AmanatidesWoo3DIterator&
      operator=(const AmanatidesWoo3DIterator& source);

      /** 
       * The equality operator returns true if both the argument and
       * *this currently reference a valid voxel, or if both the
       * argument and *this currently reference an invalid voxel.  In all
       * other cases the return is false.
       *
       * NOTE: This behavior is not exactly what you'd expect for an
       * equality operator.  references the same voxel as the argument.
       * 
       * @param other This argument is a second AmanatidesWoo3DIterator
       * instance that is to be compared with *this.
       *
       * @return The return value is true if *this and other both
       * reference in-bounds voxels, or if *this and other both
       * reference out-of-bounds voxels, false otherwise.
       */
      inline bool
      operator==(const AmanatidesWoo3DIterator& other);

      /** 
       * The equality operator returns false if both the argument and
       * *this currently reference a valid voxel, or if both the
       * argument and *this currently reference an invalid voxel.  In all
       * other cases the return is true.
       *
       * NOTE: This behavior is not exactly what you'd expect for an
       * equality operator.  references the same voxel as the argument.
       * 
       * @param other This argument is a second AmanatidesWoo3DIterator
       * instance that is to be compared with *this.
       *
       * @return The return value is false if *this and other both
       * reference in-bounds voxels, or if *this and other both
       * reference out-of-bounds voxels, true otherwise.
       */
      inline bool
      operator!=(const AmanatidesWoo3DIterator& other);

    private:
      ARRAY3D& m_data;
      bool m_inBounds;
      int m_stepU;
      int m_stepV;
      int m_stepW;
      double m_tDeltaU;
      double m_tDeltaV;
      double m_tDeltaW;
      double m_tEntry;
      double m_tMaxU;
      double m_tMaxV;
      double m_tMaxW;
      int m_U;
      int m_uLimit;
      int m_V;
      int m_vLimit;
      int m_W;
      int m_wLimit;
    };

  } // namespace numeric

} // namespace dlr


/* ======= Declarations to maintain compatibility with legacy code. ======= */

namespace dlr {

  using numeric::AmanatidesWoo3DIterator;

} // namespace dlr


/* ========================================================= */
/* Implementation follows                                    */
/* ========================================================= */

namespace dlr {

  namespace numeric {
    
    // The class constructor is initialized with all of the internal
    // variables of the voxel traversal algorithm, plus the starting
    // value of the ray parameter.
    template<class ARRAY3D>
    AmanatidesWoo3DIterator<ARRAY3D>::
    AmanatidesWoo3DIterator(ARRAY3D& data,
                            int U, int V, int W,
                            int stepU, int stepV, int stepW,
                            double tMaxU, double tMaxV, double tMaxW,
                            double tDeltaU, double tDeltaV, double tDeltaW,
                            double tStart)
      : m_data(data),
        m_inBounds(true),
        m_stepU(stepU),
        m_stepV(stepV),
        m_stepW(stepW),
        m_tDeltaU(tDeltaU),
        m_tDeltaV(tDeltaV),
        m_tDeltaW(tDeltaW),
        m_tEntry(tStart),
        m_tMaxU(tMaxU),
        m_tMaxV(tMaxV),
        m_tMaxW(tMaxW),
        m_U(U),
        m_uLimit(stepU > 0 ? static_cast<int>(data.shape()[2]) : -1),
        m_V(V),
        m_vLimit(stepV > 0 ? static_cast<int>(data.shape()[1]) : -1),
        m_W(W),
        m_wLimit(stepW > 0 ? static_cast<int>(data.shape()[0]) : -1)
    {
      if((m_U < 0)
         || (m_V < 0)
         || (m_W < 0)
         || (m_U >= static_cast<int>(data.shape()[2]))
         || (m_V >= static_cast<int>(data.shape()[1]))
         || (m_W >= static_cast<int>(data.shape()[0]))) {
        m_inBounds = false;      
      }
    }

    // Copy constructor.
    template<class ARRAY3D>
    AmanatidesWoo3DIterator<ARRAY3D>::
    AmanatidesWoo3DIterator(const AmanatidesWoo3DIterator& source)
      : m_data(source.m_data),
        m_inBounds(source.m_inBounds),
        m_stepU(source.m_stepU),
        m_stepV(source.m_stepV),
        m_stepW(source.m_stepW),
        m_tDeltaU(source.m_tDeltaU),
        m_tDeltaV(source.m_tDeltaV),
        m_tDeltaW(source.m_tDeltaW),
        m_tEntry(source.m_tEntry),
        m_tMaxU(source.m_tMaxU),
        m_tMaxV(source.m_tMaxV),
        m_tMaxW(source.m_tMaxW),
        m_U(source.m_U),
        m_uLimit(source.m_uLimit),
        m_V(source.m_V),
        m_vLimit(source.m_vLimit),
        m_W(source.m_W),
        m_wLimit(source.m_wLimit)
    {
      // Empty
    }

    // This operator returns a reference to the Array3D element at the
    // current voxel.
    template<class ARRAY3D>
    inline typename ARRAY3D::value_type& // element_type?
    AmanatidesWoo3DIterator<ARRAY3D>::
    operator*()
    {
      return m_data(m_W, m_V, m_U);
    }

    // This operator returns a pointer to the Array3D element at the
    // current voxel.
    template<class ARRAY3D>
    inline typename ARRAY3D::value_type* // element_type?
    AmanatidesWoo3DIterator<ARRAY3D>::
    operator->()
    {
      return &(this->operator*());
    }

    // The pre-increment operator increments the iterator so that it
    // points to the next voxel along the path.
    template<class ARRAY3D>
    inline AmanatidesWoo3DIterator<ARRAY3D>&
    AmanatidesWoo3DIterator<ARRAY3D>::
    operator++()
    {
      if(m_tMaxU < m_tMaxV) {
        if(m_tMaxW < m_tMaxU) {
          m_tEntry = m_tMaxW;
          m_W += m_stepW;
          m_tMaxW += m_tDeltaW;
          if(m_W == m_wLimit) {
            m_inBounds = false;
          }
        } else {
          m_tEntry = m_tMaxU;
          m_U += m_stepU;
          m_tMaxU += m_tDeltaU;
          if(m_U == m_uLimit) {
            m_inBounds = false;
          }
        }
      } else {
        if(m_tMaxW < m_tMaxV) {
          m_tEntry = m_tMaxW;
          m_W += m_stepW;
          m_tMaxW += m_tDeltaW;
          if(m_W == m_wLimit) {
            m_inBounds = false;
          }
        } else {
          m_tEntry = m_tMaxV;
          m_V += m_stepV;
          m_tMaxV += m_tDeltaV;
          if(m_V == m_vLimit) {
            m_inBounds = false;
          }
        }
      }
      return *this;
    }

    // The post-increment operator increments the iterator so that it
    // points to the next voxel along the path.  It differs from the
    // pre-increment operator in its return value.
    template<class ARRAY3D>
    inline AmanatidesWoo3DIterator<ARRAY3D>
    AmanatidesWoo3DIterator<ARRAY3D>::
    operator++(int)
    {
      AmanatidesWoo3DIterator<ARRAY3D> thisCopy(*this);
      ++this;
      return thisCopy;
    }

    // This is the assignment operator.  It copies the value of its
    // argument into *this.
    template<class ARRAY3D>
    AmanatidesWoo3DIterator<ARRAY3D>&
    AmanatidesWoo3DIterator<ARRAY3D>::
    operator=(const AmanatidesWoo3DIterator& source)
    {
      m_data = source.m_data;
      m_inBounds = source.m_inBounds;
      m_stepU = source.m_stepU;
      m_stepV = source.m_stepV;
      m_stepW = source.m_stepW;
      m_tDeltaU = source.m_tDeltaU;
      m_tDeltaV = source.m_tDeltaV;
      m_tDeltaW = source.m_tDeltaW;
      m_tEntry = source.m_tEntry;
      m_tMaxU = source.m_tMaxU;
      m_tMaxV = source.m_tMaxV;
      m_tMaxW = source.m_tMaxW;
      m_U = source.m_U;
      m_uLimit = source.m_uLimit;
      m_V = source.m_V;
      m_vLimit = source.m_vLimit;
      m_W = source.m_W;
      m_wLimit = source.m_wLimit;
    }

    // The equality operator returns true if *this currently
    // references the same voxel as the argument.
    template<class ARRAY3D>
    inline bool
    AmanatidesWoo3DIterator<ARRAY3D>::
    operator==(const AmanatidesWoo3DIterator& other)
    {
      // Return true if both refer to valid voxels or if both refer to
      // invalid voxels.
      return (m_inBounds == other.m_inBounds);
    }

    // The inequality operator returns true if *this currently
    // references the a different voxel than the argument.
    template<class ARRAY3D>
    inline bool
    AmanatidesWoo3DIterator<ARRAY3D>::
    operator!=(const AmanatidesWoo3DIterator& other)
    {
      return !(this->operator==(other));
    }
  
  } // namespace numeric

} // namespace dlr

#endif // #ifndef _DLR_NUMERIC_AMANATIDESWOO3DITERATOR_H_
