/**
***************************************************************************
* @file linearAlgebraTest.cpp
* Source file defining linearAlgebraTest class.
*
* Copyright (C) 2005 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 942 $
* $Date: 2007-07-09 16:29:54 -0400 (Mon, 09 Jul 2007) $
***************************************************************************
**/

#include <algorithm>
#include <dlrCommon/functional.h>
#include <dlrLinearAlgebra/linearAlgebra.h>

#include <dlrTest/testFixture.h>

namespace dlr {

  class LinearAlgebraTest : public TestFixture<LinearAlgebraTest> {

  public:

    LinearAlgebraTest();
    ~LinearAlgebraTest() {}

    void setUp(const std::string& testName) {}
    void tearDown(const std::string& testName) {}

    void testEigenvaluesSymmetric();
    void testEigenvectorsSymmetric();
    void testInverse();
    void testLinearLeastSquares();
    void testLinearSolveInPlace();
    void testSingularValueDecomposition();
    void testSingularValues();

  private:

    bool
    approximatelyEqual(const Array1D<double>& array0,
                       const Array1D<double>& array1);

    bool
    approximatelyEqual(const Array2D<double>& array0,
                       const Array2D<double>& array1);


    static const size_t numberOfTestMatrixSets = 2;
    
    std::vector< Array2D<double> > m_aMatrices;
    std::vector< Array1D<double> > m_bVectors;
    std::vector< Array2D<double> > m_inverseMatrices;
    std::vector< Array1D<double> > m_squareBVectors;
    std::vector< Array2D<double> > m_squareMatrices;
    std::vector< Array1D<double> > m_squareXVectors;
    std::vector< Array1D<double> > m_symmetricEigenvalues;
    std::vector< Array2D<double> > m_symmetricEigenvectors;
    std::vector< Array2D<double> > m_symmetricMatrices;
    std::vector< Array1D<double> > m_sVectors;
    std::vector< Array2D<double> > m_testMatrices;
    std::vector< Array2D<double> > m_uMatrices;
    std::vector< Array2D<double> > m_vtMatrices;
    std::vector< Array1D<double> > m_xVectors;
    
  }; // class LinearAlgebraTest


  /* ============== Member Function Definititions ============== */

  LinearAlgebraTest::
  LinearAlgebraTest()
    : TestFixture<LinearAlgebraTest>("LinearAlgebraTest"),
      m_aMatrices(LinearAlgebraTest::numberOfTestMatrixSets),
      m_bVectors(LinearAlgebraTest::numberOfTestMatrixSets),
      m_inverseMatrices(LinearAlgebraTest::numberOfTestMatrixSets),
      m_squareBVectors(LinearAlgebraTest::numberOfTestMatrixSets),
      m_squareMatrices(LinearAlgebraTest::numberOfTestMatrixSets),
      m_squareXVectors(LinearAlgebraTest::numberOfTestMatrixSets),
      m_symmetricEigenvalues(LinearAlgebraTest::numberOfTestMatrixSets),
      m_symmetricEigenvectors(LinearAlgebraTest::numberOfTestMatrixSets),
      m_symmetricMatrices(LinearAlgebraTest::numberOfTestMatrixSets),
      m_sVectors(LinearAlgebraTest::numberOfTestMatrixSets),
      m_testMatrices(LinearAlgebraTest::numberOfTestMatrixSets),
      m_uMatrices(LinearAlgebraTest::numberOfTestMatrixSets),
      m_vtMatrices(LinearAlgebraTest::numberOfTestMatrixSets),
      m_xVectors(LinearAlgebraTest::numberOfTestMatrixSets)      
  {
    // Register all tests.
    DLR_TEST_REGISTER_MEMBER(testEigenvaluesSymmetric);
    DLR_TEST_REGISTER_MEMBER(testEigenvectorsSymmetric);
    DLR_TEST_REGISTER_MEMBER(testInverse);
    DLR_TEST_REGISTER_MEMBER(testLinearLeastSquares);
    DLR_TEST_REGISTER_MEMBER(testLinearSolveInPlace);
    DLR_TEST_REGISTER_MEMBER(testSingularValueDecomposition);
    DLR_TEST_REGISTER_MEMBER(testSingularValues);

    // Set up test matrices.
    if(LinearAlgebraTest::numberOfTestMatrixSets != 2) {
      DLR_THROW(LogicException, "LinearAlgebraTest::LinearAlgebraTest()",
                "Incorrect number of test sets.");
    }

    m_aMatrices[0] = Array2D<double>(
      "[[1.0, 2.0, 3.0],"
      " [4.0, 5.0, 6.0],"
      " [10.0, 3.0, 1.0],"
      " [1.0, 5.0, 9.0],"
      " [6.0, 6.0, 6.0]]");
      
    m_aMatrices[1] = Array2D<double>(
      "[[1.0, 2.0, 3.0, 4.0],"
      " [4.0, 2.0, 6.0, 1.0],"
      " [10.0, 3.0, 1.0, 1.0]]");

    
    m_bVectors[0] = Array1D<double>(
      "[14.0, 32.0, 19.0, 38.0, 36.0]");

    m_bVectors[1] = Array1D<double>(
      "[23.5, 20.5, 21.0]");

    
    m_inverseMatrices[0] = Array2D<double>(
      "[[-1.47619048,  0.61904762, -0.14285714],"
      " [ 1.28571429, -0.57142857,  0.28571429],"
      " [-0.14285714,  0.28571429, -0.14285714]]");

    m_inverseMatrices[1] = Array2D<double>(
      "[[ 0.06666667,  0.33333333, -0.06666667, -2.73333333],"
      " [ 1.26666667, -0.66666667, -0.26666667, -1.93333333],"
      " [-1.53333333,  0.33333333,  0.53333333,  4.86666667],"
      " [ 0.        ,  0.        ,  0.        ,  1.        ]]");


    m_squareBVectors[0] = Array1D<double>(
      "[4., 2., 7.]");

    m_squareBVectors[1] = Array1D<double>(
      "[1., 2., 7., 4.]");


    m_squareMatrices[0] = Array2D<double>(
      "[[ 0.,  1.,  2.],"
      " [ 3.,  4.,  5.],"
      " [ 6.,  7.,  1.]]");
      
    m_squareMatrices[1] = Array2D<double>(
      "[[ 4.,  3.,  2.,  7.],"
      " [ 4.,  1.,  1.,  8.],"
      " [ 9.,  8.,  7.,  6.],"
      " [ 0.,  0.,  0.,  1.]]");


    m_squareXVectors[0] = Array1D<double>(
      "[-5.66666667, 6., -1.]");


    m_squareXVectors[1] = Array1D<double>(
      "[-10.66666667, -9.66666667, 22.33333333, 4.]");

    
    m_symmetricEigenvalues[0] = Array1D<double>(
      "[24.06253512, 0.5580362, 0.18491359, -0.80548492]");

    m_symmetricEigenvalues[1] = Array1D<double>(
      "[22.25237164, 10.12484443, -2.53641946, -12.84079662]");


    m_symmetricEigenvectors[0] = Array2D<double>(
      "[[ -0.22593827, 0.56047033, -0.32976505, -0.72531367],"
      " [ -0.44322186, -0.77633831, -0.3153256, -0.31846973],"
      " [ -0.57278788, 0.05844028, 0.805111, -0.14246073],"
      " [ -0.6514755, 0.28241204, -0.37897369, 0.59346614]]");

    m_symmetricEigenvectors[1] = Array2D<double>(
      "[[ 0.57388773, -0.40927089, 0.65613403, -0.26951503],"
      " [ 0.65948776, -0.25973728, -0.69857373, 0.09801622],"
      " [ -0.48540238, -0.84196315, -0.16831305, -0.16478259],"
      " [ 0.01064409, -0.2369218, 0.23055064, 0.94371668]]");

    
    m_symmetricMatrices[0] = Array2D<double>(
      "[[  1.,   2.,   3.,   4.],"
      " [  2.,   5.,   6.,   7.],"
      " [  3.,   6.,   8.,   9.],"
      " [  4.,   7.,   9.,  10.]]");

    m_symmetricMatrices[1] = Array2D<double>(
      "[[  7.,  11.,  -3.,   4.],"
      " [ 11.,   9.,  -5.,   0.],"
      " [ -3.,  -5.,  12.,   4.],"
      " [  4.,   0.,   4., -11.]]");
    

    m_sVectors[0] = Array1D<double>(
      "[ 18.52607955,   6.51255861,   0.60906229]");

    m_sVectors[1] = Array1D<double>(
      "[ 18.67643567,   6.03617265,   0.86912045]");

    
    m_testMatrices[0] = Array2D<double>(
      "[[  0.,   1.,   2.],"
      " [  3.,   4.,   5.],"
      " [  6.,   7.,   8.],"
      " [  9.,  10.,  1.]]");

    m_testMatrices[1] = Array2D<double>(
      "[[  0.,   1.,   2.,   3.],"
      " [  4.,   5.,   6.,   7.],"
      " [  8.,   9.,  10.,  1.]]");


    m_uMatrices[0] = Array2D<double>(
      "[[-0.08198544, -0.23824706,  0.877411  ],"
      " [-0.35697841, -0.38359746,  0.24238963],"
      " [-0.63197137, -0.52894786, -0.39263173],"
      " [-0.68297656,  0.718544  ,  0.13129179]]");

    m_uMatrices[1] = Array2D<double>(
      "[[-0.14130401,  0.42003822, -0.89643799],"
      " [-0.55423802,  0.71674227,  0.42320295],"
      " [-0.82027642, -0.55664029, -0.13152259]]");

    
    m_vtMatrices[0] = Array2D<double>(
      "[[-0.5942732 , -0.68894578, -0.41496154],"
      " [ 0.32896694,  0.26259542, -0.90709669],"
      " [-0.73390743,  0.67557188, -0.07058696]]");

    m_vtMatrices[1] = Array2D<double>(
      "[[-0.47006632, -0.5512284 , -0.63239049, -0.27434863],"
      " [-0.26277466, -0.16666405, -0.07055345,  0.94773139],"
      " [ 0.73710275,  0.04127557, -0.65455161,  0.16290502]]");

    
    m_xVectors[0] = Array1D<double>(
      "[1.0, 2.0, 3.0]");

    m_xVectors[1] = Array1D<double>(
      "[1.070794, 1.72098834, 1.52915047, 3.59994448]");

  }


  void
  LinearAlgebraTest::
  testEigenvaluesSymmetric()
  {
    for(size_t index0 = 0;
        index0 < LinearAlgebraTest::numberOfTestMatrixSets;
        ++index0) {
      // Do the computation.
      Array2D<double> aMatrix = m_symmetricMatrices[index0].copy();
      Array1D<double> eigenvalueArray = eigenvaluesSymmetric(aMatrix);

      // Check that input is unchanged.
      DLR_TEST_ASSERT(
        this->approximatelyEqual(aMatrix, m_symmetricMatrices[index0]));

      // Check that results are correct.
      DLR_TEST_ASSERT(
        this->approximatelyEqual(
          eigenvalueArray, m_symmetricEigenvalues[index0]));
    }
  }

  
  void
  LinearAlgebraTest::
  testEigenvectorsSymmetric()
  {
    for(size_t index0 = 0;
        index0 < LinearAlgebraTest::numberOfTestMatrixSets;
        ++index0) {
      // Do the computation.
      Array2D<double> aMatrix = m_symmetricMatrices[index0].copy();
      Array1D<double> eigenvalueArray;
      Array2D<double> eigenvectorArray;
      eigenvectorsSymmetric(aMatrix, eigenvalueArray, eigenvectorArray);

      // Check that input is unchanged.
      DLR_TEST_ASSERT(
        this->approximatelyEqual(aMatrix, m_symmetricMatrices[index0]));

      // Check that results are correct.
      DLR_TEST_ASSERT(
        this->approximatelyEqual(
          eigenvalueArray, m_symmetricEigenvalues[index0]));
      DLR_TEST_ASSERT(
        this->approximatelyEqual(
          eigenvectorArray, m_symmetricEigenvectors[index0]));
    }
  }


  void
  LinearAlgebraTest::
  testInverse()
  {
    for(size_t index0 = 0;
        index0 < LinearAlgebraTest::numberOfTestMatrixSets;
        ++index0) {
      // Do the inverse.
      Array2D<double> squareMatrix = m_squareMatrices[index0].copy();
      Array2D<double> inverseMatrix = inverse(squareMatrix);

      // Check that input is unchanged.
      DLR_TEST_ASSERT(
        this->approximatelyEqual(squareMatrix, m_squareMatrices[index0]));

      // Check that results are correct.
      DLR_TEST_ASSERT(
        this->approximatelyEqual(inverseMatrix, m_inverseMatrices[index0]));
    }
  }


  void
  LinearAlgebraTest::
  testLinearLeastSquares()
  {
    for(size_t index0 = 0;
        index0 < LinearAlgebraTest::numberOfTestMatrixSets;
        ++index0) {
      // Compute the solution.
      Array2D<double> aMatrix = m_aMatrices[index0].copy();
      Array1D<double> bVector = m_bVectors[index0].copy();
      Array1D<double> xVector = linearLeastSquares(aMatrix, bVector);
      
      // Check that input is unchanged.
      DLR_TEST_ASSERT(
        this->approximatelyEqual(aMatrix, m_aMatrices[index0]));
      DLR_TEST_ASSERT(
        this->approximatelyEqual(bVector, m_bVectors[index0]));

      // Check that results are correct.
      DLR_TEST_ASSERT(
        this->approximatelyEqual(xVector, m_xVectors[index0]));
    }
  }
  

  void
  LinearAlgebraTest::
  testLinearSolveInPlace()
  {
    for(size_t index0 = 0;
        index0 < LinearAlgebraTest::numberOfTestMatrixSets;
        ++index0) {
      // Compute the solution.
      Array2D<double> aMatrix = m_squareMatrices[index0].copy();
      Array1D<double> bVector = m_squareBVectors[index0].copy();
      linearSolveInPlace(aMatrix, bVector);
      
      // Check that results are correct.
      DLR_TEST_ASSERT(
        this->approximatelyEqual(bVector, m_squareXVectors[index0]));
    }
  }
  

  void
  LinearAlgebraTest::
  testSingularValueDecomposition()
  {
    Array2D<double> aMatrix;
    Array2D<double> uMatrix;
    Array1D<double> sVector;
    Array2D<double> vtMatrix;

    for(size_t index0 = 0;
        index0 < LinearAlgebraTest::numberOfTestMatrixSets;
        ++index0) {
      // Do the SVD.
      aMatrix = m_testMatrices[index0].copy();
      singularValueDecomposition(aMatrix, uMatrix, sVector, vtMatrix);

      // Check that input is unchanged.
      DLR_TEST_ASSERT(
        this->approximatelyEqual(aMatrix, m_testMatrices[index0]));

      // Check that results are correct.
      DLR_TEST_ASSERT(this->approximatelyEqual(uMatrix, m_uMatrices[index0]));
      DLR_TEST_ASSERT(this->approximatelyEqual(sVector, m_sVectors[index0]));
      DLR_TEST_ASSERT(
        this->approximatelyEqual(vtMatrix, m_vtMatrices[index0]));
    }
  }


  void
  LinearAlgebraTest::
  testSingularValues()
  {
    for(size_t index0 = 0;
        index0 < LinearAlgebraTest::numberOfTestMatrixSets;
        ++index0) {
      // Do the SVD.
      Array2D<double> aMatrix = m_testMatrices[index0].copy();
      Array1D<double> sVector = singularValues(aMatrix);

      // Check that input is unchanged.
      DLR_TEST_ASSERT(
        this->approximatelyEqual(aMatrix, m_testMatrices[index0]));

      // Check that results are correct.
      DLR_TEST_ASSERT(this->approximatelyEqual(sVector, m_sVectors[index0]));
    }
  }


  bool
  LinearAlgebraTest::
  approximatelyEqual(const Array1D<double>& array0,
                     const Array1D<double>& array1)
  {
    if(array0.size() != array1.size()) {
      return false;
    }
    return std::equal(array0.begin(), array0.end(), array1.begin(),
                      ApproximatelyEqualFunctor<double>(1.0E-5));
  }

  
  bool
  LinearAlgebraTest::
  approximatelyEqual(const Array2D<double>& array0,
                     const Array2D<double>& array1)
  {
    if(array0.rows() != array1.rows()) {
      return false;
    }
    if(array0.columns() != array1.columns()) {
      return false;
    }
    return std::equal(array0.begin(), array0.end(), array1.begin(),
                      ApproximatelyEqualFunctor<double>(1.0E-5));
  }

  
} // namespace dlr


#if 0

int main(int argc, char** argv)
{
  dlr::LinearAlgebraTest currentTest;
  bool result = currentTest.run();
  return (result ? 0 : 1);
}

#else

namespace {

  dlr::LinearAlgebraTest currentTest;

}

#endif
