/**
***************************************************************************
* @file dlrLinearAlgebra/linearAlgebra.h
*
* Header file declaring linear algebra functions.  Many of these depend
* on the LAPACK and BLAS libraries.
*
* Copyright (C) 2001-2004 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 942 $
* $Date: 2007-07-09 16:29:54 -0400 (Mon, 09 Jul 2007) $
***************************************************************************
**/

#ifndef _DLR_LINEARALGEBRA_LINEARALGEBRA_H_
#define _DLR_LINEARALGEBRA_LINEARALGEBRA_H_

#include <dlrCommon/types.h>
#include <dlrNumeric/array1D.h>
#include <dlrNumeric/array2D.h>

namespace dlr {

  namespace linearAlgebra {

  /** 
   * This function computes the determinant of a square
   * Array2D<Float64> instance.
   * 
   * @param A This argument represents the matrix from which to
   * compute the determinant.
   * @return The return value is a Float64 representing the determinant
   * of the input argument.
   */
  Float64
  determinant(const Array2D<Float64>& A);


  /** 
   * This function computes the eigenvalues of a symmetric real matrix.
   * 
   * @param inputArray This argument is and Array2D<Float64> instance
   * representing a symmetric matrix.  It must be square and have
   * non-zero size.  Only the data in the upper triangular portion of
   * the array (including the diagonal) will be examined.  The
   * remaining elements are assumed to be symmetric.
   * 
   * @return The return value is an Array1D<Float64> instance
   * containing the eigenvalues, sorted into descending order.
   */
  Array1D<Float64>
  eigenvaluesSymmetric(const Array2D<Float64>& inputArray);


  /** 
   * This function computes the eigenvalues and eigenvectors of a
   * symmetric real matrix.
   * 
   * @param inputArray This argument is and Array2D<Float64> instance
   * representing a symmetric matrix.  It must be square and have
   * non-zero size.  Only the data in the upper triangular portion of
   * the array (including the diagonal) will be examined.  The
   * remaining elements are assumed to be symmetric.
   * 
   * @param eigenvalues This argument is used to return an
   * Array1D<Float64> instance containing the eigenvalues, sorted into
   * descending order.
   * 
   * @param eigenvectors This argument is used to return the
   * eigenvectors.  On return, the first column contains the
   * eigenvector corresponding to the first eigenvalue, the second
   * column contains the eigenvector corresponding to the second
   * eigenvalue, and so on.
   */
  void
  eigenvectorsSymmetric(const Array2D<Float64>& inputArray,
                        Array1D<Float64>& eigenvalues,
                        Array2D<Float64>& eigenvectors);

  
  /** 
   * This function accepts a square Array2D<Float64> instance and
   * returns an Array2D<Float64> instance such that the matrix product
   * of the two is equal to the identity matrix.  It is an error if
   * the argument is not invertible.
   * 
   * @param A This argument is the matrix to be inverted.
   * @return The return value is the inverse of argument A.
   */
  Array2D<Float64>
  inverse(const Array2D<Float64>& A);

  
  /** 
   * This function computes the best linear fit between the two input
   * arrays.  That is, it solves for scalars a and b which minimize
   * the quantity
   *
   *   e = |(a * array0 + b) - array1|^2,
   *
   * where array0 and array1 are the two input arrays.  Put another
   * way, this means we're solving (in the least squares sense) for
   * the variables a and b which most nearly satisfy the following
   * equation:
   *
   *   a * array0 + b = array1
   * 
   * @param array0 This argument is the first input array.
   * @param array1 This argument is the second input array.
   * @return The return value is a pair of Float64s in which the first
   * element is the variable a and the second is the variable b.
   */
  std::pair<Float64, Float64>
  linearFit(const Array1D<Float64>& array0,
            const Array1D<Float64>& array1);

  
  /** 
   * This function solves the system of equations A*x = b, where A and
   * b are known Array2D<Float64> instances.  The contents of the
   * arguments are not modified.  If the solution fails, a
   * ValueException will be generated.
   * 
   * @param A This argument specifies the A matrix in the system "Ax =
   * b," and must either be square or have more rows than columns.
   *
   * @param b This argument specifies the b vector in the system "Ax =
   * b."  It must have the same number of elements as argument A has
   * rows.
   *
   * @return The return value is the vector x which most nearly
   * satisfies the equation.  "Nearly" is defined in the least-squares
   * sense.
   */
  Array1D<Float64>
  linearLeastSquares(const Array2D<Float64>& A, const Array1D<Float64>& b);


  /** 
   * This function solves the system of equations A*x = b, where A is
   * a known matrix, and b is a known vector.  The contents of both
   * arguments are modified as part of the process.  If the solution
   * fails, a ValueException will be generated.
   * 
   * @param A This argument specifies the A matrix in the system "Ax =
   * b," and must be square.
   *
   * @param b This argument specifies the b vector in the system "Ax =
   * b."  It must have the same size as x, and will be replaced with
   * the recovered value of x.
   */
  void
  linearSolveInPlace(Array2D<Float64>& A, Array1D<Float64>& b);


  /** 
   * This function is identical to linearSolveInPlace(Array2D<Float64>,
   * Array1D<Float64>&), except that b (and therefore x) is not
   * constrained to be a vector.
   * 
   * @param A This argument specifies the A matrix in the system "Ax =
   * b," and must be square.  In the current implementation, it will
   * be replaced the LU decomposition of A, however this behavior may
   * change in the future.
   *
   * @param b This argument specifies the b matrix in the system "Ax =
   * b."  It must have the same size as x, and will be replaced with
   * the recovered value of x.
   */
  void
  linearSolveInPlace(Array2D<Float64>& A, Array2D<Float64>& b);
  

  /** 
   * This function solves the system of equations A*x = b, where A is
   * a known tridiagonal matrix and b is a known vector.  The contents
   * of the arguments are not modified.  If the solution fails, a
   * ValueException will be generated.
   * 
   * @param subDiagonal This argument specifies the lower diagonal
   * of the A matrix in the system "A*x = b."
   *
   * @param centerDiagonal This argument specifies the center diagonal
   * of the A matrix in the system "A*x = b."  It's length must be 1
   * greater than the length of argument subDiagonal.
   *
   * @param superDiagonal This argument specifies the upper diagonal
   * of the A matrix in the system "A*x = b."  It's length must be the
   * same as the length of argument subDiagonal.
   *
   * @param b This argument specifies the b vector in the system "Ax =
   * b."  It must have the same length as argument centerDiagonal.
   *
   * @return The return value is the vector x which most nearly
   * satisfies the equation.  "Nearly" is defined in the least-squares
   * sense.
   */
  Array1D<Float64>
  linearSolveTridiagonal(const Array1D<Float64>& subDiagonal,
                         const Array1D<Float64>& centerDiagonal,
                         const Array1D<Float64>& superDiagonal,
                         const Array1D<Float64>& bVector);
  
    
  /** 
   * This function accepts an Array2D<Float64> instance having at least
   * as many rows as columns, and returns the Moore-Penrose
   * pseudoinverse.  That is, for an input matrix A, it returns
   * inverse(matrixMultiply(transpose(A), A)).  It is an error if the
   * rank of A is less than A.columns().
   * 
   * @param A This argument is the matrix to be psuedoinverted.
   *
   * @return The return value is the pseudoinverse of argument A.
   */
  Array2D<Float64>
  pseudoinverse(const Array2D<Float64>& A);


  /** 
   * This function computes the singular value decomposition of a
   * matrix.
   * 
   * @param inputArray This argument is the matrix to be decomposed.
   * 
   * @param uArray This argument will be filled in with an orthonormal
   * basis spanning the range of the input matrix, one basis vector
   * per column.
   * 
   * @param sigmaArray This argument will be filled in with the
   * singular values of the matrix, in descending order.
   * 
   * @param vTransposeArray This argument will be filled in with an
   * orthonormal basis spanning the domain of the input matrix, one
   * basis vector per row.
   */
  void
  singularValueDecomposition(const Array2D<Float64>& inputArray,
                             Array2D<Float64>& uArray,
                             Array1D<Float64>& sigmaArray,
                             Array2D<Float64>& vTransposeArray);

  
  /** 
   * This function computes the singular values a matrix without
   * computing the associated U and V matrices.
   * 
   * @param inputArray This argument is the matrix from which to compute
   * the singular values.
   * 
   * @return The return value is an Array1D of singular values in
   * descending order.
   */
  Array1D<Float64>
  singularValues(const Array2D<Float64>& inputArray);
  
  } // namespace linearAlgebra
  
} // namespace dlr


/* ======= Declarations to maintain compatibility with legacy code. ======= */

namespace dlr {

  using linearAlgebra::determinant;
  using linearAlgebra::eigenvaluesSymmetric;
  using linearAlgebra::eigenvectorsSymmetric;
  using linearAlgebra::inverse;
  using linearAlgebra::linearFit;
  using linearAlgebra::linearLeastSquares;
  using linearAlgebra::linearSolveInPlace;
  using linearAlgebra::linearSolveTridiagonal;
  using linearAlgebra::pseudoinverse;
  using linearAlgebra::singularValueDecomposition;
  using linearAlgebra::singularValues;
  
} // namespace dlr

#endif // #ifndef _DLR_LINEARALGEBRA_LINEARALGEBRA_H_
