/**
***************************************************************************
* @file dlrGeometry/plane3D.cpp
*
* Source file defining the Plane3D class.
*
* Copyright (C) 2007 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 885 $
* $Date: 2007-05-04 01:01:15 -0400 (Fri, 04 May 2007) $
***************************************************************************
**/

#include <dlrGeometry/plane3D.h>
#include <dlrNumeric/utilities.h>

namespace dnum = dlr::numeric;

namespace dlr {

  namespace geometry {
    
    // This constructor initializes the plane using three points.
    Plane3D::
    Plane3D(const Vector3D& point0,
            const Vector3D& point1,
            const Vector3D& point2,
            bool orthonormalize)
      : m_origin(point0),
        m_directionVector0(point1 - point0),
        m_directionVector1(point2 - point0)
    {
      if(orthonormalize) {
        m_directionVector0 /= dnum::magnitude(m_directionVector0);
        m_directionVector1 -=
          (dnum::dot(m_directionVector1, m_directionVector0)
           * m_directionVector0);
        m_directionVector1 /= dnum::magnitude(m_directionVector1);
      }
    }

    
    // The copy constructor deep copies its argument.
    Plane3D::
    Plane3D(const Plane3D& source)
      : m_origin(source.m_origin),
        m_directionVector0(source.m_directionVector0),
        m_directionVector1(source.m_directionVector1)
    {
      // Empty.
    }


    // The assignment operator deep copies its argument.
    Plane3D&
    Plane3D::
    operator=(const Plane3D& source)
    {
      if(&source != this) {
        m_origin = source.m_origin;
        m_directionVector0 = source.m_directionVector0;
        m_directionVector1 = source.m_directionVector1;
      }
      return *this;
    }


    /* ======= Non-member functions. ======= */

    std::ostream&
    operator<<(std::ostream& stream, const Plane3D& plane)
    {
      stream << "Plane3D{ "
             << plane.getOrigin() << ", "
             << plane.getDirectionVector0() << ", "
             << plane.getDirectionVector1() << " }";
      return stream;
    }


  } // namespace utilities
    
} // namespace dlr
