/**
***************************************************************************
* @file naiveSnakeTest.cpp
*
* Source file defining tests for naive Snake class.
*
* Copyright (C) 2006 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: $
* $Date: $
***************************************************************************
**/

#include <dlrComputerVision/naiveSnake.h>
#include <dlrNumeric/subArray2D.h>
#include <dlrTest/testFixture.h>

// xxx
#include <iostream>
#include <sstream>
#include <iomanip>
#include <dlrComputerVision/utilities.h>
#include <dlrComputerVision/imageIO.h>

using namespace dlr::computerVision;

namespace dlr {

  class NaiveSnakeTest
    : public TestFixture<NaiveSnakeTest> {

  public:

    NaiveSnakeTest();
    ~NaiveSnakeTest() {}
    void setVerboseOutput() {m_verbose = true;}
    
    void setUp(const std::string& testName) {}
    void tearDown(const std::string& testName) {}

    // Tests.
    void testExternalForce();
    void testStretchingAndBendingForces();

  private:

    void writeOutputFile(const std::vector<Vector2D>& snakePoints,
                         const std::string& fileNameBase,
                         size_t iterationCount);
    
    
    std::vector<bool> m_cornerFlags1;
    Image<GRAY1> m_interestImage0;
    std::vector<Vector2D> m_seedPoints0;
    std::vector<Vector2D> m_seedPoints1;
    std::vector<Vector2D> m_seedPoints2;
    size_t m_testImageColumns;
    size_t m_testImageRows;
    bool m_verbose;
    
  }; // class NaiveSnakeTest


  /* ============== Member Function Definititions ============== */

  NaiveSnakeTest::
  NaiveSnakeTest()
    : TestFixture<NaiveSnakeTest>("NaiveSnakeTest"),
      m_cornerFlags1(),
      m_interestImage0(),
      m_seedPoints0(),
      m_seedPoints1(),
      m_testImageColumns(320),
      m_testImageRows(240),
      m_verbose(false)
  {
    DLR_TEST_REGISTER_MEMBER(testExternalForce);
    DLR_TEST_REGISTER_MEMBER(testStretchingAndBendingForces);
    
    // Build an interest image.
    m_interestImage0.reinit(m_testImageRows, m_testImageColumns);
    m_interestImage0 = false;
    subArray(m_interestImage0, Slice(70, 150), 100) = true;
    subArray(m_interestImage0, Slice(70, 150), 220) = true;
    subArray(m_interestImage0, 70, Slice(100, 221)) = true;
    subArray(m_interestImage0, 149, Slice(100, 221)) = true;

    // Build some initial snake points.
    m_seedPoints0.push_back(Vector2D(80, 60));
    m_seedPoints0.push_back(Vector2D(240, 60));
    m_seedPoints0.push_back(Vector2D(240, 180));
    m_seedPoints0.push_back(Vector2D(80, 180));

    m_seedPoints1.push_back(Vector2D(36, 69));
    m_seedPoints1.push_back(Vector2D(145, 6));
    m_seedPoints1.push_back(Vector2D(311, 82));
    m_seedPoints1.push_back(Vector2D(176, 237));
    m_cornerFlags1.push_back(true);
    m_cornerFlags1.push_back(true);
    m_cornerFlags1.push_back(true);
    m_cornerFlags1.push_back(true);
    
    m_seedPoints2.push_back(Vector2D(145, 22));
    m_seedPoints2.push_back(Vector2D(137, 217));
    m_seedPoints2.push_back(Vector2D(166, 222));
    m_seedPoints2.push_back(Vector2D(182, 20));
  }


  void
  NaiveSnakeTest::
  testExternalForce()
  {
    Snake snake;
    snake.setInterestImage(m_interestImage0.copy());
    snake.setSeedPoints(m_seedPoints1);

    // Convigure snake to use only external force.
    snake.setStretchingConstant(0.0);
    snake.setBendingConstant(0.0);

    std::vector<Vector2D> snakePoints;    
    if(m_verbose) {
      // snake.setStepsPerIteration(1);
      // snake.setMinimumSpanLength(10);
      // snake.setMaximumSpanLength(30);
      size_t iterationCount = 0;
      while(!snake.isConverged()) {
        snakePoints = snake.runOneIteration();
        this->writeOutputFile(
          snakePoints, "externalForceTest", iterationCount);
        ++iterationCount;
      }
    } else {
      snakePoints = snake.run();
    }

    for(size_t index0 = 0; index0 < snakePoints.size(); ++index0) {
      size_t row = size_t(snakePoints[index0].y() + 0.5);
      size_t column = size_t(snakePoints[index0].x() + 0.5);
      DLR_TEST_ASSERT(row >= 0 && row < m_interestImage0.rows());
      DLR_TEST_ASSERT(column >= 0 && column < m_interestImage0.columns());
      DLR_TEST_ASSERT(m_interestImage0(row, column) == true);
    }
  }



  void
  NaiveSnakeTest::
  testStretchingAndBendingForces()
  {
    Snake snake;
    snake.setInterestImage(m_interestImage0.copy());
    snake.setSeedPoints(m_seedPoints1, m_cornerFlags1);

    // Convigure snake to use only external force.
    snake.setStretchingConstant(0.0);
    snake.setBendingConstant(10.0);
    snake.enableCornerAdditionAndDeletion();
    snake.setCornerAdditionAngle(0.25);
    snake.setCornerDeletionAngle(0.20);

    std::vector<Vector2D> snakePoints;    
    if(m_verbose) {
      // snake.setStepsPerIteration(1);
      // snake.setMinimumSpanLength(10);
      // snake.setMaximumSpanLength(30);
      size_t iterationCount = 0;
      while(!snake.isConverged()) {
        snakePoints = snake.runOneIteration();
        this->writeOutputFile(
          snakePoints, "stretchingAndBendingForceTest", iterationCount);
        ++iterationCount;
      }
    } else {
      snakePoints = snake.run();
    }

    for(size_t index0 = 0; index0 < snakePoints.size(); ++index0) {
      size_t row = size_t(snakePoints[index0].y() + 0.5);
      size_t column = size_t(snakePoints[index0].x() + 0.5);
      DLR_TEST_ASSERT(row >= 0 && row < m_interestImage0.rows());
      DLR_TEST_ASSERT(column >= 0 && column < m_interestImage0.columns());
      DLR_TEST_ASSERT(m_interestImage0(row, column) == true);
    }
  }


  void
  NaiveSnakeTest::
  writeOutputFile(const std::vector<Vector2D>& snakePoints,
                  const std::string& fileNameBase,
                  size_t iterationCount)
  {
    Image<GRAY1> resultImage(m_testImageRows, m_testImageColumns);
    resultImage = false;
    for(size_t index0 = 0; index0 < snakePoints.size(); ++index0) {
      size_t row = size_t(snakePoints[index0].y() + 0.5);
      size_t column = size_t(snakePoints[index0].x() + 0.5);
      DLR_TEST_ASSERT(row >= 0 && row < resultImage.rows());
      DLR_TEST_ASSERT(column >= 0 && column < resultImage.columns());
      resultImage(row, column) = true;
    }
    DLR_TEST_ASSERT(resultImage.size() != 0);
    std::ostringstream fileNameStream;
    fileNameStream << fileNameBase << std::setw(3) << std::setfill('0')
                   << iterationCount << ".pgm";
    writePGM8(fileNameStream.str(), convertColorspace<GRAY8>(resultImage));
  }
  
} // namespace dlr


#if 0

int main(int argc, char** argv)
{
  dlr::NaiveSnakeTest currentTest;
  if(argc > 1) {
    currentTest.setVerboseOutput();
  }
  bool result = currentTest.run();
  return (result ? 0 : 1);
}

#else

namespace {

  dlr::NaiveSnakeTest currentTest;

}

#endif
