/**
***************************************************************************
* @file getEuclideanDistanceTest.cpp
*
* Source file defining tests for getEuclideanDistance().
*
* Copyright (C) 2006 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: $
* $Date: $
***************************************************************************
**/

#include <dlrComputerVision/getEuclideanDistance.h>
#include <dlrTest/testFixture.h>

namespace dlr {

  namespace computerVision {
    
  class GetEuclideanDistanceTest
    : public TestFixture<GetEuclideanDistanceTest> {

  public:

    GetEuclideanDistanceTest();
    ~GetEuclideanDistanceTest() {}

    void setUp(const std::string& testName) {}
    void tearDown(const std::string& testName) {}

    // Tests.
    void testGetEuclideanDistance();

  private:

  }; // class GetEuclideanDistanceTest


  /* ============== Member Function Definititions ============== */

  GetEuclideanDistanceTest::
  GetEuclideanDistanceTest()
    : TestFixture<GetEuclideanDistanceTest>("GetEuclideanDistanceTest")
  {
    DLR_TEST_REGISTER_MEMBER(testGetEuclideanDistance);
  }


  void
  GetEuclideanDistanceTest::
  testGetEuclideanDistance()
  {
    const size_t testImageRows = 240;
    const size_t testImageColumns = 320;
    std::vector<int> uCoords;
    std::vector<int> vCoords;

    uCoords.push_back(150);
    vCoords.push_back(100);

    uCoords.push_back(80);
    vCoords.push_back(130);

    uCoords.push_back(85);
    vCoords.push_back(170);
    
    uCoords.push_back(175);
    vCoords.push_back(165);

    // A simple test with only one "bright" pixel.
    Image<GRAY8> testImage(testImageRows, testImageColumns);
    testImage = UnsignedInt8(0);
    testImage(vCoords[0], uCoords[0]) = UnsignedInt8(1);

    Array2D<double> referenceDistances(testImage.rows(), testImage.columns());
    for(int row = 0; row < static_cast<int>(testImage.rows()); ++row) {
      for(int column = 0; column < static_cast<int>(testImage.columns());
          ++column) {
        double deltaU = column - uCoords[0];
        double deltaV = row - vCoords[0];
        double distance = std::sqrt(deltaU * deltaU + deltaV * deltaV);
        referenceDistances(row, column) = distance;
      }
    }
    
    Array2D<double> distances = getEuclideanDistance(testImage, 10);

    DLR_TEST_ASSERT(distances.rows() == referenceDistances.rows());
    DLR_TEST_ASSERT(distances.columns() == referenceDistances.columns());
    for(size_t index0 = 0; index0 < distances.size(); ++index0) {
      DLR_TEST_ASSERT(
        approximatelyEqual(distances[index0], referenceDistances[index0],
                           1.0E-6));
    }

    // A slightly harder test with additional bright pixels.
    for(size_t pointIndex = 1; pointIndex < uCoords.size(); ++pointIndex) {
      testImage(vCoords[pointIndex], uCoords[pointIndex]) = UnsignedInt8(1);
    }

    for(int row = 0; row < static_cast<int>(testImage.rows()); ++row) {
      for(int column = 0; column < static_cast<int>(testImage.columns());
          ++column) {
        double deltaU = column - uCoords[0];
        double deltaV = row - vCoords[0];
        double distance = std::sqrt(deltaU * deltaU + deltaV * deltaV);
        for(size_t pointIndex = 1; pointIndex < uCoords.size(); ++pointIndex) {
          deltaU = column - uCoords[pointIndex];
          deltaV = row - vCoords[pointIndex];
          distance =
            std::min(distance, std::sqrt(deltaU * deltaU + deltaV * deltaV));
        }
        referenceDistances(row, column) = distance;
      }
    }

    distances = getEuclideanDistance(testImage, 10);

    DLR_TEST_ASSERT(distances.rows() == referenceDistances.rows());
    DLR_TEST_ASSERT(distances.columns() == referenceDistances.columns());
    for(size_t index0 = 0; index0 < distances.size(); ++index0) {
      DLR_TEST_ASSERT(
        approximatelyEqual(distances[index0], referenceDistances[index0],
                           1.0E-6));
    }
    
  }

  } // namespace computerVision

} // namespace dlr


#if 0

int main(int argc, char** argv)
{
  dlr::computerVision::GetEuclideanDistanceTest currentTest;
  bool result = currentTest.run();
  return (result ? 0 : 1);
}

#else

namespace {

  dlr::computerVision::GetEuclideanDistanceTest currentTest;

}

#endif
