/**
***************************************************************************
* @file dlrCommon/traceable.h
*
* Header file declaring tools for generating debugging traces when
* failures occur.
*
* Copyright (C) 2005-2007, David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 877 $
* $Date: 2007-05-04 00:12:02 -0400 (Fri, 04 May 2007) $
***************************************************************************
**/

#ifndef _DLR_COMMON_TRACEABLE_H_
#define _DLR_COMMON_TRACEABLE_H_

#include <iostream>
#include <sstream>
#include <vector>
#include <dlrCommon/argumentDescription.h>
#include <dlrCommon/exception.h>

// This file defines some preprocessor macros.  The definitions of
// these macros depend on whether or not we want to generate traces.
// We test here to find out which definitions to use.
#ifndef _DLRCOMMON_USE_TRACEABLE_

// Looks like we're not supposed to generate traces.  Define the
// macros to do nothing.

/**
 ** If _DLRCOMMON_USE_TRACEABLE_ is defined, this macro works with
 ** END_TRACEABLE to build a stack trace.  Use it inside each function
 ** definition, like this:
 **
 **   int myFunction(int argument0, int argument1)
 **   {
 **     BEGIN_TRACEABLE;
 **     // Your code goes here.
 **     return 0;
 **     END_TRACEABLE(myFunction, (argument0, argument1));
 **   }
 **/
#define BEGIN_TRACEABLE


/**
 ** If _DLRCOMMON_USE_TRACEABLE_ is defined, this macro works with
 ** BEGIN_TRACEABLE to build the stack trace.  Use it inside each
 ** function definition, like this:
 **
 **   int myFunction(int argument0, int argument1)
 **   {
 **     BEGIN_TRACEABLE;
 **     // Your code goes here.
 **     return 0;
 **     END_TRACEABLE(myFunction, (argument0, argument1));
 **   }
 **/
#define END_TRACEABLE(functionName, argumentList)


#else /* #ifndef _DLRCOMMON_USE_TRACEABLE_ */

/**
 ** If _DLRCOMMON_USE_TRACEABLE_ is defined, this macro works with
 ** END_TRACEABLE to build a stack trace.  Use it inside each function
 ** definition, like this:
 **
 **   int myFunction(int argument0, int argument1)
 **   {
 **     BEGIN_TRACEABLE;
 **     // Your code goes here.
 **     return 0;
 **     END_TRACEABLE(myFunction, (argument0, argument1));
 **   }
 **/
#define BEGIN_TRACEABLE \
  try {


/**
 ** If _DLRCOMMON_USE_TRACEABLE_ is defined, this macro works with
 ** BEGIN_TRACEABLE to build the stack trace.  Use it inside each
 ** function definition, like this:
 **
 **   int myFunction(int argument0, int argument1)
 **   {
 **     BEGIN_TRACEABLE;
 **     // Your code goes here.
 **     return 0;
 **     END_TRACEABLE(myFunction, (argument0, argument1));
 **   }
 **/
#define END_TRACEABLE(functionName, argumentList) \
  } catch(dlr::Exception& caughtException) { \
    try { \
      std::ostringstream dlr_end_traceable_message; \
      dlr_end_traceable_message \
        << "\n\n  (" << __FILE__ << ", " << __LINE__ << "): " \
        << #functionName << "()"; \
      dlr_end_traceable_message \
        << dlr::describeArguments argumentList; \
      caughtException.addTrace(dlr_end_traceable_message.str().c_str()); \
    } catch(...) { \
      /* Empty. The most likely reason to get here is std::bad_alloc */ \
      /* during the call to describeArguments() or during the call to */ \
      /* Exception::addTrace(). */ \
    } \
    /* This throw statment should rethrow the dlr::Exception instance, */ \
    /* _not_ any exception caught by the catch(...) statement above. */ \
    throw; \
  }

#endif /* #ifndef _DLRCOMMON_USE_TRACEABLE_ */


namespace dlr {

  namespace common {
    
  /** 
   * This function template formats its entire argument list for
   * inclusion in a stack trace.  You'll need to have templates like
   * this defined which take 1, 2, 3, 4, etc., arguments.
   * 
   * @return Since this function has no arguments, the return value is
   * an empty string.
   */
  inline std::string
  describeArguments()
  {
    return std::string("");
  }


  /** 
   * This function template formats its entire argument list for
   * inclusion in a stack trace.  You'll need to have templates like
   * this defined which take 1, 2, 3, 4, etc., arguments.
   * 
   * @param argument0 This argument is the first argument to be included
   * in the formatted output.
   * 
   * @return The return value is a string describing the argument list.
   */
  template <class Type0>
    std::string
    describeArguments(const Type0& argument0)
    {
      std::ostringstream message;
      message << "\n    (arg0): ";
      addArgumentDescription(message, argument0);
      return message.str();
    }


  /** 
   * This function template formats its entire argument list for
   * inclusion in a stack trace.  You'll need to have templates like
   * this defined which take 1, 2, 3, 4, etc., arguments.
   * 
   * @param argument0 This argument is the first argument to be included
   * in the formatted output.
   *
   * @param argument1 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @return The return value is a string describing the argument list.
   */
  template <class Type0, class Type1>
    std::string
    describeArguments(const Type0& argument0,
		      const Type1& argument1)
    {
      std::ostringstream message;
      message << "\n    (arg0): ";
      addArgumentDescription(message, argument0);
      message << "\n    (arg1): ";
      addArgumentDescription(message, argument1);
      return message.str();
    }


  /** 
   * This function template formats its entire argument list for
   * inclusion in a stack trace.  You'll need to have templates like
   * this defined which take 1, 2, 3, 4, etc., arguments.
   * 
   * @param argument0 This argument is the first argument to be included
   * in the formatted output.
   *
   * @param argument1 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument2 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @return The return value is a string describing the argument list.
   */
  template <class Type0, class Type1, class Type2>
    std::string
    describeArguments(const Type0& argument0,
		      const Type1& argument1,
		      const Type2& argument2)
    {
      std::ostringstream message;
      message << "\n    (arg0): ";
      addArgumentDescription(message, argument0);
      message << "\n    (arg1): ";
      addArgumentDescription(message, argument1);
      message << "\n    (arg2): ";
      addArgumentDescription(message, argument2);
      return message.str();
    }


  /** 
   * This function template formats its entire argument list for
   * inclusion in a stack trace.  You'll need to have templates like
   * this defined which take 1, 2, 3, 4, etc., arguments.
   * 
   * @param argument0 This argument is the first argument to be included
   * in the formatted output.
   *
   * @param argument1 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument2 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument3 This argument is the second argument to be included
   * in the formatted output.
   *
   * @return The return value is a string describing the argument list.
   */
  template <class Type0, class Type1, class Type2, class Type3>
    std::string
    describeArguments(const Type0& argument0,
		      const Type1& argument1,
		      const Type2& argument2,
		      const Type3& argument3)
    {
      std::ostringstream message;
      message << "\n    (arg0): ";
      addArgumentDescription(message, argument0);
      message << "\n    (arg1): ";
      addArgumentDescription(message, argument1);
      message << "\n    (arg2): ";
      addArgumentDescription(message, argument2);
      message << "\n    (arg3): ";
      addArgumentDescription(message, argument3);
      return message.str();
    }


  /** 
   * This function template formats its entire argument list for
   * inclusion in a stack trace.  You'll need to have templates like
   * this defined which take 1, 2, 3, 4, etc., arguments.
   * 
   * @param argument0 This argument is the first argument to be included
   * in the formatted output.
   *
   * @param argument1 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument2 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument3 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument4 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @return The return value is a string describing the argument list.
   */
  template <class Type0, class Type1, class Type2, class Type3, class Type4>
    std::string
    describeArguments(const Type0& argument0,
		      const Type1& argument1,
		      const Type2& argument2,
		      const Type3& argument3,
		      const Type4& argument4)
    {
      std::ostringstream message;
      message << "\n    (arg0): ";
      addArgumentDescription(message, argument0);
      message << "\n    (arg1): ";
      addArgumentDescription(message, argument1);
      message << "\n    (arg2): ";
      addArgumentDescription(message, argument2);
      message << "\n    (arg3): ";
      addArgumentDescription(message, argument3);
      message << "\n    (arg4): ";
      addArgumentDescription(message, argument4);
      return message.str();
    }


  /** 
   * This function template formats its entire argument list for
   * inclusion in a stack trace.  You'll need to have templates like
   * this defined which take 1, 2, 3, 4, etc., arguments.
   * 
   * @param argument0 This argument is the first argument to be included
   * in the formatted output.
   *
   * @param argument1 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument2 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument3 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument4 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument5 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @return The return value is a string describing the argument list.
   */
  template <class Type0, class Type1, class Type2, class Type3, class Type4,
    class Type5>
    std::string
    describeArguments(const Type0& argument0,
		      const Type1& argument1,
		      const Type2& argument2,
		      const Type3& argument3,
		      const Type4& argument4,
		      const Type5& argument5)
    {
      std::ostringstream message;
      message << "\n    (arg0): ";
      addArgumentDescription(message, argument0);
      message << "\n    (arg1): ";
      addArgumentDescription(message, argument1);
      message << "\n    (arg2): ";
      addArgumentDescription(message, argument2);
      message << "\n    (arg3): ";
      addArgumentDescription(message, argument3);
      message << "\n    (arg4): ";
      addArgumentDescription(message, argument4);
      message << "\n    (arg5): ";
      addArgumentDescription(message, argument5);
      return message.str();
    }


  /** 
   * This function template formats its entire argument list for
   * inclusion in a stack trace.  You'll need to have templates like
   * this defined which take 1, 2, 3, 4, etc., arguments.
   * 
   * @param argument0 This argument is the first argument to be included
   * in the formatted output.
   *
   * @param argument1 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument2 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument3 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument4 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument5 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument6 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @return The return value is a string describing the argument list.
   */
  template <class Type0, class Type1, class Type2, class Type3, class Type4,
    class Type5, class Type6>
    std::string
    describeArguments(const Type0& argument0,
		      const Type1& argument1,
		      const Type2& argument2,
		      const Type3& argument3,
		      const Type4& argument4,
		      const Type5& argument5,
		      const Type6& argument6)
    {
      std::ostringstream message;
      message << "\n    (arg0): ";
      addArgumentDescription(message, argument0);
      message << "\n    (arg1): ";
      addArgumentDescription(message, argument1);
      message << "\n    (arg2): ";
      addArgumentDescription(message, argument2);
      message << "\n    (arg3): ";
      addArgumentDescription(message, argument3);
      message << "\n    (arg4): ";
      addArgumentDescription(message, argument4);
      message << "\n    (arg5): ";
      addArgumentDescription(message, argument5);
      message << "\n    (arg6): ";
      addArgumentDescription(message, argument6);
      return message.str();
    }


  /** 
   * This function template formats its entire argument list for
   * inclusion in a stack trace.  You'll need to have templates like
   * this defined which take 1, 2, 3, 4, etc., arguments.
   * 
   * @param argument0 This argument is the first argument to be included
   * in the formatted output.
   *
   * @param argument1 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument2 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument3 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument4 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument5 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument6 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument7 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @return The return value is a string describing the argument list.
   */
  template <class Type0, class Type1, class Type2, class Type3, class Type4,
    class Type5, class Type6, class Type7>          
    std::string
    describeArguments(const Type0& argument0,
		      const Type1& argument1,
		      const Type2& argument2,
		      const Type3& argument3,
		      const Type4& argument4,
		      const Type5& argument5,
		      const Type6& argument6,
		      const Type7& argument7)
    {
      std::ostringstream message;
      message << "\n    (arg0): ";
      addArgumentDescription(message, argument0);
      message << "\n    (arg1): ";
      addArgumentDescription(message, argument1);
      message << "\n    (arg2): ";
      addArgumentDescription(message, argument2);
      message << "\n    (arg3): ";
      addArgumentDescription(message, argument3);
      message << "\n    (arg4): ";
      addArgumentDescription(message, argument4);
      message << "\n    (arg5): ";
      addArgumentDescription(message, argument5);
      message << "\n    (arg6): ";
      addArgumentDescription(message, argument6);
      message << "\n    (arg7): ";
      addArgumentDescription(message, argument7);
      return message.str();
    }


  /** 
   * This function template formats its entire argument list for
   * inclusion in a stack trace.  You'll need to have templates like
   * this defined which take 1, 2, 3, 4, etc., arguments.
   * 
   * @param argument0 This argument is the first argument to be included
   * in the formatted output.
   *
   * @param argument1 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument2 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument3 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument4 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument5 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument6 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument7 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument8 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @return The return value is a string describing the argument list.
   */
  template <class Type0, class Type1, class Type2, class Type3, class Type4,
    class Type5, class Type6, class Type7, class Type8>
    std::string
    describeArguments(const Type0& argument0,
		      const Type1& argument1,
		      const Type2& argument2,
		      const Type3& argument3,
		      const Type4& argument4,
		      const Type5& argument5,
		      const Type6& argument6,
		      const Type7& argument7,
		      const Type8& argument8)
    {
      std::ostringstream message;
      message << "\n    (arg0): ";
      addArgumentDescription(message, argument0);
      message << "\n    (arg1): ";
      addArgumentDescription(message, argument1);
      message << "\n    (arg2): ";
      addArgumentDescription(message, argument2);
      message << "\n    (arg3): ";
      addArgumentDescription(message, argument3);
      message << "\n    (arg4): ";
      addArgumentDescription(message, argument4);
      message << "\n    (arg5): ";
      addArgumentDescription(message, argument5);
      message << "\n    (arg6): ";
      addArgumentDescription(message, argument6);
      message << "\n    (arg7): ";
      addArgumentDescription(message, argument7);
      message << "\n    (arg8): ";
      addArgumentDescription(message, argument8);
      return message.str();
    }


  /** 
   * This function template formats its entire argument list for
   * inclusion in a stack trace.  You'll need to have templates like
   * this defined which take 1, 2, 3, 4, etc., arguments.
   * 
   * @param argument0 This argument is the first argument to be included
   * in the formatted output.
   *
   * @param argument1 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument2 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument3 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument4 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument5 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument6 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument7 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument8 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument9 This argument is the second argument to be included
   * in the formatted output.
   *
   * @return The return value is a string describing the argument list.
   */
  template <class Type0, class Type1, class Type2, class Type3, class Type4,
    class Type5, class Type6, class Type7, class Type8, class Type9>
    std::string
    describeArguments(const Type0& argument0,
		      const Type1& argument1,
		      const Type2& argument2,
		      const Type3& argument3,
		      const Type4& argument4,
		      const Type5& argument5,
		      const Type6& argument6,
		      const Type7& argument7,
		      const Type8& argument8,
		      const Type9& argument9)
    {
      std::ostringstream message;
      message << "\n    (arg0): ";
      addArgumentDescription(message, argument0);
      message << "\n    (arg1): ";
      addArgumentDescription(message, argument1);
      message << "\n    (arg2): ";
      addArgumentDescription(message, argument2);
      message << "\n    (arg3): ";
      addArgumentDescription(message, argument3);
      message << "\n    (arg4): ";
      addArgumentDescription(message, argument4);
      message << "\n    (arg5): ";
      addArgumentDescription(message, argument5);
      message << "\n    (arg6): ";
      addArgumentDescription(message, argument6);
      message << "\n    (arg7): ";
      addArgumentDescription(message, argument7);
      message << "\n    (arg8): ";
      addArgumentDescription(message, argument8);
      message << "\n    (arg9): ";
      addArgumentDescription(message, argument9);
      return message.str();
    }


  /** 
   * This function template formats its entire argument list for
   * inclusion in a stack trace.  You'll need to have templates like
   * this defined which take 1, 2, 3, 4, etc., arguments.
   * 
   * @param argument0 This argument is the first argument to be included
   * in the formatted output.
   *
   * @param argument1 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument2 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument3 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument4 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument5 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument6 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument7 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument8 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument9 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument10 This argument is the second argument to be included
   * in the formatted output.
   *
   * @return The return value is a string describing the argument list.
   */
  template <class Type0, class Type1, class Type2, class Type3, class Type4,
    class Type5, class Type6, class Type7, class Type8, class Type9,
    class Type10>
    std::string
    describeArguments(const Type0& argument0,
		      const Type1& argument1,
		      const Type2& argument2,
		      const Type3& argument3,
		      const Type4& argument4,
		      const Type5& argument5,
		      const Type6& argument6,
		      const Type7& argument7,
		      const Type8& argument8,
		      const Type9& argument9,
		      const Type10& argument10)
    {
      std::ostringstream message;
      message << "\n    (arg0): ";
      addArgumentDescription(message, argument0);
      message << "\n    (arg1): ";
      addArgumentDescription(message, argument1);
      message << "\n    (arg2): ";
      addArgumentDescription(message, argument2);
      message << "\n    (arg3): ";
      addArgumentDescription(message, argument3);
      message << "\n    (arg4): ";
      addArgumentDescription(message, argument4);
      message << "\n    (arg5): ";
      addArgumentDescription(message, argument5);
      message << "\n    (arg6): ";
      addArgumentDescription(message, argument6);
      message << "\n    (arg7): ";
      addArgumentDescription(message, argument7);
      message << "\n    (arg8): ";
      addArgumentDescription(message, argument8);
      message << "\n    (arg9): ";
      addArgumentDescription(message, argument9);
      message << "\n    (arg10): ";
      addArgumentDescription(message, argument10);
      return message.str();
    }


  /** 
   * This function template formats its entire argument list for
   * inclusion in a stack trace.  You'll need to have templates like
   * this defined which take 1, 2, 3, 4, etc., arguments.
   * 
   * @param argument0 This argument is the first argument to be included
   * in the formatted output.
   *
   * @param argument1 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument2 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument3 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument4 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument5 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument6 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument7 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument8 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument9 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument10 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument11 This argument is the second argument to be included
   * in the formatted output.
   *
   * @return The return value is a string describing the argument list.
   */
  template <class Type0, class Type1, class Type2, class Type3, class Type4,
    class Type5, class Type6, class Type7, class Type8, class Type9,
    class Type10, class Type11>
    std::string
    describeArguments(const Type0& argument0,
		      const Type1& argument1,
		      const Type2& argument2,
		      const Type3& argument3,
		      const Type4& argument4,
		      const Type5& argument5,
		      const Type6& argument6,
		      const Type7& argument7,
		      const Type8& argument8,
		      const Type9& argument9,
		      const Type10& argument10,
		      const Type11& argument11)
    {
      std::ostringstream message;
      message << "\n    (arg0): ";
      addArgumentDescription(message, argument0);
      message << "\n    (arg1): ";
      addArgumentDescription(message, argument1);
      message << "\n    (arg2): ";
      addArgumentDescription(message, argument2);
      message << "\n    (arg3): ";
      addArgumentDescription(message, argument3);
      message << "\n    (arg4): ";
      addArgumentDescription(message, argument4);
      message << "\n    (arg5): ";
      addArgumentDescription(message, argument5);
      message << "\n    (arg6): ";
      addArgumentDescription(message, argument6);
      message << "\n    (arg7): ";
      addArgumentDescription(message, argument7);
      message << "\n    (arg8): ";
      addArgumentDescription(message, argument8);
      message << "\n    (arg9): ";
      addArgumentDescription(message, argument9);
      message << "\n    (arg10): ";
      addArgumentDescription(message, argument10);
      message << "\n    (arg11): ";
      addArgumentDescription(message, argument11);
      return message.str();
    }


  /** 
   * This function template formats its entire argument list for
   * inclusion in a stack trace.  You'll need to have templates like
   * this defined which take 1, 2, 3, 4, etc., arguments.
   * 
   * @param argument0 This argument is the first argument to be included
   * in the formatted output.
   *
   * @param argument1 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument2 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument3 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument4 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument5 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument6 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument7 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument8 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument9 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument10 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument11 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument12 This argument is the second argument to be included
   * in the formatted output.
   *
   * @return The return value is a string describing the argument list.
   */
  template <class Type0, class Type1, class Type2, class Type3, class Type4,
    class Type5, class Type6, class Type7, class Type8, class Type9,
    class Type10, class Type11, class Type12>
    std::string
    describeArguments(const Type0& argument0,
		      const Type1& argument1,
		      const Type2& argument2,
		      const Type3& argument3,
		      const Type4& argument4,
		      const Type5& argument5,
		      const Type6& argument6,
		      const Type7& argument7,
		      const Type8& argument8,
		      const Type9& argument9,
		      const Type10& argument10,
		      const Type11& argument11,
		      const Type12& argument12)
    {
      std::ostringstream message;
      message << "\n    (arg0): ";
      addArgumentDescription(message, argument0);
      message << "\n    (arg1): ";
      addArgumentDescription(message, argument1);
      message << "\n    (arg2): ";
      addArgumentDescription(message, argument2);
      message << "\n    (arg3): ";
      addArgumentDescription(message, argument3);
      message << "\n    (arg4): ";
      addArgumentDescription(message, argument4);
      message << "\n    (arg5): ";
      addArgumentDescription(message, argument5);
      message << "\n    (arg6): ";
      addArgumentDescription(message, argument6);
      message << "\n    (arg7): ";
      addArgumentDescription(message, argument7);
      message << "\n    (arg8): ";
      addArgumentDescription(message, argument8);
      message << "\n    (arg9): ";
      addArgumentDescription(message, argument9);
      message << "\n    (arg10): ";
      addArgumentDescription(message, argument10);
      message << "\n    (arg11): ";
      addArgumentDescription(message, argument11);
      message << "\n    (arg12): ";
      addArgumentDescription(message, argument12);
      return message.str();
    }


  /** 
   * This function template formats its entire argument list for
   * inclusion in a stack trace.  You'll need to have templates like
   * this defined which take 1, 2, 3, 4, etc., arguments.
   * 
   * @param argument0 This argument is the first argument to be included
   * in the formatted output.
   *
   * @param argument1 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument2 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument3 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument4 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument5 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument6 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument7 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument8 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument9 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument10 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument11 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument12 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument13 This argument is the second argument to be included
   * in the formatted output.
   *
   * @return The return value is a string describing the argument list.
   */
  template <class Type0, class Type1, class Type2, class Type3, class Type4,
    class Type5, class Type6, class Type7, class Type8, class Type9,
    class Type10, class Type11, class Type12, class Type13>
    std::string
    describeArguments(const Type0& argument0,
		      const Type1& argument1,
		      const Type2& argument2,
		      const Type3& argument3,
		      const Type4& argument4,
		      const Type5& argument5,
		      const Type6& argument6,
		      const Type7& argument7,
		      const Type8& argument8,
		      const Type9& argument9,
		      const Type10& argument10,
		      const Type11& argument11,
		      const Type12& argument12,
		      const Type13& argument13)
    {
      std::ostringstream message;
      message << "\n    (arg0): ";
      addArgumentDescription(message, argument0);
      message << "\n    (arg1): ";
      addArgumentDescription(message, argument1);
      message << "\n    (arg2): ";
      addArgumentDescription(message, argument2);
      message << "\n    (arg3): ";
      addArgumentDescription(message, argument3);
      message << "\n    (arg4): ";
      addArgumentDescription(message, argument4);
      message << "\n    (arg5): ";
      addArgumentDescription(message, argument5);
      message << "\n    (arg6): ";
      addArgumentDescription(message, argument6);
      message << "\n    (arg7): ";
      addArgumentDescription(message, argument7);
      message << "\n    (arg8): ";
      addArgumentDescription(message, argument8);
      message << "\n    (arg9): ";
      addArgumentDescription(message, argument9);
      message << "\n    (arg10): ";
      addArgumentDescription(message, argument10);
      message << "\n    (arg11): ";
      addArgumentDescription(message, argument11);
      message << "\n    (arg12): ";
      addArgumentDescription(message, argument12);
      message << "\n    (arg13): ";
      addArgumentDescription(message, argument13);
      return message.str();
    }


  /** 
   * This function template formats its entire argument list for
   * inclusion in a stack trace.  You'll need to have templates like
   * this defined which take 1, 2, 3, 4, etc., arguments.
   * 
   * @param argument0 This argument is the first argument to be included
   * in the formatted output.
   *
   * @param argument1 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument2 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument3 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument4 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument5 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument6 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument7 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument8 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument9 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument10 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument11 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument12 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument13 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument14 This argument is the second argument to be included
   * in the formatted output.
   *
   * @return The return value is a string describing the argument list.
   */
  template <class Type0, class Type1, class Type2, class Type3, class Type4,
    class Type5, class Type6, class Type7, class Type8, class Type9,
    class Type10, class Type11, class Type12, class Type13, class Type14>
    std::string
    describeArguments(const Type0& argument0,
		      const Type1& argument1,
		      const Type2& argument2,
		      const Type3& argument3,
		      const Type4& argument4,
		      const Type5& argument5,
		      const Type6& argument6,
		      const Type7& argument7,
		      const Type8& argument8,
		      const Type9& argument9,
		      const Type10& argument10,
		      const Type11& argument11,
		      const Type12& argument12,
		      const Type13& argument13,
		      const Type14& argument14)
    {
      std::ostringstream message;
      message << "\n    (arg0): ";
      addArgumentDescription(message, argument0);
      message << "\n    (arg1): ";
      addArgumentDescription(message, argument1);
      message << "\n    (arg2): ";
      addArgumentDescription(message, argument2);
      message << "\n    (arg3): ";
      addArgumentDescription(message, argument3);
      message << "\n    (arg4): ";
      addArgumentDescription(message, argument4);
      message << "\n    (arg5): ";
      addArgumentDescription(message, argument5);
      message << "\n    (arg6): ";
      addArgumentDescription(message, argument6);
      message << "\n    (arg7): ";
      addArgumentDescription(message, argument7);
      message << "\n    (arg8): ";
      addArgumentDescription(message, argument8);
      message << "\n    (arg9): ";
      addArgumentDescription(message, argument9);
      message << "\n    (arg10): ";
      addArgumentDescription(message, argument10);
      message << "\n    (arg11): ";
      addArgumentDescription(message, argument11);
      message << "\n    (arg12): ";
      addArgumentDescription(message, argument12);
      message << "\n    (arg13): ";
      addArgumentDescription(message, argument13);
      message << "\n    (arg14): ";
      addArgumentDescription(message, argument14);
      return message.str();
    }


  /** 
   * This function template formats its entire argument list for
   * inclusion in a stack trace.  You'll need to have templates like
   * this defined which take 1, 2, 3, 4, etc., arguments.
   * 
   * @param argument0 This argument is the first argument to be included
   * in the formatted output.
   *
   * @param argument1 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument2 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument3 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument4 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument5 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument6 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument7 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument8 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument9 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument10 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument11 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument12 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument13 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument14 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument15 This argument is the second argument to be included
   * in the formatted output.
   *
   * @return The return value is a string describing the argument list.
   */
  template <class Type0, class Type1, class Type2, class Type3, class Type4,
    class Type5, class Type6, class Type7, class Type8, class Type9,
    class Type10, class Type11, class Type12, class Type13, class Type14,
    class Type15>
    std::string
    describeArguments(const Type0& argument0,
		      const Type1& argument1,
		      const Type2& argument2,
		      const Type3& argument3,
		      const Type4& argument4,
		      const Type5& argument5,
		      const Type6& argument6,
		      const Type7& argument7,
		      const Type8& argument8,
		      const Type9& argument9,
		      const Type10& argument10,
		      const Type11& argument11,
		      const Type12& argument12,
		      const Type13& argument13,
		      const Type14& argument14,
		      const Type15& argument15)
    {
      std::ostringstream message;
      message << "\n    (arg0): ";
      addArgumentDescription(message, argument0);
      message << "\n    (arg1): ";
      addArgumentDescription(message, argument1);
      message << "\n    (arg2): ";
      addArgumentDescription(message, argument2);
      message << "\n    (arg3): ";
      addArgumentDescription(message, argument3);
      message << "\n    (arg4): ";
      addArgumentDescription(message, argument4);
      message << "\n    (arg5): ";
      addArgumentDescription(message, argument5);
      message << "\n    (arg6): ";
      addArgumentDescription(message, argument6);
      message << "\n    (arg7): ";
      addArgumentDescription(message, argument7);
      message << "\n    (arg8): ";
      addArgumentDescription(message, argument8);
      message << "\n    (arg9): ";
      addArgumentDescription(message, argument9);
      message << "\n    (arg10): ";
      addArgumentDescription(message, argument10);
      message << "\n    (arg11): ";
      addArgumentDescription(message, argument11);
      message << "\n    (arg12): ";
      addArgumentDescription(message, argument12);
      message << "\n    (arg13): ";
      addArgumentDescription(message, argument13);
      message << "\n    (arg14): ";
      addArgumentDescription(message, argument14);
      message << "\n    (arg15): ";
      addArgumentDescription(message, argument15);
      return message.str();
    }


  /** 
   * This function template formats its entire argument list for
   * inclusion in a stack trace.  You'll need to have templates like
   * this defined which take 1, 2, 3, 4, etc., arguments.
   * 
   * @param argument0 This argument is the first argument to be included
   * in the formatted output.
   *
   * @param argument1 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument2 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument3 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument4 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument5 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument6 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument7 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument8 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument9 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument10 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument11 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument12 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument13 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument14 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument15 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument16 This argument is the second argument to be included
   * in the formatted output.
   *
   * @return The return value is a string describing the argument list.
   */
  template <class Type0, class Type1, class Type2, class Type3, class Type4,
    class Type5, class Type6, class Type7, class Type8, class Type9,
    class Type10, class Type11, class Type12, class Type13, class Type14,
    class Type15, class Type16>
    std::string
    describeArguments(const Type0& argument0,
		      const Type1& argument1,
		      const Type2& argument2,
		      const Type3& argument3,
		      const Type4& argument4,
		      const Type5& argument5,
		      const Type6& argument6,
		      const Type7& argument7,
		      const Type8& argument8,
		      const Type9& argument9,
		      const Type10& argument10,
		      const Type11& argument11,
		      const Type12& argument12,
		      const Type13& argument13,
		      const Type14& argument14,
		      const Type15& argument15,
		      const Type16& argument16)
    {
      std::ostringstream message;
      message << "\n    (arg0): ";
      addArgumentDescription(message, argument0);
      message << "\n    (arg1): ";
      addArgumentDescription(message, argument1);
      message << "\n    (arg2): ";
      addArgumentDescription(message, argument2);
      message << "\n    (arg3): ";
      addArgumentDescription(message, argument3);
      message << "\n    (arg4): ";
      addArgumentDescription(message, argument4);
      message << "\n    (arg5): ";
      addArgumentDescription(message, argument5);
      message << "\n    (arg6): ";
      addArgumentDescription(message, argument6);
      message << "\n    (arg7): ";
      addArgumentDescription(message, argument7);
      message << "\n    (arg8): ";
      addArgumentDescription(message, argument8);
      message << "\n    (arg9): ";
      addArgumentDescription(message, argument9);
      message << "\n    (arg10): ";
      addArgumentDescription(message, argument10);
      message << "\n    (arg11): ";
      addArgumentDescription(message, argument11);
      message << "\n    (arg12): ";
      addArgumentDescription(message, argument12);
      message << "\n    (arg13): ";
      addArgumentDescription(message, argument13);
      message << "\n    (arg14): ";
      addArgumentDescription(message, argument14);
      message << "\n    (arg15): ";
      addArgumentDescription(message, argument15);
      message << "\n    (arg16): ";
      addArgumentDescription(message, argument16);
      return message.str();
    }


  /** 
   * This function template formats its entire argument list for
   * inclusion in a stack trace.  You'll need to have templates like
   * this defined which take 1, 2, 3, 4, etc., arguments.
   * 
   * @param argument0 This argument is the first argument to be included
   * in the formatted output.
   *
   * @param argument1 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument2 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument3 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument4 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument5 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument6 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument7 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument8 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument9 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument10 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument11 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument12 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument13 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument14 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument15 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument16 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument17 This argument is the second argument to be included
   * in the formatted output.
   *
   * @return The return value is a string describing the argument list.
   */
  template <class Type0, class Type1, class Type2, class Type3, class Type4,
    class Type5, class Type6, class Type7, class Type8, class Type9,
    class Type10, class Type11, class Type12, class Type13, class Type14,
    class Type15, class Type16, class Type17>
    std::string
    describeArguments(const Type0& argument0,
		      const Type1& argument1,
		      const Type2& argument2,
		      const Type3& argument3,
		      const Type4& argument4,
		      const Type5& argument5,
		      const Type6& argument6,
		      const Type7& argument7,
		      const Type8& argument8,
		      const Type9& argument9,
		      const Type10& argument10,
		      const Type11& argument11,
		      const Type12& argument12,
		      const Type13& argument13,
		      const Type14& argument14,
		      const Type15& argument15,
		      const Type16& argument16,
		      const Type17& argument17)
    {
      std::ostringstream message;
      message << "\n    (arg0): ";
      addArgumentDescription(message, argument0);
      message << "\n    (arg1): ";
      addArgumentDescription(message, argument1);
      message << "\n    (arg2): ";
      addArgumentDescription(message, argument2);
      message << "\n    (arg3): ";
      addArgumentDescription(message, argument3);
      message << "\n    (arg4): ";
      addArgumentDescription(message, argument4);
      message << "\n    (arg5): ";
      addArgumentDescription(message, argument5);
      message << "\n    (arg6): ";
      addArgumentDescription(message, argument6);
      message << "\n    (arg7): ";
      addArgumentDescription(message, argument7);
      message << "\n    (arg8): ";
      addArgumentDescription(message, argument8);
      message << "\n    (arg9): ";
      addArgumentDescription(message, argument9);
      message << "\n    (arg10): ";
      addArgumentDescription(message, argument10);
      message << "\n    (arg11): ";
      addArgumentDescription(message, argument11);
      message << "\n    (arg12): ";
      addArgumentDescription(message, argument12);
      message << "\n    (arg13): ";
      addArgumentDescription(message, argument13);
      message << "\n    (arg14): ";
      addArgumentDescription(message, argument14);
      message << "\n    (arg15): ";
      addArgumentDescription(message, argument15);
      message << "\n    (arg16): ";
      addArgumentDescription(message, argument16);
      message << "\n    (arg17): ";
      addArgumentDescription(message, argument17);
      return message.str();
    }


  /** 
   * This function template formats its entire argument list for
   * inclusion in a stack trace.  You'll need to have templates like
   * this defined which take 1, 2, 3, 4, etc., arguments.
   * 
   * @param argument0 This argument is the first argument to be included
   * in the formatted output.
   *
   * @param argument1 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument2 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument3 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument4 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument5 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument6 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument7 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument8 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument9 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument10 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument11 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument12 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument13 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument14 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument15 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument16 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument17 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument18 This argument is the second argument to be included
   * in the formatted output.
   *
   * @return The return value is a string describing the argument list.
   */
  template <class Type0, class Type1, class Type2, class Type3, class Type4,
    class Type5, class Type6, class Type7, class Type8, class Type9,
    class Type10, class Type11, class Type12, class Type13, class Type14,
    class Type15, class Type16, class Type17, class Type18>
    std::string
    describeArguments(const Type0& argument0,
		      const Type1& argument1,
		      const Type2& argument2,
		      const Type3& argument3,
		      const Type4& argument4,
		      const Type5& argument5,
		      const Type6& argument6,
		      const Type7& argument7,
		      const Type8& argument8,
		      const Type9& argument9,
		      const Type10& argument10,
		      const Type11& argument11,
		      const Type12& argument12,
		      const Type13& argument13,
		      const Type14& argument14,
		      const Type15& argument15,
		      const Type16& argument16,
		      const Type17& argument17,
		      const Type18& argument18)
    {
      std::ostringstream message;
      message << "\n    (arg0): ";
      addArgumentDescription(message, argument0);
      message << "\n    (arg1): ";
      addArgumentDescription(message, argument1);
      message << "\n    (arg2): ";
      addArgumentDescription(message, argument2);
      message << "\n    (arg3): ";
      addArgumentDescription(message, argument3);
      message << "\n    (arg4): ";
      addArgumentDescription(message, argument4);
      message << "\n    (arg5): ";
      addArgumentDescription(message, argument5);
      message << "\n    (arg6): ";
      addArgumentDescription(message, argument6);
      message << "\n    (arg7): ";
      addArgumentDescription(message, argument7);
      message << "\n    (arg8): ";
      addArgumentDescription(message, argument8);
      message << "\n    (arg9): ";
      addArgumentDescription(message, argument9);
      message << "\n    (arg10): ";
      addArgumentDescription(message, argument10);
      message << "\n    (arg11): ";
      addArgumentDescription(message, argument11);
      message << "\n    (arg12): ";
      addArgumentDescription(message, argument12);
      message << "\n    (arg13): ";
      addArgumentDescription(message, argument13);
      message << "\n    (arg14): ";
      addArgumentDescription(message, argument14);
      message << "\n    (arg15): ";
      addArgumentDescription(message, argument15);
      message << "\n    (arg16): ";
      addArgumentDescription(message, argument16);
      message << "\n    (arg17): ";
      addArgumentDescription(message, argument17);
      message << "\n    (arg18): ";
      addArgumentDescription(message, argument18);
      return message.str();
    }


  /** 
   * This function template formats its entire argument list for
   * inclusion in a stack trace.  You'll need to have templates like
   * this defined which take 1, 2, 3, 4, etc., arguments.
   * 
   * @param argument0 This argument is the first argument to be included
   * in the formatted output.
   *
   * @param argument1 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument2 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument3 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument4 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument5 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument6 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument7 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument8 This argument is the second argument to be included
   * in the formatted output.
   * 
   * @param argument9 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument10 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument11 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument12 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument13 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument14 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument15 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument16 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument17 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument18 This argument is the second argument to be included
   * in the formatted output.
   *
   * @param argument19 This argument is the second argument to be included
   * in the formatted output.
   *
   * @return The return value is a string describing the argument list.
   */
  template <class Type0, class Type1, class Type2, class Type3, class Type4,
    class Type5, class Type6, class Type7, class Type8, class Type9,
    class Type10, class Type11, class Type12, class Type13, class Type14,
    class Type15, class Type16, class Type17, class Type18, class Type19>
    std::string
    describeArguments(const Type0& argument0,
		      const Type1& argument1,
		      const Type2& argument2,
		      const Type3& argument3,
		      const Type4& argument4,
		      const Type5& argument5,
		      const Type6& argument6,
		      const Type7& argument7,
		      const Type8& argument8,
		      const Type9& argument9,
		      const Type10& argument10,
		      const Type11& argument11,
		      const Type12& argument12,
		      const Type13& argument13,
		      const Type14& argument14,
		      const Type15& argument15,
		      const Type16& argument16,
		      const Type17& argument17,
		      const Type18& argument18,
		      const Type19& argument19)
    {
      std::ostringstream message;
      message << "\n    (arg0): ";
      addArgumentDescription(message, argument0);
      message << "\n    (arg1): ";
      addArgumentDescription(message, argument1);
      message << "\n    (arg2): ";
      addArgumentDescription(message, argument2);
      message << "\n    (arg3): ";
      addArgumentDescription(message, argument3);
      message << "\n    (arg4): ";
      addArgumentDescription(message, argument4);
      message << "\n    (arg5): ";
      addArgumentDescription(message, argument5);
      message << "\n    (arg6): ";
      addArgumentDescription(message, argument6);
      message << "\n    (arg7): ";
      addArgumentDescription(message, argument7);
      message << "\n    (arg8): ";
      addArgumentDescription(message, argument8);
      message << "\n    (arg9): ";
      addArgumentDescription(message, argument9);
      message << "\n    (arg10): ";
      addArgumentDescription(message, argument10);
      message << "\n    (arg11): ";
      addArgumentDescription(message, argument11);
      message << "\n    (arg12): ";
      addArgumentDescription(message, argument12);
      message << "\n    (arg13): ";
      addArgumentDescription(message, argument13);
      message << "\n    (arg14): ";
      addArgumentDescription(message, argument14);
      message << "\n    (arg15): ";
      addArgumentDescription(message, argument15);
      message << "\n    (arg16): ";
      addArgumentDescription(message, argument16);
      message << "\n    (arg17): ";
      addArgumentDescription(message, argument17);
      message << "\n    (arg18): ";
      addArgumentDescription(message, argument18);
      message << "\n    (arg19): ";
      addArgumentDescription(message, argument19);
      return message.str();
    }

  } // namespace common
  
} // namespace dlr


/* ======= Declarations to maintain compatibility with legacy code. ======= */

namespace dlr {

  using common::describeArguments;
  
} // namespace dlr

#endif /* #ifndef _DLR_COMMON_TRACEABLE_H_ */
