/**
***************************************************************************
* @file bayesClassifier.h
* Header file declaring BayesClassifier class.
*
* Copyright (C) 2003-2004 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 706 $
* $Date: 2006-08-04 19:41:11 -0400 (Fri, 04 Aug 2006) $
***************************************************************************
**/

#ifndef _DLR_BAYESCLASSIFIER_H_
#define _DLR_BAYESCLASSIFIER_H_

#include <string>
#include <map>
#include <vector>

namespace dlr {

  /**
   ** WARNING: A replacement for this library is under development,
   ** and the current code is abandoned.  Please do not use it.
   **
   ** The BayesClassifier class template implements a simple Bayes
   ** classifier.  It is templated on the type of distribution to use
   ** for classification.  Usually this will be GaussianDistribution<
   ** Array1D<double> > or something similar.  This type should
   ** provide the following interface:
   **
   **   - It must provide a public typedef, vector_type describing the
   **   type used to represent vectors for this distribution.
   **
   **   - TBD;
   **
   **   - TBD;
   **/
  template <class DistributionType>
  class BayesClassifier {
  public:
    /* ******** Public typedefs ******** */
    
    /** 
     * This public typedef indicates the vector type for the
     * probability distributions.  Examples are Array1D<double>,
     * std::vector<float>, etc.
     */
    typedef typename DistributionType::vector_type vector_type;
    

    /* ******** Public member functions ******** */

    /** 
     * Default constructor creates an uninitialized classifier.
     */
    BayesClassifier();

    /** 
     * This constructor implicitly learns classes and distributions
     * based on training data.  It is equivalent to constructing a
     * BayesClassifier instance using the default constructor, and
     * then calling its train(...) method.
     * 
     * @param trainingData This argument is a std::vector of labeled
     * training data.  Each element of the vector is a
     * std::pair<std::string, vector_type> in which the string
     * indicates the name of the class to which this vector_type
     * instance belongs.
     */
    BayesClassifier(
      const std::vector< std::pair<std::string, vector_type> >& trainingData);

    /** 
     * This constructor is appropriate for cases in which the class
     * distributions are known a priori, and can be set explicitly at
     * the time of construction.  If this constructor is used, there
     * is no need to late call the train(...) method, although it will
     * probably be necessary to call the setPriorProbability(double)
     * method for each class.  Prior probabilities will be initialized
     * to 1.0 / N, where N is the number of classes in the input
     * vector.
     * 
     * @param distributionData This argument is a std::vector
     * containing the class names and distributions.  Each element is
     * a std::pair<std::string, DistributionType> in which the string
     * indicates the class name, and the DistributionType instance
     * indicates its distribution.
     */
    BayesClassifier(
      const std::vector< std::pair<std::string, DistributionType> >&
      distributionData);

    /** 
     * The copy constructor does a deep copy.
     *
     * @param source The BayesClassifier instance to be copied.
     */
    BayesClassifier(const BayesClassifier<DistributionType> &source);
    
    /**
     * The destructor destroys the BayesClassifier instance and
     * cleans up any allocated storage.
     */
    ~BayesClassifier();

    /** 
     * This method learns classes and distributions based on training
     * data.  One class will be created for each class name occurring
     * in the training data, and prior probabilities will be computed
     * based on frequency of occurrence in the training data.
     * 
     * @param trainingData This argument is a std::vector of labeled
     * training data.  Each element of the vector is a
     * std::pair<std::string, vector_type> in which the string
     * indicates the name of the class to which this vector_type
     * instance belongs.
     */
    void
    train(
      const std::vector< std::pair<std::string, vector_type> >& trainingData);

    /** 
     * This method allows the user to explicitly set the component
     * distributions.  It is useful in cases where the class
     * statistics are known, and should not be inferred from training
     * data.  Use this method instead of train(...), above.
     * 
     * @param distributionData This argument is a std::vector
     * containing the class names and distributions.  Each element is
     * a std::pair<std::string, DistributionType> in which the string
     * indicates the class name, and the DistributionType instance
     * indicates its distribution.
     */
    void
    setDistributions(
      const std::vector< std::pair<std::string, DistributionType> >&
      distributionData);

    /** 
     * This method classifies an input vector and returns the label of
     * the associated class.
     *
     * @param inputVector This argument is the vector to be classified.
     *
     * @return The return value is the label of the most likely class.
     */
    std::string
    operator()(const vector_type& inputVector) const;

    /** 
     * The assignment operator deep copies its argument.
     * 
     * @param source This is the BayesClassifier instance to be
     * copied.
     *
     * @return The return value is a reference to *this.
     */
    BayesClassifier<DistributionType>&
    operator=(const BayesClassifier<DistributionType>& source);
    
  private:
    /* ********Private member functions******** */

    /** 
     * This private member function reports whether or not the
     * classifier has a class corresponding to the provided label.
     * 
     * @param label This argument specifies the name of the class to
     * look for.
     *
     * @return The return value is true if the classifier instance has
     * a class matching the provided label, and false otherwise.
     */
    inline bool
    haveClass(const std::string& label) const;
    
    /* ********Private data members******** */

    std::map<std::string, size_t> m_classLookup;
    std::vector<std::string> m_classLabels;
    std::vector<DistributionType> m_distributions;
    std::vector<double> m_priorProbabilities;
  };

  /* Non-member functions which should maybe wind up in a different file */

  // None
  
}; // namespace dlr


/*================================================================
 * Member function definitions follow.  This would be a .C file
 * if it weren't templated.
 *================================================================*/

#include <sstream>
#include <vector>
#include <dlrCommon/exception.h>

namespace dlr {

  // Default constructor creates an uninitialized classifier.
  template <class DistributionType>
  BayesClassifier<DistributionType>::
  BayesClassifier()
    : m_classLookup(),
      m_classLabels(),
      m_distributions(),
      m_priorProbabilities()
  {
    // Empty
  }

  // This constructor implicitly learns classes and distributions
  // based on training data.
  template <class DistributionType>
  BayesClassifier<DistributionType>::
  BayesClassifier(
    const std::vector< std::pair<std::string, vector_type> >& trainingData)
    : m_classLookup(),
      m_classLabels(),
      m_distributions(),
      m_priorProbabilities()
  {
    this->train(trainingData);
  }

  // This constructor is appropriate for cases in which the class
  // distributions are known a priori, and can be set explicitly at
  // the time of construction.
  template <class DistributionType>
  BayesClassifier<DistributionType>::
  BayesClassifier(
    const std::vector< std::pair<std::string, DistributionType> >&
    distributionData)
    : m_classLookup(),
      m_classLabels(distributionData.size()),
      m_distributions(distributionData.size()),
      m_priorProbabilities(distributionData.size())
  {
    this->setDistributions(distributionData);
  }

  // The copy constructor does a deep copy.
  template <class DistributionType>
  BayesClassifier<DistributionType>::
  BayesClassifier(const BayesClassifier<DistributionType> &source)
    : m_classLookup(source.m_classLookup),
      m_classLabels(source.m_classLabels),
      m_distributions(source.m_distributions),
      m_priorProbabilities(source.m_priorProbabilities)
  {
    // Empty
  }
    
  // The destructor destroys the BayesClassifier instance and
  // cleans up any allocated storage.
  template <class DistributionType>
  BayesClassifier<DistributionType>::
  ~BayesClassifier()
  {
    // Empty
  }
  
  // This method learns classes and distributions based on training
  // data.
  template <class DistributionType>
  void
  BayesClassifier<DistributionType>::
  train(
    const std::vector< std::pair<std::string, vector_type> >& trainingData)
  {
    // First reset internal state.
    m_classLookup.clear();
    m_classLabels.clear();
    m_distributions.clear();
    m_priorProbabilities.clear();

    // For each trainig sample.
    for(size_t index = 0; index < trainingData.size(); ++index) {
      std::string classLabel = trainingData[index].first;
      // Have we already seen this label?
      if(!this->haveClass(classLabel)) {
        // No.  Set up a new distribution.
        m_classLookup[classLabel] = m_classLabels.size();
        m_classLabels.push_back(classLabel);
        m_distributions.push_back(DistributionType());
        m_priorProbabilities.push_back(0.0);
      }
      // Update the relevant distribution.
      size_t classIndex = m_classLookup[classLabel];
      m_distributions[classIndex].addSample(trainingData[index].second);
      // And increment our record of the number of times we've seen this
      // class.
      m_priorProbabilities[classIndex] += 1.0;
    }
    // Finally, convert occurrence counts to prior probabilities by dividing.
    std::transform(m_priorProbabilities.begin(), m_priorProbabilities.end(),
                   m_priorProbabilities.begin(),
                   std::bind2nd(std::multiplies<double>(),
                                1.0 / m_classLabels.size()));
  }
  

  // This method allows the user to explicitly set the component
  // distributions.
  template <class DistributionType>
  void
  BayesClassifier<DistributionType>::
  setDistributions(
    const std::vector< std::pair<std::string, DistributionType> >&
      distributionData)
  {
    // First reset internal state.
    m_classLookup.clear();
    m_classLabels.clear();
    m_distributions.clear();
    m_priorProbabilities.clear();

    // Now copy the input data.
    for(size_t index = 0; index < distributionData.size(); ++index) {
      if(this->haveClass(distributionData[index].first)) {
        std::ostringstream message;
        message << "Class number " << index << " has duplicate class label "
                << distributionData[index].first << ".";
        DLR_THROW3(ValueException,
                   "BayesClassifier::setDistributions(...)",
                   message.str().c_str());
      }
      m_classLookup[distributionData[index].first] = index;
      m_classLabels.push_back(distributionData[index].first);
      m_distributions.push_back(distributionData[index].second);
      m_priorProbabilities.push_back(1.0 / distributionData.size());
    }
  }

  // This method classifies an input vector and returns the label of
  // the associated class.
  template <class DistributionType>
  std::string
  BayesClassifier<DistributionType>::
  operator()(const vector_type& inputVector) const
  {
    // First check state.
    if(m_classLabels.size() == 0) {
      DLR_THROW3(StateException,
                 "BayesClassifier::operator()(const vector_type&)",
                 "Classifier instance has not been initialized."
                 "Please call train() or setDistributions() before "
                 "trying to classify data.");
    }
    // Next compute probability for each candidate class.
    std::vector<double> probabilities(m_classLabels.size());
    for(size_t index = 0; index < m_classLabels.size(); ++index) {
      probabilities[index] = (m_priorProbabilities[index]
                              * (m_distributions[index])(inputVector));
    }
    // Select the highest probability.
    size_t bestIndex =
      static_cast<size_t>(std::max_element(probabilities.begin(),
                                           probabilities.end())
                          - probabilities.begin());
    // And return the corresponding label.
    return m_classLabels[bestIndex];
  }

  // The assignment operator deep copies its argument.
  template <class DistributionType>
  BayesClassifier<DistributionType>&
  BayesClassifier<DistributionType>::
  operator=(const BayesClassifier<DistributionType>& source)
  {
    m_classLookup = source.m_classLookup;
    m_classLabels = source.m_classLabels;
    m_distributions = source.m_distributions;
    m_priorProbabilities = source.m_priorProbabilities;
    return *this;
  }

  // This private member function reports whether or not the
  // classifier has a class corresponding to the provided label.
  template <class DistributionType>
  bool
  BayesClassifier<DistributionType>::
  haveClass(const std::string& label) const
  {
    // First a typedef to make things easier to type.
    typedef typename std::vector<std::string>::const_iterator VSIterator;

    // Search for the provided label.
    VSIterator vsIterator =
      std::find(m_classLabels.begin(), m_classLabels.end(), label);

    // And return true if a match was found.
    return (vsIterator != m_classLabels.end());
  }
  
}; // namespace dlr

#endif // #ifdef _DLR_BAYESCLASSIFIER_H_

