/**
***************************************************************************
* @file dlrUtilities/optionParser.h
*
* Header file declaring the OptionParser class.
*
* Copyright (C) 2006-2007 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 706 $
* $Date: 2006-08-04 19:41:11 -0400 (Fri, 04 Aug 2006) $
***************************************************************************
**/

#ifndef _DLRUTILITIES_OPTIONPARSER_H_
#define _DLRUTILITIES_OPTIONPARSER_H_

#include <map>
#include <string>
#include <vector>
#include <dlrUtilities/optionDescription.h>
#include <dlrUtilities/stringManipulation.h>

namespace dlr {

  namespace utilities {

    
    /**
     ** The OptionParser class parses program options.  It supports
     ** the following features.
     **
     **   - Short style options: "du -s -k"
     **   - Long gnu-style options: "gnumeric --version --no-splash"
     **   - Options with values: "xclock -g 200x200 -rgb5"
     **   - Named positional arguments: "ls foo.txt"
     **   - Repeated options: "gcc -I/usr/X11/include -I. -I../include"
     **   - Default values for unspecified options.
     **   - Automatic option documentation for usage messages.
     **   - Short option stacking (if desired): "ls -ltr"
     **   - Automatic program exit and printing of usage messages (if desired).
     **
     ** Here is example code:
     **
     ** @code
     **   // Create an OptionParser which automatically generates
     **   // help messages, and which prints usage and calls exit(65)
     **   // if it encounters an inappropriate command line.
     **   OptionParser optionParser(65);
     **
     **   // Specify options which do not take arguments.
     **   optionParser.addOption(
     **     "NOLINK", "-c", "--no_link",
     **     "Don't run the linker.  Instead, generate .o files.");
     **   optionParser.addOption(
     **     "DEBUG", "-g", "--debug",
     **     "Include debugging symbols in generated object code.");
     **
     **   // Specify options which require values.
     **   optionParser.addOptionWithValue(
     **     "OPTIMIZATION_LEVEL", "-O", "--optimization_level", "0",
     **     "Set optimization level.");
     **   optionParser.addOptionWithValue(
     **     "INCLUDE_DIR", "-I", "--include_dir", "",
     **     "Add INCLUDE_DIR to include search path.");
     **
     **   // Specify required positional arguments.
     **   optionParser.addPositionalArgument(
     **     "INPUT_FILE", "File to be compiled.", true);
     **
     **   // Specify positional arguments which are not required.
     **   optionParser.addPositionalArgument(
     **     "SECOND_FILE", "Additional file to be compiled.", false);
     **
     **   // Parse program arguments.
     **   optionParser.parseCommandLine(argc, argv);
     **
     **   // Recover results of parse.
     **   bool debug = optionParser.getCount("DEBUG") != 0;
     **   int  optimizationLevel =
     **     optionParser.convertValue<int>("OPTIMIZATION_LEVEL");
     **   std::string inputFile0 = optionParser.getValue("INPUT_FILE");
     **   std::string inputFile1 = optionParser.getValue("SECOND_FILE");
     **   std::vector<std::string> includeDirs;
     **   for(size_t includeIndex = 0;
     **       includeIndex < optionParser.getCount("INCLUDE_DIR");
     **       ++includeIndex) {
     **     includeDirs.push_back(
     **       optionParser.getValue("INCLUDE_DIR", includeIndex));
     **   }
     ** @endcode
     **
     ** A command line argument is considered to match the short
     ** version of an option if it begins with the entire text of the
     ** short version.  For example, each of "-p", "-pdf", and "-pp"
     ** would match an option with a short version of "-p".  Arguments
     ** which match the short version of an option, but have
     ** additional text (such as "-pdf" in the example above) are
     ** interpreted in one of two ways.  If constructor argument
     ** allowStackedShortOptions was set to true, and the short
     ** version of the matched option starts with the character '-',
     ** and the matched option does not take an value, then the
     ** remaining text will be prepended with a '-' character and
     ** re-parsed.  In this case, the "-pdf" from the previous example
     ** would be parsed as if it were typed "-p -df".  In all other
     ** cases, the remaining text will be interpreted as an argument
     ** to the matched option, that is "-pdf" would be parsed as "-p"
     ** with the argument "df".
     **
     ** A command line argument is considered to match the long
     ** version of an option if one of three things is true: first,
     ** the text of the argument matches the text of the long version
     ** of the option exactly; second the option was specified with
     ** "allowPartialMatch" set to true, and the entire text of the
     ** argument matches an initial substring of the long version of
     ** the option; or third, the argument begins with a string that
     ** matches under one of the other two cases, and then continues
     ** with a string that starts with an equals sign.  For an option
     ** with long version "--optimization_level", these three cases
     ** are illustrated by the three argument strings
     ** "--optimization_level", "--opt", and "--opt=3".  In the final
     ** case, the text following the equals sign is interpreted as a
     ** value being specified for the option.
     **/
    class OptionParser {
    public:

      /** 
       * Default constructor.  OptionParser instances constructed in
       * this way will throw an IOException from parseCommandLine() if
       * the command line doesn't match the specified options.  You
       * can catch the exception and the print usage, etc.
       * 
       * @param allowExtraArguments This argument specifies whether
       * extra positional argument should be permitted.  If
       * allowExtraArguments is false, then member function
       * parseCommandLine() will throw an IOException whenever the
       * number of positional arguments on the command line exceeds
       * the number specified using member function
       * addPositionalArgument().  If allowExtraArguments is true,
       * then any extra positional arguments will be recorded without
       * throwing an exception.  These arguments can then be recovered
       * using member function getExtraPositionalArguments().
       * 
       * @param allowStackedShortOptions This argument specifies
       * whether it is permissible to decompose arguments such as
       * "-plr" into "-p -l -r".  If allowStackedShortOptions is true,
       * then this type of decomposition will be carried out for short
       * options which do not accept values.  If option "-p" does
       * accept a value, then "-plr" will (as always) be interpreted
       * as "-p" with the argument "lr".  Note that this type of
       * decomposition may lead to ambiguity.  For example, if you
       * have an option "-p" and an option "-pl", then it's hard to
       * know how "-plr" should parse.  If any of your short options
       * are longer than a dash plus a single character, then you may
       * wish disable stacked short options by setting
       * allowStackedShortOptions to false.
       * 
       * @param allowOptionishArguments This argument specifies how to
       * handle command line arguments which start with "-", but which
       * don't correspond to any of the specified options.  If
       * allowOptionishArguments is true, then such arguments will be
       * simply recorded as positional arguments.  If
       * allowOptionishArguments is false, then such arguments will
       * cause an IOException to be thrown for "unrecognized option."
       */
      OptionParser(bool allowExtraArguments = true,
                   bool allowStackedShortOptions = true,
                   bool allowOptionishArguments = false);


      /** 
       * This constructor specifies that if a malformed commandline is
       * parsed, the OptionParser should exit instead of throwing an
       * exception, optionally printing a usage message first.
       * 
       * @param exitCode This argument specifies the exit code which
       * should be used to report a bad command line.
       * 
       * @param handleMinusMinusHelp This argument specifies whether
       * the parser should automatically watch of "--help" arguments.
       * If handleMinusMinusHelp is true, the parser will print a
       * usage message and then exit(0) when a "--help" option is
       * encountered.
       * 
       * @param printUsage This argument specifies whether, when a
       * malformed commandline is parsed, a usage message should be
       * printed before exiting with the specified exit code.
       * 
       * @param allowExtraArguments Please refer to the documentation
       * for the default constructor to find a description of this
       * argument.
       * 
       * @param allowStackedShortOptions Please refer to the
       * documentation for the default constructor to find a
       * description of this argument.
       * 
       * @param allowOptionishArguments Please refer to the
       * documentation for the default constructor to find a
       * description of this argument.
       */
      explicit
      OptionParser(int exitCode,
                   bool handleMinusMinusHelp = true,
                   bool printUsage = true,
                   bool allowExtraArguments = true,
                   bool allowStackedShortOptions = true,
                   bool allowOptionishArguments = false);


      /**
       * Destructor.
       */
      ~OptionParser();


      /** 
       * This member function adds an option to be recognized during
       * command line parsing.
       * 
       * @param name This argument specifies a unique name for the
       * option.  It will be used later to refer to this option when
       * checking parsing results.
       * 
       * @param shortVersion This argument specifies a short version
       * of the option, such as "-p" or "-rgb".  Setting this argument
       * to "" indicates that this option does not have a short
       * version.
       * 
       * @param longVersion This argument specifies a long version of
       * the option, such as "--do_calibration".  Setting this
       * argument to "" indicates that this option does not have a
       * long version.
       * 
       * @param docString This argument specifies a breif description
       * for the option.  If the automatic documentation features of
       * the optionParser are not being used, then this argument can
       * be safely set to "".
       * 
       * @param allowPartialMatch This argument specifies whether to
       * allow partial matches for the long version of this option.
       * For more information, please see the class comment for
       * OptionParser.
       */
      void
      addOption(const std::string& name,
                const std::string& shortVersion,
                const std::string& longVersion,
                const std::string& docString,
                bool allowPartialMatch = true);

      
      /** 
       * This member function adds an option which requires a value to
       * the list which will be recognized during command line
       * parsing.
       * 
       * @param name This argument specifies a unique name for the
       * option.  It will be used later to refer to this option when
       * checking parsing results.
       * 
       * @param shortVersion This argument specifies a short version
       * of the option, such as "-p" or "-rgb".  Setting this argument
       * to "" indicates that this option does not have a short
       * version.
       * 
       * @param longVersion This argument specifies a long version of
       * the option, such as "--do_calibration".  Setting this
       * argument to "" indicates that this option does not have a
       * long version.
       * 
       * @param defaultValue This argument specifies the default value
       * for the option.  If the option is not found on the command
       * line, then the default value will be used.
       * 
       * @param docString This argument specifies a breif description
       * for the option.  If the automatic documentation features of
       * the optionParser are not being used, then this argument can
       * be safely set to "".
       * 
       * @param requireArgument This argument is currently not supported.
       * 
       * @param allowPartialMatch This argument specifies whether to
       * allow partial matches for the long version of this option.
       * For more information, please see the class comment for
       * OptionParser.
       * 
       * @param allowOptionishValue This argument specifies whether or
       * not the supplied value (on the command line) is permitted to
       * begin with the character '-'.
       */
      void
      addOptionWithValue(const std::string& name,
                         const std::string& shortVersion,
                         const std::string& longVersion,
                         const std::string& defaultValue,
                         const std::string& docString,
                         bool requireArgument = true,
                         bool allowPartialMatch = true,
                         bool allowOptionishValue = false);
      

      /** 
       * This member function adds a named positional argument.  The
       * order in which positional arguments are specified is
       * important.  This is most clearly illustrated with an example:
       *
       * @code
       *   int argc = 4;
       *   char* argv = {"progname", "arg0", "arg1", "arg2"};
       *   OptionParser optionParser;
       *   optionParser.addPositionalArgument("FOO", "");
       *   optionParser.addPositionalArgument("BAR", "");
       *   optionParser.addPositionalArgument("BAZ", "");
       *   optionParser.parseCommandLine(argc, argv);
       *   // The next line returns "arg0".
       *   std::string firstArgument = optionParser.getValue("FOO");
       *   // The next line returns "arg1".
       *   std::string secondArgument = optionParser.getValue("BAR");
       *   // The next line returns "arg2".
       *   std::string thirdArgument = optionParser.getValue("BAZ");
       * @endcode
       *
       * @param name This argument specifies a unique name for the
       * positional argument.  It will be used later to refer to this
       * argument when checking parsing results.
       * 
       * @param docString This argument specifies a breif description
       * for the argument.  If the automatic documentation features of
       * the optionParser are not being used, then this argument can
       * be safely set to "".
       * 
       * @param isRequired This argument specifies whether the absence
       * of this positional argument should cause the parsing of a
       * command line to fail.  If isRequired is set to false, and
       * this positional argument is missing from the command line,
       * then the default value specified by argument defaultValue
       * will be used.
       */
      void
      addPositionalArgument(const std::string& name,
                            const std::string& docString,
                            bool isRequired = false,
                            const std::string& defaultValue = "");


      /** 
       * This member function is like getValue(const std::string&),
       * but attempts to convert the returned string to the specified
       * type.  It works for all built-in types, and for user-defined
       * types which define a stream input operator.  If the
       * conversion fails, and you used the OptionParser constructor
       * which specifies an exit code, then the OptionParser will
       * behave as if the command line parsing failed, printing usage
       * and calling exit() as appropriate.  If the conversion fails
       * and the non-exit code constructor was used, then getValue()
       * will throw a ConversionException.
       * 
       * @param name This argument specifies the option or positional
       * argument whose value is to be queried.  If name does not
       * match the name argument of a previous call to addOption(),
       * addOptionWithValue(), or addPositionalArgument(), then the
       * returned value will be the empty string.
       * 
       * @return The return value is the value for the requested
       * option or positional argument.
       */
      template<class Type>
      Type
      convertValue(const std::string& name);


      /** 
       * This member function is like getValue(const std::string&, int),
       * but attempts to convert the returned string to the specified
       * type.  It works for all built-in types, and for user-defined
       * types which define a stream input operator.  If the
       * conversion fails, and you used the OptionParser constructor
       * which specifies an exit code, then the OptionParser will
       * behave as if the command line parsing failed, printing usage
       * and calling exit() as appropriate.  If the conversion fails
       * and the non-exit code constructor was used, then getValue()
       * will throw a ConversionException.
       * 
       * @param name This argument is as described in the
       * documentation for getValue(const std::string&, int).
       * 
       * @param valueIndex This argument is as described in the
       * documentation for getValue(const std::string&, int).
       * 
       * @return The return value is the value for the requested
       * option or positional argument.
       */
      template<class Type>
      Type
      convertValue(const std::string& name, int valueIndex);


      /** 
       * This member function indicates how many times the specified
       * option or positional argument was specified in the most
       * recently parsed command line.  For positional arguments, the
       * return value is always 0 or 1.  For options, the return value
       * may be higher, since options may be specified more than once.
       * 
       * @param name This argument is as described in the
       * documentation for getValue(const std::string&, int).
       * 
       * @return The return value is the value specified on the
       * commandline for the indicated option or positional argument.
       * If the command line did not provide that option or argument,
       * a default value will be returned.
       */
      size_t
      getCount(const std::string& name) const;


      /** 
       * If constructor argument allowExtraArguments was set to true,
       * and if number of positional arguments on the most recently
       * parsed command line exceeds the number of named positional
       * arguments specified using addPositionalArgument(), then this
       * member function will return the extra positional arguments,
       * in the order they were found on the command line, as a vector
       * of strings.
       * 
       * @return The return value is a vector of extra positional
       * arguments.
       */
      std::vector<std::string>
      getExtraPositionalArguments();


      /** 
       * This member function returns a formatted string describing
       * the available command line options, as specified using
       * addOption() and addOptionWithValue().  It is useful for
       * building your own usage messages if you used the default
       * constructor.  If you used the constructor which specifies an
       * exit code, you probably have no use for
       * getOptionsDescription().  The string returned by
       * getOptionsDescription() will be much more informative if the
       * docString arguments of the addOption() and
       * addOptionWithValue() calls were not empty.
       * 
       * @return The return value is a formatted string describing the
       * configured options.
       */
      std::string
      getOptionsDescription();


      /** 
       * This member function returns a formatted string suitable for
       * a usage message.  It includes the string returned by
       * getOptionsDescription(), so if you call getUsage(), you
       * probably don't need to call getOptionsDescription().  If you
       * used the constructor which specifies an exit code, you
       * probably have no use for getUsage().  The string returned by
       * getOptionsDescription() will be much more informative if the
       * docString arguments of addOption(), addOptionWithValue(), and
       * addPositionalArgument() calls were not empty.
       * 
       * @return The return value is a formatted usage message.
       */
      std::string
      getUsage();


      /** 
       * Following a successful parse of a command line, this member
       * function returns the requested option value or positional
       * argument value as a string.  If an option was specified more
       * than once, this member function returns the value specified
       * latest on the command line.  This member function is
       * equivalent to calling getValue<Type>(const std::string, int)
       * with the int argument set to -1.  If name does not match any
       * specified option or positional argument, then a
       * ValueException will be thrown.
       * 
       * @param name This argument specifies the option or positional
       * argument whose value is to be queried.  If name does not
       * match the name argument of a previous call to addOption(),
       * addOptionWithValue(), or addPositionalArgument(), then the
       * returned value will be the empty string.
       * 
       * @return The return value is the value for the requested
       * option or positional argument, or the empty string if no such
       * value was specified on the most recently parsed command line.
       */
      std::string
      getValue(const std::string& name);


      /** 
       * Following a successful parse of a command line, this member
       * function returns the (valueIndex)th value of the requested
       * option or positional argument.
       * 
       * @param name This argument specifies the option or positional
       * argument whose value is to be queried.  If name does not
       * match the name argument of a previous call to addOption(),
       * addOptionWithValue(), or addPositionalArgument(), then the
       * returned value will be the empty string.
       * 
       * @param valueIndex This argument specifies which occurrence of
       * the option/argument is being queried.  If valueIndex is set
       * to -1, than the last occurrence of the option will be
       * returned.  ValueIndex must always be less than the number
       * returned by getCount(name), and therefore must always be 0 or
       * -1 for positional arguments.
       *
       * @return The return value is the value for the requested
       * option or positional argument, or the empty string if no such
       * value was specified on the most recently parsed command line.
       */
      std::string
      getValue(const std::string& name, int valueIndex);


      /**
       * @deprecated {Replaced by member function convertValue().}
       *
       * This deprecated member function is equivalent to
       * convertValue<Type>(const std::string&).
       * 
       * @param name This argument is as described in the
       * documentation for getValue(const std::string&).
       * 
       * @return The return value is the recovered argument value.
       */
      template<class Type>
      Type
      getValue(const std::string& name, type_tag<Type>);

    
      /** 
       * @deprecated {Replaced by member function convertValue().}
       *
       * This deprecated member function is equivalent to
       * convertValue<Type>(const std::string&, int).
       * 
       * @param name This argument is as described in the
       * documentation for getValue(const std::string&, int).
       * 
       * @param valueIndex This argument is as described in the
       * documentation for getValue(const std::string&, int).
       * 
       * @return The return value is the recovered argument value.
       */
      template<class Type>
      Type
      getValue(const std::string& name, int valueIndex, type_tag<Type>);


      /** 
       * This member function should be called after all calls to
       * addOption(), addOptionWithValue(), and
       * addPositionalArgument() are completed in order to parse a
       * command line and store the result for subsequent access using
       * the getCount() and getValue() methods.  If the command line
       * doesn't match the expected options & positional arguments,
       * one of several things will happen, depending on which
       * constructor & constructor arguments were used to create the
       * OptionParser.  The default behavior is to throw an
       * IOException, but the OptionParser can also be configured to
       * exit with a specific error code, or to print a usage message
       * before exiting with a specific error code.  If the
       * allowMinusMinusHelp constructor argument was specified, and a
       * "--help" option is encountered, then the OptionParser
       * instance will print a usage message and then call exit(0).
       * 
       * @param argc This argument can be taken directly from the argc
       * argument to main().
       * 
       * @param argv This argument can be taken directly from the argv
       * argument to main().
       */
      void
      parseCommandLine(int argc, const char* argv[]);
    

      /** 
       * This member function is identical to parseCommandLine(int
       * argc, const char* argv[]), except that it does not require a
       * const qualifier on its second argument.
       * 
       * @param argc This argument can be taken directly from the argc
       * argument to main().
       * 
       * @param argv This argument can be taken directly from the argv
       * argument to main().
       */
      void
      parseCommandLine(int argc, char* argv[]);
    

    private:

      // Private enum for specifying how to handle illegal commandlines.
      enum ErrorBehavior {ExitOnError, ThrowOnError, UsageAndExitOnError};


      // Private member functions
      bool
      findOptionDescription(const std::string& argument,
                            OptionDescription& optionDescription,
                            size_t& typedLength,
                            bool& isShortMatch);

      
      void
      recordPositionalArgument(const std::string& argument);
      

      // Private data members.
      bool m_allowExtraArguments;
      bool m_allowOptionishArguments;
      bool m_allowStackedShortOptions;
      ErrorBehavior m_errorBehavior;
      int m_exitCode;
      std::vector<std::string> m_extraArgumentValues;
      bool m_handleHelp;
      size_t m_numberOfPosArgRequired;
      size_t m_numberOfPosArgParsed;
      std::map<std::string, size_t> m_optionCounts;
      std::map<std::string, OptionDescription> m_optionDescriptions;
      std::map< std::string, std::vector<std::string> > m_optionValues;
      std::vector<std::string> m_positionalArgumentNames;
      std::vector<std::string> m_positionalArgumentDefaultValues;
      std::vector<std::string> m_positionalArgumentDocs;
      std::string m_programName;

    }; // class OptionParser

  } // namespace utilities

} // namespace dlr

/* ========= Definitions of inline and template functions below. ========== */

namespace dlr {

  namespace utilities {

    
    template<class Type>
    Type
    OptionParser::
    convertValue(const std::string& name)
    {
      try {
        return convertString<Type>(this->getValue(name));
      } catch(ConversionException&) {
        std::ostringstream message;
        message << "Error when parsing option " << name << ": "
                << "can't convert \"" << this->getValue(name)
                << "\" to appropriate type.";
        DLR_THROW(ConversionException, "OptionParser::convertValue()",
                  message.str().c_str());
      }
    }
    
    
    template<class Type>
    Type
    OptionParser::
    convertValue(const std::string& name, int valueIndex)
    {
      try {
        return convertString<Type>(this->getValue(name, valueIndex));
      } catch(ConversionException&) {
        std::ostringstream message;
        message << "Error when parsing option " << name << ": "
                << "can't convert \"" << this->getValue(name, valueIndex)
                << "\" to appropriate type.";
        DLR_THROW(ConversionException, "OptionParser::convertValue()",
                  message.str().c_str());
      }
    }
    
    
    template<class Type>
    Type
    OptionParser::
    getValue(const std::string& name, type_tag<Type>)
    {
      return this->convertValue<Type>(name);
    }
    
    
    template<class Type>
    Type
    OptionParser::
    getValue(const std::string& name, int valueIndex, type_tag<Type>)
    {
      return this->convertValue<Type>(name, valueIndex);
    }
    
    
  } // namespace utilities

} // namespace dlr

#endif // #ifndef _DLRUTILITES_OPTIONPARSER_H_
