/**
***************************************************************************
* @file dlrUtilities/path.h
*
* Header file declaring routines for working with the filesystem
*
* Copyright (C) 2003-2007, David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 1035 $
* $Date: 2008-09-18 15:21:43 -0400 (Thu, 18 Sep 2008) $
***************************************************************************
**/

#ifndef _DLR_PATH_H_
#define _DLR_PATH_H_

#include <string>
#include <vector>

namespace dlr {

  namespace utilities {

    /** 
     * This function returns a bool indicating whether or not the
     * specified path is a directory.  If the path is a symbolic link,
     * the return value is currently unspecified, but will eventually be
     * true iff the link points to a directory.
     * 
     * @param path This argument is the filename to evaluate.
     * 
     * @return The return value is true if path refers to a directory,
     * false otherwise.
     */
    bool
    isDirectory(const std::string& path);
  
  
    bool
    isExistingPath(const std::string& path);
  
  
    bool
    isRegularFile(const std::string& path);
    
    
    /**
     ** Joins two path elements with the appropriate path delimiter.
     ** For example:
     **   joinPath("foo", "bar");
     ** might give
     **   "foo/bar"
     ** while
     **   joinPath("foo/baz/", "bar");
     ** might give
     **   "/foo/baz/bar"
     **/
    std::string
    joinPath(const std::string& part0, const std::string& part1);

  
    /**
     ** Returns the names of the entries in the specified directory, in
     ** no particular order.  For example, 
     **   listDirectory("/etc");
     ** might give
     **   ["fstab", "init.d", "modules.conf", ...]
     ** while
     **   listDirectory("/etc", true);
     ** might give
     **   ["/etc/fstab", "/etc/init.d", "/etc/modules.conf", ...]
     **/
    std::vector<std::string>
    listDirectory(const std::string& directoryName, bool fullPath=false);


    /** 
     * Returns the names of files in the directory tree below the
     * specified directory.  For example,
     *
     *   recursiveListDirectory("/etc");
     *
     * might give
     *
     *   ["fstab", "init.d/cron", "init.d/ssh", "modules.conf", ...]
     *
     * while
     *
     *   recursiveListDirectory("/etc", true);
     *
     * might give
     *
     *   ["fstab", "init.d", "init.d/cron", "init.d/ssh", "modules.conf", ...]
     *
     * and
     *
     *   recursiveListDirectory("/etc", true, true);
     *
     * might give
     *
     *   ["/etc/fstab", "/etc/init.d", "/etc/init.d/cron", "/etc/init.d/ssh",
     *    "/etc/modules.conf", ...]
     *
     * @param directoryName This argument specifies the directory to be listed.
     * 
     * @return The return value is a vector of file names.
     */
    std::vector<std::string>
    recursiveListDirectory(const std::string& directoryName,
                           bool fullPath=false,
                           bool includeDirectoryNames=false);


    /** 
     * This function searches a sequence of directories looking for
     * the specified file.  If the file is found, the complete path to
     * the found file is returned through reference argument fullPath.
     * If argument discardInputDir is true, then any preceding
     * directory information will be stripped from the input fileName
     * before searching.  If the requested file exists in more than
     * one directory, the first match will be returned.
     *
     * For example, imagine the following directory structure:
     *
     * @code
     *   /home
     *     /rick
     *       /dir0
     *         /subdir0
     *           foo.txt
     *       /dir1
     *         bar.txt
     *       /dir2
     *         foo.txt
     *       /dir3
     *         bar.txt
     * @endcode
     *
     * And assume the following vector of directory names:
     *
     * @code
     *   std::vector<std::string> searchPath(3);
     *   searchPath[0] = /home/rick/dir0;
     *   searchPath[1] = /home/rick/dir1;
     *   searchPath[2] = /home/rick/dir2;
     * @endcode
     *
     * With this directory structure, the call:
     *
     * @code
     *   std::string result;
     *   searchForFile("subdir0/foo.txt", searchPath.begin(), searchPath.end(),
     *                 result, false);
     * @endcode
     *
     * would return true and set result to
     * "/home/rick/dir0/subdir0/foo.txt".
     *
     * The call:
     * 
     * @code
     *   std::string result;
     *   searchForFile("subdir0/foo.txt", searchPath.begin(), searchPath.end(),
     *                 result, true);
     * @endcode
     *
     * would return true and set result to
     * "/home/rick/dir2/foo.txt".
     *
     * The call:
     *
     * @code
     *   std::string result;
     *   searchForFile("subdir0/baz.txt", searchPath.begin(), searchPath.end(),
     *                 result, true);
     * @endcode
     *
     * would return false and not touch result.
     *   
     * @param fileName This argument is the file to be found.
     * 
     * @param pathBegin This argument is an interator pointing to the
     * beginning of a sequence of directories in which to search.
     * 
     * @param pathEnd This argument is an interator pointing to the
     * end (in the STL sense) of a sequence of directories in which to
     * search.
     * 
     * @param fullPath This argument is used to return the full path
     * to the located file.  If the file is not found, this argument
     * will not be accessed.
     *
     * @param discardInputDir If this argument is set to true, than
     * any preceding directory information will be removed from
     * argument fileName before the search is conducted.  See the
     * examples above for more information.
     * 
     * @return The return value is true if the file is found, false otherwise.
     */
    template <class IterType>
    bool
    searchForFile(const std::string& fileName,
                  IterType pathBegin, IterType pathEnd,
                  std::string& fullPath,
                  bool discardInputDir = false);
    
  
    /**
     ** Returns a std::pair<std::string, std::string> containing the fileName
     ** without its extension, and the extension.  For example:
     **   splitExt("/foo/bar.baz")
     ** might return
     **   {"/foo/bar", ".baz"}
     **/
    std::pair<std::string, std::string>
    splitExtension(const std::string& fileName);


    /** 
     * This function accepts a path returns a pair of strings in which
     * the first element is the directory name and the second is the
     * filename.  For example:
     *
     *   splitPath("/foo/bar.baz")
     *
     * might return
     *
     *   {"/foo/", "bar.baz"}
     *
     * Also,
     *
     *   splitPath("bar.baz")
     *
     * might return
     *
     *   {"", "bar.baz"}
     *
     *   splitPath("/foo/")
     *
     * might return
     *
     *   {"/foo/", ""}
     *
     * 
     * @return The return value is a pair containing first the directory
     * name, and second the file name.
     */
    std::pair<std::string, std::string>
    splitPath(const std::string& path);

  } // namespace utilities
    
} // namespace dlr


/* ======= Declarations to maintain compatibility with legacy code. ======= */

namespace dlr {

  using utilities::isDirectory;
  using utilities::isExistingPath;
  using utilities::isRegularFile;
  using utilities::joinPath;
  using utilities::listDirectory;
  using utilities::recursiveListDirectory;
  using utilities::splitExtension;
  using utilities::splitPath;
  
} // namespace dlr


/* ======= Declarations of inline and template functions. ======= */

namespace dlr {

  namespace utilities {
    
    // This function searches a sequence of directories looking for
    // the specified file.
    template <class IterType>
    bool
    searchForFile(const std::string& fileName,
                  IterType pathBegin, IterType pathEnd,
                  std::string& fullPath,
                  bool discardInputDir)
    {
      std::string baseName;
      if(discardInputDir) {
        baseName = splitPath(fileName).second;
      } else {
        baseName = fileName;
      }
      while(pathBegin != pathEnd) {
        std::string candidateName = joinPath(*pathBegin, baseName);
        if(isRegularFile(candidateName)) {
          fullPath = candidateName;
          return true;
        }
        ++pathBegin;
      }
      return false;
    }

  } // namespace utilities
  
} // namespace dlr

#endif // #ifndef _DLR_PATH_H_
