/**
***************************************************************************
* @file dlrUtilities/date.cpp
*
* Source file defining Date class.
*
* Copyright (C) 2003-2005 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 977 $
* $Date: 2007-12-30 02:50:58 -0500 (Sun, 30 Dec 2007) $
***************************************************************************
**/

#include <iostream>
#include <sstream>
#include <iomanip>
#include <dlrUtilities/stringManipulation.h>
#include <dlrUtilities/date.h>

namespace dlr {

  namespace utilities {

    /// Set from an ISO date string
    void
    Date::
    setISODate(const std::string& isoString)
    {
      // Be a little forgiving of format.  Replace both "-" and "/"
      // with whitespace (not just "-")
      std::string string0 = replaceString(isoString, "/", " ");
      std::string string1 = replaceString(isoString, "-", " ");

      // Now parse string.
      std::istringstream inputStream(string1);
      inputStream >> m_year >> m_month >> m_day;
      if(!inputStream) {
        std::ostringstream message;
        message << "Couldn't parse ISO date string: " << isoString << std::endl;
        DLR_THROW(ValueException, "Date::setISODate()", message.str().c_str());
      }

      // Make sure month and day are sane.
      this->sanitize();
    }

    // Returns the number of days in a given month.
    size_t Date::
    dayCount(size_t month, size_t year) {
      switch(month) {
      case 1:
      case 3:
      case 5:
      case 7:
      case 8:
      case 10:
      case 12:
        return 31;
        break;
      case 4:
      case 6:
      case 9:
      case 11:
        return 30;
        break;
      case 2:
        if(this->isLeapYear(year)) {
          return 29;
        }
        return 28;
        break;
      default:
        std::ostringstream message;
        message << "Invalid month: " << month << std::endl;
        DLR_THROW(ValueException, "Date::dayCount()", message.str().c_str());
        break;
      }
      DLR_THROW(LogicException, "Date::dayCount()", "Faulty switch logic\n");
      return 0;
    }

    // Chech for leapyear.
    bool Date::
    isLeapYear(size_t year)
    {
      if(year % 4 == 0) {
        return true;
      }
      return false;
    }

    // Make sure day and date numbers are calender-possible.
    void Date::
    sanitize()
    {
      while(1) {
        // Fix month first, since this might bump us into a leapyear
        while(m_month > 12) {
          m_month -= 12;
          m_year += 1;
        }
        // Done?
        if(m_day <= dayCount(m_month, m_year)) {
          break;
        }
        m_day -= dayCount(m_month, m_year);
        m_month++;
      }
    }

    bool
    operator<(const Date& date0, const Date& date1)
    {
      if(date0.year() > date1.year()) {
        return false;
      }
      if(date0.year() == date1.year()) {
        if(date0.month() > date1.month()) {
          return false;
        }
        if(date0.month() == date1.month()) {
          if(date0.day() >= date1.day()) {
            return false;
          }
        }
      }
      return true;
    }

    bool
    operator>(const Date& date0, const Date& date1)
    {
      return (!((date0 == date1) || (date0 < date1)));
    }

    bool
    operator<=(const Date& date0, const Date& date1)
    {
      return ((date0 == date1) || (date0 < date1));
    }

    bool
    operator>=(const Date& date0, const Date& date1)
    {
      return !(date0 < date1);
    }

    bool
    operator==(const Date& date0, const Date& date1)
    {
      return ((date0.year() == date1.year())
              && (date0.month() == date1.month())
              && (date0.day() == date1.day()));
    }

    bool
    operator!=(const Date& date0, const Date& date1)
    {
      return !(date0 == date1);
    }

    std::ostream&
    operator<<(std::ostream& stream, const Date& date)
    {
      stream << std::setfill('0') << std::setw(4) << date.year() << "-"
             << std::setw(2) << date.month() << "-"
             << std::setw(2) << date.day() << std::setfill(' ');
      return stream;
    }

    std::istream&
    operator>>(std::istream& stream, Date& date)
    {
      std::string isoString;
      stream >> isoString;

      // Can't read from stream if it's in a bad state.
      if(!stream) {
        return stream;
      }

      try {
        Date newDate(isoString);	// Make sure things parse OK before
        date = newDate;		// assigning to argument date.
      } catch(const ValueException&) {
        stream.clear(std::ios_base::failbit);
        return stream;
      }

      return stream;
    }

    
  } // namespace utilities
    
} // namespace dlr
