/**
***************************************************************************
* @file dlrUtilities/optionDescription.cpp
* Source file defining the OptionDescription class.
*
* Copyright (C) 2006 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 706 $
* $Date: 2006-08-04 19:41:11 -0400 (Fri, 04 Aug 2006) $
***************************************************************************
**/

#include <sstream>
#include <dlrUtilities/stringManipulation.h>
#include <dlrUtilities/optionDescription.h>

namespace dlr {

  namespace utilities {
  
    // Constructor.
    OptionDescription::
    OptionDescription()
      : m_name(),
        m_shortAppearance(),
        m_longAppearance(),
        m_requiresValue(),
        m_allowPartialMatch(),
        m_docString(),
        m_defaultValue()
    {
      // Empty.
    }
    
      
    // Constructor.
    OptionDescription::
    OptionDescription(const std::string& name,
                      const std::string& shortAppearance,
                      const std::string& longAppearance,
                      bool requiresValue,
                      bool allowPartialMatch,
                      const std::string& docString,
                      const std::string& defaultValue)
      : m_name(name),
        m_shortAppearance(shortAppearance),
        m_longAppearance(longAppearance),
        m_requiresValue(requiresValue),
        m_allowPartialMatch(allowPartialMatch),
        m_docString(docString),
        m_defaultValue(defaultValue)
    {
      // Empty.
    }


    // Destructor.
    OptionDescription::
    ~OptionDescription()
    {
      // Empty.
    }


    size_t
    OptionDescription::
    getMatchLength(const std::string& argument,
                   bool& isShortMatch) const
    {
      if(m_shortAppearance.size() != 0) {
        if((argument.size() == m_shortAppearance.size())
           && (argument == m_shortAppearance)) {
          isShortMatch = true;
          return argument.size();
        }
        if((argument.size() > m_shortAppearance.size())
           && (argument.compare(0, m_shortAppearance.size(), m_shortAppearance)
               == 0)) {
          isShortMatch = true;
          return m_shortAppearance.size();
        }
      }
      if(m_longAppearance.size() != 0) {
        // Split arguments of the form "--foobar=baz".
        std::vector<std::string> argumentParts =
          splitString(argument, "=", true, 1);
        if(argumentParts[0].size() == m_longAppearance.size()
           && argumentParts[0] == m_longAppearance) {
          if(argumentParts.size() == 1) {
            isShortMatch = false;
            return m_longAppearance.size();
          } else {
            isShortMatch = false;
            return m_longAppearance.size() + 1;
          }
        }
        if(m_allowPartialMatch
           && argumentParts[0].size() > 0
           && argumentParts[0].size() < m_longAppearance.size()
           && (m_longAppearance.compare(0, argumentParts[0].size(), argument)
               == 0)) {
          if(argumentParts.size() == 1) {
            isShortMatch = false;
            return argumentParts[0].size();
          } else {
            isShortMatch = false;
            return argumentParts[0].size() + 1;
          }
        }
      }
      isShortMatch = false;
      return 0;
    }


    std::string
    OptionDescription::
    getOptionDoc() const
    {
      std::string indentString = "        ";
      std::ostringstream docStream;
      if(m_shortAppearance != "") {
        if(m_requiresValue) {
          docStream << m_shortAppearance << " " << m_name;
        } else {
          docStream << m_shortAppearance;
        }
      }
      if(m_shortAppearance != "" && m_longAppearance != "") {
        docStream << ", ";
      }
      if(m_longAppearance != "") {
        if(m_requiresValue) {
          docStream << m_longAppearance << "=" << m_name;
        } else {
          docStream << m_longAppearance;
        }
      }
      docStream << "\n"
                << wrapString(indentString + m_docString, indentString);
      if(m_defaultValue != "") {
        docStream << "\n"
                  << wrapString(indentString + "Default value: \""
                                + m_defaultValue + "\"", indentString);
      }
      return docStream.str();
    }

    
    bool
    OptionDescription::
    isMatch(const std::string& argument) const
    {
      bool dummy;
      return (this->getMatchLength(argument, dummy) != 0);
    }

    
    std::ostream&
    operator<<(std::ostream& stream,
               const OptionDescription& optionDescription)
    {
      stream << "  " << optionDescription.getOptionDoc();
      return stream;
    }
    
      
  } // namespace utilities

} // namespace dlr
