/**
***************************************************************************
* @file pathTest.cpp
* Source file defining tests for file name handling.
*
* Copyright (C) 2004 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 923 $
* $Date: 2007-05-19 21:59:13 -0400 (Sat, 19 May 2007) $
***************************************************************************
**/

#include <string>

#include <dlrUtilities/path.h>
#include <dlrTest/testFixture.h>


// We disable a bunch of tests due to unsafe file IO.
#ifndef _DLR_UNITTEST_USE_UNSAFE_FILEIO_
#define _DLR_UNITTEST_USE_UNSAFE_FILEIO_ 0
#endif

namespace dlr {

  class PathTest : public TestFixture<PathTest> {

  public:

    PathTest();
    ~PathTest() {}

    void setUp(const std::string& testName) {}
    void tearDown(const std::string& testName) {}

    void testIsDirectory();
    void testJoinPath();
    void testListDirectory();
    void testRecursiveListDirectory();
    void testSplitExtension();
    void testSplitPath();

  }; // class PathTest


  /* ============== Member Function Definititions ============== */

  PathTest::
  PathTest()
    : TestFixture<PathTest>("PathTest")
  {
    DLR_TEST_REGISTER_MEMBER(testIsDirectory);
    DLR_TEST_REGISTER_MEMBER(testJoinPath);
    DLR_TEST_REGISTER_MEMBER(testListDirectory);
    DLR_TEST_REGISTER_MEMBER(testRecursiveListDirectory);
    DLR_TEST_REGISTER_MEMBER(testSplitExtension);
    DLR_TEST_REGISTER_MEMBER(testSplitPath);
  }


  void
  PathTest::
  testIsDirectory()
  {

#if _DLR_UNITTEST_USE_UNSAFE_FILEIO_

    // Set up a test directory.
    system("rm -rf _disposable_test_directory_");
    system("mkdir _disposable_test_directory_");
    system("touch _disposable_test_directory_/foo.txt");
    system("touch _disposable_test_directory_/bar.png");
    system("touch _disposable_test_directory_/baz.sxw");

    // Verify results.
    try{ 
      DLR_TEST_ASSERT(isDirectory("_disposable_test_directory_"));
      DLR_TEST_ASSERT(!isDirectory("_disposable_test_directory_/foo.txt"));
      DLR_TEST_ASSERT(!isDirectory("_disposable_test_directory_/bar.png"));
      DLR_TEST_ASSERT(!isDirectory("_disposable_test_directory_/baz.sxw"));
    } catch(const TestException&) {
      // Clean up.
      system("rm -rf _disposable_test_directory_");
      throw;
    }
    
    // Clean up.
    system("rm -rf _disposable_test_directory_");

#endif /* #if _DLR_UNITTEST_USE_UNSAFE_FILEIO_ */
    
  }


  void
  PathTest::
  testJoinPath()
  {
#ifdef _WIN32
    DLR_TEST_ASSERT(joinPath("foo", "bar.baz") == "foo\\bar.baz");
    DLR_TEST_ASSERT(joinPath("foo\\bar", "baz.hum") == "foo\\bar\\baz.hum");
    DLR_TEST_ASSERT(joinPath("foo\\bar", "") == "foo\\bar");
    DLR_TEST_ASSERT(joinPath("\\foo", "bar\\baz.hum") == "\\foo\\bar\\baz.hum");
    DLR_TEST_ASSERT(joinPath("", "bar\\baz.hum") == "bar\\baz.hum");
    DLR_TEST_ASSERT(joinPath("", "\\bar\\baz.hum") == "\\bar\\baz.hum");
    DLR_TEST_ASSERT(joinPath("\\foo", "bar\\baz.hum") == "\\foo\\bar\\baz.hum");
    DLR_TEST_ASSERT(joinPath("\\foo", "\\bar\\baz.hum") == "\\foo\\\\bar\\baz.hum");
    DLR_TEST_ASSERT(joinPath("foo\\bar", "\\baz.hum") == "foo\\bar\\\\baz.hum");

    DLR_TEST_ASSERT(joinPath("foo\\", "bar.baz") == "foo\\bar.baz");
    DLR_TEST_ASSERT(joinPath("foo\\bar\\", "baz.hum") == "foo\\bar\\baz.hum");
    DLR_TEST_ASSERT(joinPath("foo\\bar\\", "") == "foo\\bar\\");
    DLR_TEST_ASSERT(joinPath("\\foo\\", "bar\\baz.hum") == "\\foo\\bar\\baz.hum");
    DLR_TEST_ASSERT(joinPath("\\", "bar\\baz.hum") == "\\bar\\baz.hum");
    DLR_TEST_ASSERT(joinPath("\\", "\\bar\\baz.hum") == "\\\\bar\\baz.hum");
    DLR_TEST_ASSERT(joinPath("\\foo\\", "bar\\baz.hum") == "\\foo\\bar\\baz.hum");
    DLR_TEST_ASSERT(joinPath("\\foo\\", "\\bar\\baz.hum") == "\\foo\\\\bar\\baz.hum");
    DLR_TEST_ASSERT(joinPath("foo\\bar\\", "\\baz.hum") == "foo\\bar\\\\baz.hum");
#else
	DLR_TEST_ASSERT(joinPath("foo", "bar.baz") == "foo/bar.baz");
    DLR_TEST_ASSERT(joinPath("foo/bar", "baz.hum") == "foo/bar/baz.hum");
    DLR_TEST_ASSERT(joinPath("foo/bar", "") == "foo/bar");
    DLR_TEST_ASSERT(joinPath("/foo", "bar/baz.hum") == "/foo/bar/baz.hum");
    DLR_TEST_ASSERT(joinPath("", "bar/baz.hum") == "bar/baz.hum");
    DLR_TEST_ASSERT(joinPath("", "/bar/baz.hum") == "/bar/baz.hum");
    DLR_TEST_ASSERT(joinPath("/foo", "bar/baz.hum") == "/foo/bar/baz.hum");
    DLR_TEST_ASSERT(joinPath("/foo", "/bar/baz.hum") == "/foo//bar/baz.hum");
    DLR_TEST_ASSERT(joinPath("foo/bar", "/baz.hum") == "foo/bar//baz.hum");

    DLR_TEST_ASSERT(joinPath("foo/", "bar.baz") == "foo/bar.baz");
    DLR_TEST_ASSERT(joinPath("foo/bar/", "baz.hum") == "foo/bar/baz.hum");
    DLR_TEST_ASSERT(joinPath("foo/bar/", "") == "foo/bar/");
    DLR_TEST_ASSERT(joinPath("/foo/", "bar/baz.hum") == "/foo/bar/baz.hum");
    DLR_TEST_ASSERT(joinPath("/", "bar/baz.hum") == "/bar/baz.hum");
    DLR_TEST_ASSERT(joinPath("/", "/bar/baz.hum") == "//bar/baz.hum");
    DLR_TEST_ASSERT(joinPath("/foo/", "bar/baz.hum") == "/foo/bar/baz.hum");
    DLR_TEST_ASSERT(joinPath("/foo/", "/bar/baz.hum") == "/foo//bar/baz.hum");
    DLR_TEST_ASSERT(joinPath("foo/bar/", "/baz.hum") == "foo/bar//baz.hum");
#endif
  }

  
  void
  PathTest::
  testListDirectory()
  {

#if _DLR_UNITTEST_USE_UNSAFE_FILEIO_

    // Set up a test directory.
    system("rm -rf _disposable_test_directory_");
    system("mkdir _disposable_test_directory_");
    system("touch _disposable_test_directory_/foo.txt");
    system("touch _disposable_test_directory_/bar.png");
    system("touch _disposable_test_directory_/baz.sxw");

    // List the contents and sort them into a predictable order.
    std::vector<std::string> listing =
      listDirectory("_disposable_test_directory_");
    std::sort(listing.begin(), listing.end());

    // Verify results.
    try{ 
      DLR_TEST_ASSERT(listing.size() == 5);
      DLR_TEST_ASSERT(listing[0] == ".");
      DLR_TEST_ASSERT(listing[1] == "..");
      DLR_TEST_ASSERT(listing[2] == "bar.png");
      DLR_TEST_ASSERT(listing[3] == "baz.sxw");
      DLR_TEST_ASSERT(listing[4] == "foo.txt");
    } catch(const TestException&) {
      // Clean up.
      system("rm -rf _disposable_test_directory_");
      throw;
    }
    
    // List the contents again and sort them into a predictable order.
    // This time ask for full path.
    listing =
      listDirectory("_disposable_test_directory_", true);
    std::sort(listing.begin(), listing.end());

    // Verify results.
    try{ 
      DLR_TEST_ASSERT(listing.size() == 5);
      DLR_TEST_ASSERT(listing[0] == "_disposable_test_directory_/.");
      DLR_TEST_ASSERT(listing[1] == "_disposable_test_directory_/..");
      DLR_TEST_ASSERT(listing[2] == "_disposable_test_directory_/bar.png");
      DLR_TEST_ASSERT(listing[3] == "_disposable_test_directory_/baz.sxw");
      DLR_TEST_ASSERT(listing[4] == "_disposable_test_directory_/foo.txt");
    } catch(const TestException&) {
      // Clean up.
      system("rm -rf _disposable_test_directory_");
      throw;
    }
    
    // Clean up.
    system("rm -rf _disposable_test_directory_");

#endif /* #if _DLR_UNITTEST_USE_UNSAFE_FILEIO_ */

  }

  
  void
  PathTest::
  testRecursiveListDirectory()
  {

#if _DLR_UNITTEST_USE_UNSAFE_FILEIO_

    // Set up a test directory.
    system("rm -rf _disposable_test_directory_");
    system("mkdir _disposable_test_directory_");
    system("touch _disposable_test_directory_/foo.txt");
    system("touch _disposable_test_directory_/hoo.sxw");
    system("mkdir _disposable_test_directory_/subdirectory0");
    system("touch _disposable_test_directory_/subdirectory0/bar.png");
    system("touch _disposable_test_directory_/subdirectory0/baz.png");
    system("mkdir _disposable_test_directory_/subdirectory0/subdirectory1");

    // List the contents and sort them into a predictable order.
    std::vector<std::string> listing =
      recursiveListDirectory("_disposable_test_directory_");
    std::sort(listing.begin(), listing.end());

    // Verify results.
    try{
      size_t index0 = 0;
      DLR_TEST_ASSERT(listing.size() == 4);
      DLR_TEST_ASSERT(listing[index0++] == "foo.txt");
      DLR_TEST_ASSERT(listing[index0++] == "hoo.sxw");
      DLR_TEST_ASSERT(listing[index0++] == "subdirectory0/bar.png");
      DLR_TEST_ASSERT(listing[index0++] == "subdirectory0/baz.png");
    } catch(const TestException&) {
      // Clean up.
      // system("rm -rf _disposable_test_directory_");
      throw;
    }
    
    // List the contents and sort them into a predictable order.  This
    // time ask for directory names to be included.
    listing =
      recursiveListDirectory("_disposable_test_directory_", false, true);
    std::sort(listing.begin(), listing.end());

    // Verify results.
    try{ 
      size_t index0 = 0;
      DLR_TEST_ASSERT(listing.size() == 12);
      DLR_TEST_ASSERT(listing[index0++] == ".");
      DLR_TEST_ASSERT(listing[index0++] == "..");
      DLR_TEST_ASSERT(listing[index0++] == "foo.txt");
      DLR_TEST_ASSERT(listing[index0++] == "hoo.sxw");
      DLR_TEST_ASSERT(listing[index0++] == "subdirectory0");
      DLR_TEST_ASSERT(listing[index0++] == "subdirectory0/.");
      DLR_TEST_ASSERT(listing[index0++] == "subdirectory0/..");
      DLR_TEST_ASSERT(listing[index0++] == "subdirectory0/bar.png");
      DLR_TEST_ASSERT(listing[index0++] == "subdirectory0/baz.png");
      DLR_TEST_ASSERT(listing[index0++] == "subdirectory0/subdirectory1");
      DLR_TEST_ASSERT(listing[index0++] == "subdirectory0/subdirectory1/.");
      DLR_TEST_ASSERT(listing[index0++] == "subdirectory0/subdirectory1/..");
    } catch(const TestException&) {
      // Clean up.
      system("rm -rf _disposable_test_directory_");
      throw;
    }
    
    // List the contents and sort them into a predictable order.  This
    // time ask for directory names to be included and request full
    // path name.
    listing =
      recursiveListDirectory("_disposable_test_directory_", true, true);
    std::sort(listing.begin(), listing.end());

    // Verify results.
    try{ 
      size_t index0 = 0;
      DLR_TEST_ASSERT(listing.size() == 12);
      DLR_TEST_ASSERT(listing[index0++]
                      == "_disposable_test_directory_/.");
      DLR_TEST_ASSERT(listing[index0++]
                      == "_disposable_test_directory_/..");
      DLR_TEST_ASSERT(listing[index0++]
                      == "_disposable_test_directory_/foo.txt");
      DLR_TEST_ASSERT(listing[index0++]
                      == "_disposable_test_directory_/hoo.sxw");
      DLR_TEST_ASSERT(listing[index0++]
                      == "_disposable_test_directory_/subdirectory0");
      DLR_TEST_ASSERT(listing[index0++]
                      == "_disposable_test_directory_/subdirectory0/.");
      DLR_TEST_ASSERT(listing[index0++]
                      == "_disposable_test_directory_/subdirectory0/..");
      DLR_TEST_ASSERT(listing[index0++]
                      == "_disposable_test_directory_/subdirectory0/bar.png");
      DLR_TEST_ASSERT(listing[index0++]
                      == "_disposable_test_directory_/subdirectory0/baz.png");
      DLR_TEST_ASSERT(
        listing[index0++]
        == "_disposable_test_directory_/subdirectory0/subdirectory1");
      DLR_TEST_ASSERT(
        listing[index0++]
        == "_disposable_test_directory_/subdirectory0/subdirectory1/.");
      DLR_TEST_ASSERT(
        listing[index0++]
        == "_disposable_test_directory_/subdirectory0/subdirectory1/..");
    } catch(const TestException&) {
      // Clean up.
      system("rm -rf _disposable_test_directory_");
      throw;
    }
    
    // Clean up.
    system("rm -rf _disposable_test_directory_");

#endif /* #if _DLR_UNITTEST_USE_UNSAFE_FILEIO_ */

  }


  void
  PathTest::
  testSplitExtension()
  {
#ifdef _WIN32
    // Try splitting a normal looking filename.
    std::pair<std::string, std::string> basename_extension =
      splitExtension("\\foo\\bar.baz");
    DLR_TEST_ASSERT(basename_extension.first == "\\foo\\bar");
    DLR_TEST_ASSERT(basename_extension.second == ".baz");

    // Make sure we don't get faked out by non-extension '.'
    // characters.
    basename_extension = splitExtension("\\foo\\bar.j\\baz");
    DLR_TEST_ASSERT(basename_extension.first == "\\foo\\bar.j\\baz");
    DLR_TEST_ASSERT(basename_extension.second == "");

    // Check with a filename that's nothing but an extension.
    basename_extension = splitExtension(".baz");
    DLR_TEST_ASSERT(basename_extension.first == "");
    DLR_TEST_ASSERT(basename_extension.second == ".baz");
#else
	// Try splitting a normal looking filename.
    std::pair<std::string, std::string> basename_extension =
      splitExtension("/foo/bar.baz");
    DLR_TEST_ASSERT(basename_extension.first == "/foo/bar");
    DLR_TEST_ASSERT(basename_extension.second == ".baz");

    // Make sure we don't get faked out by non-extension '.'
    // characters.
    basename_extension = splitExtension("/foo/bar.j/baz");
    DLR_TEST_ASSERT(basename_extension.first == "/foo/bar.j/baz");
    DLR_TEST_ASSERT(basename_extension.second == "");

    // Check with a filename that's nothing but an extension.
    basename_extension = splitExtension(".baz");
    DLR_TEST_ASSERT(basename_extension.first == "");
    DLR_TEST_ASSERT(basename_extension.second == ".baz");
#endif
  }

  void
  PathTest::
  testSplitPath()
  {
    std::pair<std::string, std::string> directoryName_fileName;

#ifdef _WIN32
    directoryName_fileName = splitPath("\\foo\\bar\\baz.hum");
    DLR_TEST_ASSERT(directoryName_fileName.first == "\\foo\\bar\\");
    DLR_TEST_ASSERT(directoryName_fileName.second == "baz.hum");

    directoryName_fileName = splitPath("bar\\baz.hum");
    DLR_TEST_ASSERT(directoryName_fileName.first == "bar\\");
    DLR_TEST_ASSERT(directoryName_fileName.second == "baz.hum");

    directoryName_fileName = splitPath("baz.hum");
    DLR_TEST_ASSERT(directoryName_fileName.first == "");
    DLR_TEST_ASSERT(directoryName_fileName.second == "baz.hum");

    directoryName_fileName = splitPath("\\foo\\bar\\baz.hum\\");
    DLR_TEST_ASSERT(directoryName_fileName.first == "\\foo\\bar\\baz.hum\\");
    DLR_TEST_ASSERT(directoryName_fileName.second == "");
#else
    directoryName_fileName = splitPath("/foo/bar/baz.hum");
    DLR_TEST_ASSERT(directoryName_fileName.first == "/foo/bar/");
    DLR_TEST_ASSERT(directoryName_fileName.second == "baz.hum");

    directoryName_fileName = splitPath("bar/baz.hum");
    DLR_TEST_ASSERT(directoryName_fileName.first == "bar/");
    DLR_TEST_ASSERT(directoryName_fileName.second == "baz.hum");

    directoryName_fileName = splitPath("baz.hum");
    DLR_TEST_ASSERT(directoryName_fileName.first == "");
    DLR_TEST_ASSERT(directoryName_fileName.second == "baz.hum");

    directoryName_fileName = splitPath("/foo/bar/baz.hum/");
    DLR_TEST_ASSERT(directoryName_fileName.first == "/foo/bar/baz.hum/");
    DLR_TEST_ASSERT(directoryName_fileName.second == "");
#endif
  }
  
} // namespace dlr


#ifdef DLR_TEST_NO_AUTOMATIC_REGISTRATION

int main(int argc, char** argv)
{
  dlr::PathTest currentTest;
  bool result = currentTest.run();
  return (result ? 0 : 1);
}

#else /* #ifdef DLR_TEST_NO_AUTOMATIC_REGISTRATION */

namespace {

  dlr::PathTest currentTest;

}

#endif /* #ifdef DLR_TEST_NO_AUTOMATIC_REGISTRATION */
