/**
***************************************************************************
* @file dlrUtilities/path.h
*
* Header file declaring routines for working with the filesystem
*
* Copyright (C) 2003-2007, David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 885 $
* $Date: 2007-05-04 01:01:15 -0400 (Fri, 04 May 2007) $
***************************************************************************
**/

#ifndef _DLR_PATH_H_
#define _DLR_PATH_H_

#include <string>
#include <vector>

namespace dlr {

  namespace utilities {

    /** 
     * This function returns a bool indicating whether or not the
     * specified path is a directory.  If the path is a symbolic link,
     * the return value is currently unspecified, but will eventually be
     * true iff the link points to a directory.
     * 
     * @param path This argument is the filename to evaluate.
     * 
     * @return The return value is true if path refers to a directory,
     * false otherwise.
     */
    bool
    isDirectory(const std::string& path);
  
  
    bool
    isExistingPath(const std::string& path);
  
  
    bool
    isRegularFile(const std::string& path);
    
    
    /**
     ** Joins two path elements with the appropriate path delimiter.
     ** For example:
     **   joinPath("foo", "bar");
     ** might give
     **   "foo/bar"
     ** while
     **   joinPath("foo/baz/", "bar");
     ** might give
     **   "/foo/baz/bar"
     **/
    std::string
    joinPath(const std::string& part0, const std::string& part1);

  
    /**
     ** Returns the names of the entries in the specified directory, in
     ** no particular order.  For example, 
     **   listDirectory("/etc");
     ** might give
     **   ["fstab", "init.d", "modules.conf", ...]
     ** while
     **   listDirectory("/etc", true);
     ** might give
     **   ["/etc/fstab", "/etc/init.d", "/etc/modules.conf", ...]
     **/
    std::vector<std::string>
    listDirectory(const std::string& directoryName, bool fullPath=false);


    /** 
     * Returns the names of files in the directory tree below the
     * specified directory.  For example,
     *
     *   recursiveListDirectory("/etc");
     *
     * might give
     *
     *   ["fstab", "init.d/cron", "init.d/ssh", "modules.conf", ...]
     *
     * while
     *
     *   recursiveListDirectory("/etc", true);
     *
     * might give
     *
     *   ["fstab", "init.d", "init.d/cron", "init.d/ssh", "modules.conf", ...]
     *
     * and
     *
     *   recursiveListDirectory("/etc", true, true);
     *
     * might give
     *
     *   ["/etc/fstab", "/etc/init.d", "/etc/init.d/cron", "/etc/init.d/ssh",
     *    "/etc/modules.conf", ...]
     *
     * @param directoryName This argument specifies the directory to be listed.
     * 
     * @return The return value is a vector of file names.
     */
    std::vector<std::string>
    recursiveListDirectory(const std::string& directoryName,
                           bool fullPath=false,
                           bool includeDirectoryNames=false);
  
  
    /**
     ** Returns a std::pair<std::string, std::string> containing the fileName
     ** without its extension, and the extension.  For example:
     **   splitExt("/foo/bar.baz")
     ** might return
     **   {"/foo/bar", ".baz"}
     **/
    std::pair<std::string, std::string>
    splitExtension(const std::string& fileName);


    /** 
     * This function accepts a path returns a pair of strings in which
     * the first element is the directory name and the second is the
     * filename.  For example:
     *
     *   splitPath("/foo/bar.baz")
     *
     * might return
     *
     *   {"/foo/", "bar.baz"}
     *
     * Also,
     *
     *   splitPath("bar.baz")
     *
     * might return
     *
     *   {"", "bar.baz"}
     *
     *   splitPath("/foo/")
     *
     * might return
     *
     *   {"/foo/", ""}
     *
     * 
     * @return The return value is a pair containing first the directory
     * name, and second the file name.
     */
    std::pair<std::string, std::string>
    splitPath(const std::string& path);

  } // namespace utilities
    
} // namespace dlr


/* ======= Declarations to maintain compatibility with legacy code. ======= */

namespace dlr {

  using utilities::isDirectory;
  using utilities::isExistingPath;
  using utilities::isRegularFile;
  using utilities::joinPath;
  using utilities::listDirectory;
  using utilities::recursiveListDirectory;
  using utilities::splitExtension;
  using utilities::splitPath;
  
} // namespace dlr


#endif // #ifndef _DLR_PATH_H_
