/**
***************************************************************************
* @file dlrUtilities/path.cpp
*
* Source file defining routines for working with the filesystem
*
* Copyright (C) 2003-2007, David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 885 $
* $Date: 2007-05-04 01:01:15 -0400 (Fri, 04 May 2007) $
***************************************************************************
**/

#include <sys/types.h>
#include <sys/stat.h>
#include <algorithm>
#include <list>
#include <dlrCommon/functional.h>
#include <dlrPortability/filesystem.h>
#include <dlrUtilities/path.h>

namespace dlr {

  namespace utilities {

    using dlr::portability::pathDelimiter;
    using dlr::portability::extensionDelimiter;

  
    // This function returns a bool indicating whether or not the
    // specified path is a directory.
    bool
    isDirectory(const std::string& path)
    {
      return dlr::portability::isDirectory(path);
    }
  

    bool
    isExistingPath(const std::string& path)
    {
      struct stat statBuf;
      memset(&statBuf, 0, sizeof(statBuf));
      int returnValue = stat(path.c_str(), &statBuf);
      if(returnValue == 0) {
        return true;
      }
      return false;
    }
    

    // This function returns a bool indicating whether or not the
    // specified path is a regular file.
    bool
    isRegularFile(const std::string& path)
    {
      struct stat statBuf;
      memset(&statBuf, 0, sizeof(statBuf));
      int returnValue = stat(path.c_str(), &statBuf);
      if(returnValue == 0) {
        // if(S_ISREG(statBuf.st_mode)) {
        if((statBuf.st_mode & S_IFREG) == S_IFREG) {
          return true;
        }
      }
      return false;
    }


    // Joins two path elements with the appropriate delimiter.
    std::string
    joinPath(const std::string& part0, const std::string& part1)
    {
      return dlr::portability::joinPath(part0, part1);
    }

  
    // Returns the names of the entries in the specified directory, in
    // no particular order.
    std::vector<std::string>
    listDirectory(const std::string& directoryName, bool fullPath)
    {
      // Dispatch to dlrPortability function.
      return dlr::portability::listDirectory(directoryName, fullPath);
    }


    // Returns the names of files in the directory tree below the
    // specified directory.
    std::vector<std::string>
    recursiveListDirectory(const std::string& directoryName,
                           bool fullPath,
                           bool includeDirectoryNames)
    {
      typedef std::vector<std::string>::const_iterator FileNameIter;
      typedef PointerToBinaryFunctionRA<std::string, std::string, std::string>
        functorType;

      // We'll accumulate the total listing in fileNameList.
      std::list<std::string> fileNameList;

      // For starters, just find the entries in the specified directory.
      std::vector<std::string> topLevelFileNameList =
        listDirectory(directoryName);

      // Process each entry in turn.
      for(FileNameIter fileNameIter = topLevelFileNameList.begin();
          fileNameIter != topLevelFileNameList.end();
          ++fileNameIter) {

        std::string fullName = joinPath(directoryName, *fileNameIter);
        if(isDirectory(fullName)) {
          // Looks like this entry is a directory.  Add it, if
          // appropriate.
          if(includeDirectoryNames) {
            fileNameList.push_back(*fileNameIter);
          }

          if(*fileNameIter != "." && *fileNameIter != "..") {
            // Recurse into non-trivial directory entries, and add the
            // result to our accumulated list.
            std::vector<std::string> subListing =
              recursiveListDirectory(fullName, false, includeDirectoryNames);
            std::transform(subListing.begin(), subListing.end(),
                           std::back_inserter(fileNameList),
                           std::bind1st(functorType(joinPath), *fileNameIter));
          }
        } else {  // if(isDirectory(...))
          fileNameList.push_back(*fileNameIter);
        }
      }
    
      // Now copy the file names into a vector.  Adding the full path,
      // if appriate.
      std::vector<std::string> finalVector(fileNameList.size());
      if(fullPath) {
        std::transform(fileNameList.begin(), fileNameList.end(),
                       finalVector.begin(),
                       std::bind1st(functorType(joinPath), directoryName));
      } else {
        std::copy(fileNameList.begin(), fileNameList.end(), finalVector.begin());
      }
      return finalVector;
    }

  
    // Returns a std::pair<std::string, std::string> containing the fileName
    // without its extension, and the extension.
    std::pair<std::string, std::string>
    splitExtension(const std::string& fileName)
    {
      std::string::size_type extensionIndex =
        fileName.rfind(extensionDelimiter());
      if(extensionIndex == std::string::npos) {
        return std::make_pair(fileName, std::string(""));
      }
      std::string::size_type pathIndex =
        fileName.rfind(pathDelimiter());
      if((pathIndex != std::string::npos)
         && (pathIndex > extensionIndex)) {
        return std::make_pair(fileName, std::string(""));
      }
      return std::make_pair(fileName.substr(0, extensionIndex),
                            fileName.substr(extensionIndex, std::string::npos));
    }

  
    // This function accepts a path returns a pair of strings in which
    // the first element is the directory name and the second is the
    // filename.
    std::pair<std::string, std::string>
    splitPath(const std::string& path)
    {
      return dlr::portability::splitPath(path);
    }

  } // namespace utilities
  
} // namespace dlr
