/**
***************************************************************************
* @file threadFunctor.cpp
* Source file defining ThreadFunctor class.
*
* Copyright (C) 2005 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 975 $
* $Date: 2007-12-30 01:57:17 -0500 (Sun, 30 Dec 2007) $
***************************************************************************
**/

#include <exception>
#include <iostream>
#include <sstream>
#include <dlrCommon/exception.h>
#include <dlrThread/threadFunctor.h>

namespace {

  /**
   ** This adaptor calls ThreadFunctor::operator()() through a
   ** pointer, allowing the virtual function dispatch to work from
   ** within a boost::threads call.
   **/
  struct TFAdaptor {

    TFAdaptor(dlr::thread::ThreadFunctor* threadFunctorPtr)
      : m_threadFunctorPtr(threadFunctorPtr) {}

    void operator()() {m_threadFunctorPtr->operator()();}

  private:
    dlr::thread::ThreadFunctor* m_threadFunctorPtr;
  };
      
} // namespace


namespace dlr {

  namespace thread {
    
    // Destroys the ThreadFunctor instance and releases any resources.
    ThreadFunctor::
    ~ThreadFunctor()
    {
      if(m_threadPtr != 0) {
        // Warning(xxx): Shouldn't we call thread::terminate() or
        // something?
        delete m_threadPtr;
      }
    }

    
    // The assignment operator does a deep copy.
    ThreadFunctor&
    ThreadFunctor::
    operator=(const ThreadFunctor& source)
    {
      if(&source != this) {
        if(m_threadPtr != 0) {
          std::ostringstream message;
          message << "[" << m_threadName << "] "
                  << "Can't complete assignment while a subordinate thread "
                  << "is running.  Please call the join() method of the "
                  << "assignee prior to assignment."; 
          DLR_THROW(StateException, "ThreadFunctor::operator=()",
                    message.str().c_str());
        }
        m_threadName = source.m_threadName;
      }
      return *this;
    }

    
    // This operator executes this->main(), catches any exceptions
    // thrown by this->main(), and handles those exceptions by
    // calling this->handleError().
    void
    ThreadFunctor::
    operator()() {
      try {
        this->main();
      } catch(const dlr::Exception& caughtException) {
        this->handleError(caughtException.what(),
                          caughtException.trace());
      } catch(const std::exception& caughtException) {
        this->handleError(caughtException.what(), "");
      } catch(...) {
        this->handleError(
          "Unknown exception.  No further information available.", "");
      }
    }


    // This member function is called if an exception is thrown by
    // this->main(), and sends notification that the thread has (is
    // about to) terminate.
    void
    ThreadFunctor::
    handleError(const std::string& errorMessage,
                const std::string& errorDetail)
    {
      std::cerr << "WARNING: Unhandled exception in thread "
                << m_threadName << ".\n"
                << "  " << errorMessage << "\n";
      if(errorDetail != "") {
        std::cerr << errorDetail << "\n";
      }
      std::cerr << "=============== Ending thread " << m_threadName
                << ". ===============\n"
                << std::endl;
    }        


    // This member function blocks until the thread associated with
    // *this exits.
    void
    ThreadFunctor::
    join()
    {
      if(m_threadPtr == 0) {
        std::ostringstream message;
        message << "[" << m_threadName << "] "
                << "No subordinate thread is currently running.";
        DLR_THROW(StateException, "ThreadFunctor::join()",
                  message.str().c_str());
      }
      m_threadPtr->join();
      delete m_threadPtr;
      m_threadPtr = 0;        
    }


    // This member function creates a new thread and executes
    // this->operator()() from within that thread.
    void
    ThreadFunctor::
    run()
    {
      if(m_threadPtr != 0) {
        std::ostringstream message;
        message << "[" << m_threadName << "] "
                << "Can't start a new subordinate thread because a "
                << "subordinate thread is already running.";
          DLR_THROW(StateException, "ThreadFunctor::run()",
                    message.str().c_str());
      }
      m_threadPtr = new boost::thread(TFAdaptor(this));
    }


  } // namespace thread

} // namespace dlr
