/**
***************************************************************************
* @file dlrThread/threadFunctor.h 
* Header file declaring ThreadFunctor class.
*
* Copyright (C) 2005 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 886 $
* $Date: 2007-05-04 01:59:27 -0400 (Fri, 04 May 2007) $
***************************************************************************
**/

#ifndef _DLR_THREAD_THREADFUNCTOR_H_
#define _DLR_THREAD_THREADFUNCTOR_H_

#include <string>
#include <boost/thread.hpp>

namespace dlr {

  namespace thread {

    /**
     ** This abstract base class provides basic thread management.
     ** Derived classes should define the main() method so that it
     ** runs whatever code is to be executed by the thread.  Derived
     ** classes may optionally define a handleError() method which
     ** will be called if the main() method throws an exception.  The
     ** default version of handleError() simply prints a message to
     ** standard error.  The run() method will start the new thread,
     ** and the join() method will block until that thread's main()
     ** method finishes.  Here's an example of how to use
     ** ThreadFunctor:
     **
     ** @code
     ** class MyThreadFunctor : public dlr::ThreadFunctor {
     ** public:
     **   MyThreadFunctor(const std::string& threadName)
     **     : dlr::ThreadFunctor(threadName) {}
     **
     **   void main() {
     **     // Do stuff.
     **   }
     ** };
     **
     **
     ** // Later, in application code...
     ** MyThreadFunctor threadFunctor0("Thread 0");
     ** MyThreadFunctor threadFunctor1("Thread 1");
     ** threadFunctor0.run();
     ** threadFunctor1.run();
     **
     ** // Later still...
     ** threadFunctor0.join();
     ** threadFunctor1.join();
     ** @endcode
     **
     ** It's OK to copy a ThreadFunctor instance, but you must
     ** remember that the copy does not refer to the same thread as
     ** the original.  This can cause some subtle errors.  For
     ** example, this code is broken:
     **
     ** @code
     ** std::vector<MyThreadFunctor> threadFunctorVector;
     ** threadFunctorVector.push_back(MyThreadFunctor("Thread 0"));
     ** threadFunctorVector[0].run();
     ** threadFunctorVector.push_back(MyThreadFunctor("Thread 1"));
     ** threadFunctorVector[1].run();
     ** threadFunctorVector[0].join();
     ** threadFunctorVector[1].join();
     ** @endcode
     **
     ** The error is that the second call to push_back() may copy the
     ** already running thread, and the calls to join() won't match
     ** up.  The following code is OK:
     **
     ** @code
     ** std::vector<MyThreadFunctor> threadFunctorVector;
     ** threadFunctorVector.push_back(MyThreadFunctor("Thread 0"));
     ** threadFunctorVector.push_back(MyThreadFunctor("Thread 1"));
     ** threadFunctorVector[0].run();
     ** threadFunctorVector[1].run();
     ** threadFunctorVector[0].join();
     ** threadFunctorVector[1].join();
     ** @endcode
     **/
    class ThreadFunctor
    {
    public:

      /** 
       * This constructor initializes the ThreadFunctor instance and
       * sets the thread name.  It is encouraged to use a unique
       * thread name for each thread, and to include the thread name
       * in any exception messages, since this will make debugging
       * easier.
       * 
       * @param threadName This argument 
       */
      ThreadFunctor(const std::string& threadName="anonymous")
        : m_threadName(threadName), m_threadPtr(0) {};


      /** 
       * The copy constructor does a deep copy.  Note that it does not
       * change the thread name, so after copying a ThreadFunctor
       * instance it may be useful to call the setName() method with a
       * unique name.
       * 
       * @param source The ThreadFunctor instance to be copied.
       */
      ThreadFunctor(const ThreadFunctor& source)
        : m_threadName(source.m_threadName), m_threadPtr(0) {};


      /** 
       * Destroys the ThreadFunctor instance and releases any
       * resources.
       */
      virtual
      ~ThreadFunctor();


      /** 
       * The assignment operator does a deep copy.  Note that it does
       * not change the thread name, so after copying a ThreadFunctor
       * instance it may be useful to call the setName() method with a
       * unique name.  It is an error to call this member function
       * while a subordinate thread is running.  That is, if you've
       * called this->run(), you must call this->join() before calling
       * this->operator=().
       * 
       * @param source The ThreadFunctor instance to be copied.
       * @return A reference to *this.
       */
      ThreadFunctor&
      operator=(const ThreadFunctor& source);


      /** 
       * This operator executes this->main(), catches any exceptions
       * thrown by this->main(), and handles those exceptions by
       * calling this->handleError().  It is normally not necessary to
       * call this member function directly.  Call this->run() instead
       * to create a new thread and automatically call
       * this->operator()() from within that thread.
       */
      virtual void
      operator()();


      /** 
       * This member function returns the name of the thread.
       * 
       * @return The return value is the name assigned to the thread
       * by constructor argument threadName.
       */
      const std::string&
      getThreadName() {return m_threadName;}

      
      /** 
       * This member function is called if an exception is thrown by
       * this->main() to send notification that the thread is about to
       * terminate.  The default implementation simply prints to the
       * standard error output.  Subclasses may override this function
       * to handle the error in other ways.
       * 
       * @param errorMessage This argument is a short string
       * describing the error.  It is generated from the what()
       * message of the caught exception.
       * 
       * @param errorDetail This argument is a longer string
       * containing more details about the error.  If such details are
       * not available, it is set to the empty string.  It is
       * generated from the trace() message of a caught
       * dlr::Exception instance.
       */
      virtual void
      handleError(const std::string& errorMessage,
                  const std::string& errorDetail);


      /** 
       * This member function blocks until the thread associated with
       * *this exits.  It is an error to call join() without first
       * calling run() to start the thread.
       */
      void
      join();


      /** 
       * This pure virtual function should be overridden by the
       * subclass to contain whatever code the ThreadFunctor instance
       * is responsible for running.
       */
      virtual void
      main() = 0;


      /** 
       * This member function creates a new thread and executes
       * this->operator()() from within that thread.  It is an error
       * to call run() twice without calling join() in between.
       */
      void
      run();
      

      /** 
       * This member function sets the name of the thread.  The thread
       * name will be included in the what() message of any exception
       * thrown by ThreadFunctor, and it is encouraged to include it
       * in the messages of any exceptions thrown by derived class
       * member functions.
       * 
       * @param threadName This argument specifies a name for the
       * thread.
       */
      void
      setName(const std::string& threadName) {m_threadName = threadName;}

    protected:

      std::string m_threadName;
      boost::thread* m_threadPtr;
    };

  } // namespace thread

} // namespace dlr

#endif /* #ifndef _DLR_THREAD_THREADFUNCTOR_H_ */
