/**
***************************************************************************
* @file dlrPortability/filesystem.h
*
* Header file declaring portability routines for dealing with filesystems.
*
* Copyright (C) 2003-2005, David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 882 $
* $Date: 2007-05-04 00:40:09 -0400 (Fri, 04 May 2007) $
***************************************************************************
**/


#ifndef _DLRPORTABILITY_FILESYSTEM_H_
#define _DLRPORTABILITY_FILESYSTEM_H_

#include <string>
#include <vector>

// Anonymous namespace for stuff local to this file.
namespace dlr {

  namespace portability {
    
    const std::string& pathDelimiter();
  

    const std::string& extensionDelimiter();


    /** 
     * This function returns a bool indicating whether or not the
     * specified path is a directory.  If the path is a symbolic link,
     * the return value is currently unspecified, but will eventually be
     * true iff the link points to a directory.
     * 
     * @param path This argument is the filename to evaluate.
     * 
     * @return The return value is true if path refers to a directory,
     * false otherwise.
     */
    bool
    isDirectory(const std::string& path);
    

    std::string
    joinPath(const std::string& part0, const std::string& part1);


    std::vector<std::string>
    listDirectory(const std::string& directoryName, bool fullPath);


    /** 
     * This function accepts a path returns a pair of strings in which
     * the first element is the directory name and the second is the
     * filename.  For example:
     *
     *   splitPath("/foo/bar.baz")
     *
     * might return
     *
     *   {"/foo/", "bar.baz"}
     *
     * Also,
     *
     *   splitPath("bar.baz")
     *
     * might return
     *
     *   {"", "bar.baz"}
     *
     *   splitPath("/foo/")
     *
     * might return
     *
     *   {"/foo/", ""}
     *
     * 
     * @return The return value is a pair containing first the directory
     * name, and second the file name.
     */
    std::pair<std::string, std::string>
    splitPath(const std::string& path);
    
  } // namespace portability
  
} // namespace dlr

#endif /* ifndef _DLRPORTABILITY_FILESYSTEM_H_ */
