/**
***************************************************************************
* @file dlrNumeric/vector3D.h
*
* Header file declaring Vector3D class.
*
* Copyright (C) 2000-2007 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 1097 $
* $Date: 2009-01-20 01:12:51 -0500 (Tue, 20 Jan 2009) $
***************************************************************************
**/

#ifndef _DLR_VECTOR3D_H_
#define _DLR_VECTOR3D_H_

#include <iostream>
#include <dlrCommon/exception.h>

namespace dlr {

  namespace numeric {
    
    /**
     ** The Vector2D class represents a real valued 2D vector.
     **/
    class Vector3D {
    public:
      /** 
       *  Default constructor initializes to (0, 0, 0).
       */
      Vector3D()
        : m_x(0), m_y(0), m_z(0) {}

      /** 
       * Explicitly sets 3D coordinates.
       * 
       * @param xCoord After construction, the vector will have this value as
       * its X coordinate.
       * @param yCoord After construction, the vector will have this value as
       * its Y coordinate.
       * @param zCoord After construction, the vector will have this value as
       * its Z coordinate.
       */
      Vector3D(double xCoord, double yCoord, double zCoord)
        : m_x(xCoord), m_y(yCoord), m_z(zCoord) {}

      /** 
       * Explicitly sets 3D homogeneous coordinates.  A 3D homogeneous
       * vector has the form (xCoord, yCoord, zCoord, alpha), and corresponds to the 3D
       * point (xCoord/alpha, yCoord/alpha, zCoord/alpha).
       * 
       * @param xCoord The homogeneous X coordinate.
       * @param yCoord The homogeneous Y coordinate.
       * @param zCoord The homogeneous Z coordinate.
       * @param alpha Scale factor.
       */
      Vector3D(double xCoord, double yCoord, double zCoord, double alpha)
        : m_x(xCoord), m_y(yCoord), m_z(zCoord) {this->normalize(alpha);}
    
      /** 
       * Copy constructor.
       * 
       * @param source The Vector3D to be copied.
       */
      Vector3D(const Vector3D& source)
        : m_x(source.m_x), m_y(source.m_y), m_z(source.m_z) {}


      /** 
       * Destructor.
       */
      ~Vector3D() {}

      
      /** 
       * Resets the vector to (0.0, 0.0, 0.0).
       *
       * @return A reference to *this.
       */
      inline Vector3D& clear() {
        m_x = 0.0; m_y = 0.0; m_z = 0.0;
        return *this;
      }

      
      /** 
       * Explicitly sets 3D coordinates.
       * 
       * @param xCoord The desired X coordinate.
       * @param yCoord The desired Y coordinate.
       * @param zCoord The desired Z coordinate.
       */
      inline void setValue(double xCoord, double yCoord, double zCoord) {
        m_x = xCoord; m_y = yCoord; m_z = zCoord;
      }

      /** 
       * Explicitly sets 3D homogeneous coordinates.
       * 
       * @param xCoord The homogeneous X coordinate.
       * @param yCoord The homogeneous Y coordinate.
       * @param zCoord The homogeneous Z coordinate.
       * @param alpha Scale factor.
       */
      inline void setValue(double xCoord, double yCoord, double zCoord,
                           double alpha) {
        m_x = xCoord; m_y = yCoord; m_z = zCoord; normalize(alpha);
      }

      /** 
       * Returns the x component of the vector by reference.
       * 
       * @return A reference to the x component of the vector.
       */
      inline double& x() {return m_x;}

      /** 
       * Returns the x component of the vector by value.
       * 
       * @return The value of the x component of the vector.
       */
      inline double x() const {return m_x;}

      /** 
       * Returns the y component of the vector by reference.
       * 
       * @return A reference to the y component of the vector.
       */
      inline double& y() {return m_y;}

      /** 
       * Returns the y component of the vector by value.
       * 
       * @return The value of the y component of the vector.
       */
      inline double y() const {return m_y;}
    
      /** 
       * Returns the z component of the vector by reference.
       * 
       * @return A reference to the z component of the vector.
       */
      inline double& z() {return m_z;}

      /** 
       * Returns the z component of the vector by value.
       * 
       * @return The value of the z component of the vector.
       */
      inline double z() const {return m_z;}

      /**
       * Assignment operator.
       * 
       * @param source The vector to be copied.
       * @return Reference to *this.
       */
      Vector3D& operator=(const Vector3D& source) {
        setValue(source.m_x, source.m_y, source.m_z); return *this;
      }

      /** 
       * The indexing operator returns a reference to the x, y, or z
       * component of *this as if *this were a three element array.
       * Out of bounds indices will return this->z().
       * 
       * @param index This argument is the index into *this.
       * 
       * @return The return value is the selected component of *this.
       */
      inline double& operator[](size_t index) {
        switch(index) {
        case 0: return this->x(); break;
        case 1: return this->y(); break;
        }
        return this->z();
      }
          
      /** 
       * The indexing operator returns the value of the x, y, or z
       * component of *this as if *this were a three element array.
       * Out of bounds indices will return this-z().
       * 
       * @param index This argument is the index into *this.
       * 
       * @return The return value is the selected component of *this.
       */
      inline double operator[](size_t index) const {
        switch(index) {
        case 0: return this->x(); break;
        case 1: return this->y(); break;
        }
        return this->z();
      }
          
      /** 
       * Multiplies each element by a scalar.
       * 
       * @param scalar X, Y, and Z values will be multiplied by this value.
       * @return Reference to *this.
       */
      Vector3D& operator*=(double scalar) {
        m_x *= scalar; m_y *= scalar; m_z *= scalar; return *this;
      }

      /** 
       * Divides each element by a scalar.
       * 
       * @param scalar X, Y, and Z values will be divided by this value.
       * @return Reference to *this.
       */
      Vector3D& operator/=(double scalar) {
        if (scalar == 0) {
          DLR_THROW(ValueException, "Vector3D::operator/=(double)",
                    "Can't divide by zero.");
        }
        m_x /= scalar; m_y /= scalar; m_z /= scalar; return *this;
      }

      /** 
       * Adds the elements of another Vector3D.
       * 
       * @param vec The elements of vec will be added to *this.
       * @return Reference to *this.
       */
      Vector3D& operator+=(const Vector3D& vec) {
        m_x += vec.m_x; m_y += vec.m_y; m_z += vec.m_z; return *this;
      }

      /** 
       * Subtracts the elements of another Vector3D.
       * 
       * @param vec The elements of vec will be subtracted from *this.
       * @return Reference to *this.
       */
      Vector3D& operator-=(const Vector3D& vec) {
        m_x -= vec.m_x; m_y -= vec.m_y; m_z -= vec.m_z; return *this;
      }

      /** 
       * Returns a Vector3D equal to *this, but with each element negated.
       * 
       * @return The result of the negation.
       */
      Vector3D operator-() {
        return Vector3D(-m_x, -m_y, -m_z);
      }
    
    private:
      // Private member functions.
      inline void normalize(double alpha) {
        if(alpha == 1.0) {return;}
        if(alpha == 0.0) {
          DLR_THROW(ValueException, "Vector3D::normalize()",
                    "Bad alpha (0.0).");
        }
        m_x /= alpha; m_y /= alpha; m_z /= alpha;
        return;
      }

      // Private data members.
      double m_x;
      double m_y;
      double m_z;
    }; // class Vector3D


    /* ============== Non-member function declarations ============== */
  
    /** 
     * This operator returns the elementwise sum of two Vector3D instances.
     * 
     * @param vector0 This is the first of the two Vector3D instances to
     * be added.
     * @param vector1 This is the second of the two Vector3D instances to
     * be added.
     * @return A Vector3D instance in which the value of each element is
     * equal to the sum of the corresponding elements of the two arguments.
     */
    Vector3D
    operator+(const Vector3D& vector0, const Vector3D& vector1);

    /** 
     * This operator returns the elementwise difference of two Vector3D
     * instances.
     * 
     * @param vector0 This is the first of the two Vector3D instances to
     * be subtracted.
     * @param vector1 This is the second of the two Vector3D instances to
     * be subtracted.
     * @return A Vector3D instance in which the value of each element is
     * equal to the difference of the corresponding elements of the two
     * arguments.
     */
    Vector3D
    operator-(const Vector3D& vector0, const Vector3D& vector1);
  
    /** 
     * This operator returns the elementwise product of two Vector3D instances.
     * 
     * @param vector0 This is the first of the two Vector3D instances to
     * be multiplied.
     * @param vector1 This is the second of the two Vector3D instances to
     * be multiplied.
     * @return A Vector3D instance in which the value of each element is
     * equal to the product of the corresponding elements of the two arguments.
     */
    Vector3D
    operator*(const Vector3D& vector0, const Vector3D& vector1);

    /** 
     * This operator returns the elementwise dividend of two Vector3D instances.
     * 
     * @param vector0 This is the Vector3D instance whose element values
     * are to be divided.
     * @param vector1 This is the Vector3D instance by whose elements
     * the first argument is to be divided.
     * @return A Vector3D instance in which the value of each element is
     * equal to the corresponding value of the first argument divided by
     * the corresponding value of the second argument.
     */
    Vector3D
    operator/(const Vector3D& vector0, const Vector3D& vector1);

    /** 
     * This operator adds a scalar and a Vector3D.
     * 
     * @param vector0 This is the Vector3D instance to which the scalar
     * should be added.
     * @param scalar0 This is amount which should be added to each
     * element of argument vector0.
     * @return A Vector3D instance in which the value of each element is
     * equal to the corresponding value of the first argument plus the
     * value of the second argument.
     */
    Vector3D operator+(const Vector3D& vector0, double scalar0);

    /** 
     * This operator subtracts a scalar from a Vector3D.
     * 
     * @param vector0 This is the Vector3D instance from which the scalar
     * should be subtracted.
     * @param scalar0 This is amount which should be subtracted from each
     * element of argument vector0.
     * @return A Vector3D instance in which the value of each element is
     * equal to the corresponding value of the first argument minus the
     * value of the second argument.
     */
    Vector3D operator-(const Vector3D& vector0, double scalar0);

    /** 
     * This operator multiplies a Vector3D by scalar.
     * 
     * @param vector0 This is the Vector3D instance which is to be
     * multiplied by the scalar.
     * @param scalar0 This is amount by which should argument vector0 is
     * to be multiplied.
     * @return A Vector3D instance in which the value of each element is
     * equal to the corresponding value of the first argument multiplied by
     * the value of the second argument.
     */
    Vector3D operator*(const Vector3D& vector0, double scalar0);

    /** 
     * This operator divides a Vector3D by scalar.
     * 
     * @param vector0 This is the Vector3D instance which is to be
     * divided by the scalar.
     * @param scalar0 This is amount by which should argument vector0 is
     * to be divided.
     * @return A Vector3D instance in which the value of each element is
     * equal to the corresponding value of the first argument divided by
     * the value of the second argument.
     */
    Vector3D operator/(const Vector3D& vector0, double scalar0);

    /**
     * This operator checks the supplied vectors for equality.
     *
     * @param  vector0  First vector to compare.
     * @param  vector1  Second vector to compare.
     * @return  Result of comparing @p vector0 to @p vector1 for equality.
     */
    bool operator==(const Vector3D& vector0, const Vector3D& vector1);

    /**
     * This operator checks the supplied vectors for inequality.
     *
     * @param  vector0  First vector to compare.
     * @param  vector1  Second vector to compare.
     * @return  Result of comparing @p vector0 to @p vector1 for
     *          inequality.
     */
    bool operator!=(const Vector3D& vector0, const Vector3D& vector1);


    /** 
     * This operator adds a scalar value to each element of a Vector3D
     * instance.
     * 
     * @param scalar0 Scalar argument of the addition.
     *
     * @param vector0 Vector argument of the addition.
     *
     * @return Vector3D instance in which the value of each element is
     * the sum of the scalar argument and the corresponding element of
     * the Vector3D argument.
     */
    Vector3D operator+(double scalar0, const Vector3D& vector0);


    /** 
     * This operator multiplies a scalar value with each element of a
     * Vector3D instance.
     * 
     * @param scalar0 Scalar argument of the multiplication.
     *
     * @param vector0 Vector argument of the multiplication.
     *
     * @return Vector3D instance in which the value of each element is
     * the product of the scalar argument and the corresponding element
     * of the Vector3D argument.
     */
    Vector3D operator*(double scalar0, const Vector3D& vector0);


    /** 
     * This function outputs a text representation of a Vector3D
     * instance to a std::ostream.  The output format looks like this:
     *
     * Vector3D(1.0, 2.0, 3.0)
     *
     * @param stream This argument is a reference to the the output
     * stream.
     *
     * @param vector0 This argument is a const reference to the
     * Vector3D instance to be output.
     *
     * @return The return value is a reference to the input stream after
     * the write has taken place.
     */
    std::ostream& operator<<(std::ostream& stream, const Vector3D& vector0);


    /** 
     * This function sets the value of a Vector3D instance from a
     * std::istream.  The input format is as described for
     * operator<<(std::ostream&, const Vector3D&) above.
     * 
     * @param stream This argument is a reference to the the input
     * stream from which to read.
     *
     * @param vector0 This argument is a reference to the Vector3D
     * which will take the input.
     *
     * @return The return value is a reference to the input stream after
     * the read has taken place.
     */
    std::istream& operator>>(std::istream& stream, Vector3D& vector0);

  } // namespace numeric

} // namespace dlr


/* ======= Declarations to maintain compatibility with legacy code. ======= */

namespace dlr {

  using numeric::Vector3D;

} // namespace dlr

/* ============ Definitions of inline & template functions ============ */

namespace dlr {

  namespace numeric {
    
    inline Vector3D operator+(double scalar0, const Vector3D& vector0)
    {
      return vector0 + scalar0;
    }
  
    inline Vector3D operator*(double scalar0, const Vector3D& vector0)
    {
      return vector0 * scalar0;
    }

  } // namespace numeric

} // namespace dlr

#endif // #ifndef _DLR_VECTOR3D_H_
