/**
***************************************************************************
* @file dlrNumeric/utilities.h
*
* Header file declaring useful functions for dlrNumeric.
*
* Copyright (C) 2001-2008 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 1211 $
* $Date: 2009-10-08 19:04:39 -0400 (Thu, 08 Oct 2009) $
***************************************************************************
**/

#ifndef DLR_NUMERIC_UTILITIES_H
#define DLR_NUMERIC_UTILITIES_H

#include <cmath>
// #include <iostream>
#include <cstdlib>

#include <dlrCommon/physicalConstants.h>
#include <dlrCommon/tags.h>
#include <dlrNumeric/numericTraits.h>
#include <dlrNumeric/array1D.h>
#include <dlrNumeric/array2D.h>
#include <dlrNumeric/array3D.h>
#include <dlrNumeric/index2D.h>
#include <dlrNumeric/solveQuadratic.h>
#include <dlrNumeric/vector2D.h>
#include <dlrNumeric/vector3D.h>


namespace dlr {

  namespace numeric {
    
    namespace constants {

      using dlr::common::constants::pi;

    } // nameapace contants
    

    /** 
     * This function returns an array of the same size and element type
     * as its input argument, in which each element is set to the
     * absolute value of the corresponding element of the input array.
     *
     * @param array0 The elements of the returned array will take on the
     * absolute value of the elements of this array.
     *
     * @return The return value is an array in which each element
     * contains the absolute value of the corresponding element in the
     * input array.
     */
    template <class Type>
    Array1D<Type>
    abs(const Array1D<Type>& array0);

  
    /** 
     * This function returns an array of the same size and element type
     * as its input argument, in which each element is set to the
     * absolute value of the corresponding element of the input array.
     *
     * @param array0 The elements of the returned array will take on the
     * absolute value of the elements of this array.
     *
     * @return The return value is an array in which each element
     * contains the absolute value of the corresponding element in the
     * input array.
     */
    template <class Type>
    Array2D<Type>
    abs(const Array2D<Type>& array0);


    /**
     * This function returns true if each element of its argument is
     * false, and returns false otherwise.
     *
     * @param array0 The elements of this array will be tested.
     *
     * @return The return value is true if all of the elements of array0
     * are false.
     */
    template <class Type>
    bool
    allFalse(const Array1D<Type>& array0);

  
    /**
     * This function returns true if each element of its argument is
     * true, and returns false otherwise.
     *
     * @param array0 The elements of this array will be tested.
     *
     * @return The return value is true if all of the elements of array0
     * are true.
     */
    template <class Type>
    bool
    allTrue(const Array1D<Type>& array0);

  
    /**
     * This function returns true if any element of its argument is
     * true, and returns false otherwise.
     *
     * @param array0 The elements of this array will be tested.
     *
     * @return The return value is true if any of the elements of array0
     * are false.
     */
    template <class Type>
    inline bool
    anyFalse(const Array1D<Type>& array0);

  
    /**
     * This function returns true if any element of its argument is
     * true, and returns false otherwise.
     *
     * @param array0 The elements of this array will be tested.
     *
     * @return The return value is true if any of the elements of array0
     * are true.
     */
    template <class Type>
    inline bool
    anyTrue(const Array1D<Type>& array0);

  
    /** 
     * This function returns the index of the largest element of its input
     * array.  This function is equivalent to the quantity
     *   (std::max_element(array0.begin(), array0.end()) - array0.begin())
     *
     * @param array0 The elements of this array will be evaluated, and
     * the index of the largest will be returned.
     *
     * @return The index of the largest element of the array.
     */
    template <class Type>
    inline size_t
    argmax(const Array1D<Type>& array0);


    /** 
     * This function returns the index of the largest element of its input
     * sequence.  This function is equivalent to the quantity
     *   (std::max_element(beginIter, endIter) - beginIter)
     *
     * @param beginIter This argument marks the start of the input
     * sequence.
     *
     * @param endIter This argument marks (one element past) the end
     * of the input sequence.
     *
     * @return The index of the largest element of the sequence.
     */
    template <class IterType>
    inline size_t
    argmax(IterType beginIter, IterType endIter);


    /** 
     * This function returns the index of the largest element of its
     * input array, where largeness is defined by the second argument.
     * This function is equivalent to the quantity
     *
     *   (std::max_element(array0.begin(), array0.end(), comparator)
     *    - array0.begin())
     *
     * NOTE: Read the argument description for comparator carefully.  It
     * is consistent with the standard library convention, but many
     * people find it to be counterintuitive.
     *
     * @param array0 The elements of this array will be evaluated, and
     * the index of the largest will be returned.
     *
     * @param comparator This argument is a functor which supports the
     * std::binary_function<Type, Type, bool> interface, and returns
     * true if its first argument is smaller than its second argument.
     *
     * @return The index of the largest element of the array.
     */
    template <class Type, class Functor>
    inline size_t
    argmax(const Array1D<Type>& array0, Functor comparator);
  

    /** 
     * This function returns the index of the largest element of its
     * input sequence, where largeness is defined by the second argument.
     * This function is equivalent to the quantity
     *
     *   (std::max_element(beginIter, endIter, comparator)
     *    - array0.begin())
     *
     * NOTE: Read the argument description for comparator carefully.  It
     * is consistent with the standard library convention, but many
     * people find it to be counterintuitive.
     *
     * @param beginIter This argument marks the start of the input
     * sequence.
     *
     * @param endIter This argument marks (one element past) the end
     * of the input sequence.
     *
     * @param comparator This argument is a functor which supports the
     * std::binary_function<Type, Type, bool> interface, and returns
     * true if its first argument is smaller than its second argument.
     *
     * @return The index of the largest element of the array.
     */
    template <class IterType, class Functor>
    inline size_t
    argmax(IterType beginIter, IterType endIter, Functor comparator);
  

    /** 
     * This function returns an Index2D instance indicating which is
     * the largest element of its input array.
     *
     * @param array0 The elements of this array will be evaluated, and
     * the index of the largest will be returned.
     *
     * @return The index of the largest element of the array.
     */
    template <class Type>
    inline Index2D
    argmax2D(const Array2D<Type>& array0);


    /** 
     * This function returns an Index2D instance indicating which is
     * the largest element of its input array, where largeness is
     * defined by the second argument.
     *
     * NOTE: Read the argument description for comparator carefully.
     * It is consistent with the standard library convention, but many
     * people find it to be counterintuitive.
     *
     * @param array0 The elements of this array will be evaluated, and
     * the index of the largest will be returned.
     *
     * @param comparator This argument is a functor which supports the
     * std::binary_function<Type, Type, bool> interface, and returns
     * true if its first argument is smaller than its second argument.
     *
     * @return The index of the largest element of the array.
     */
    template <class Type, class Functor>
    inline Index2D
    argmax2D(const Array2D<Type>& array0, Functor comparator);
  

    /** 
     * This function returns the index of the smallest element of its input
     * array.  This function is equivalent to
     *   (std::min_element(array0.begin(), array0.end()) - array0.begin());
     *
     * @param array0 The elements of this array will be evaluated, and
     * the index of the smallest will be returned.
     *
     * @return The index of the smallest element of the array.
     */
    template <class Type>
    inline size_t
    argmin(const Array1D<Type>& array0);


    /** 
     * This function returns the index of the smallest element of its
     * input array, where smallness is defined by the second argument.
     * This function is equivalent to the quantity
     *
     *   (std::min_element(array0.begin(), array0.end(), comparator)
     *    - array0.begin())
     *
     * @param array0 The elements of this array will be evaluated, and
     * the index of the smallest will be returned.
     *
     * @param comparator This argument is a functor which supports the
     * std::binary_function<Type, Type, bool> interface, and returns
     * true if its first argument is smaller than its second argument.
     *
     * @return The index of the smallest element of the array.
     */
    template <class Type, class Functor>
    inline size_t
    argmin(const Array1D<Type>& array0, Functor comparator);


    /** 
     * This function returns an Index2D instance indicating which is
     * the smallest element of its input array.
     *
     * @param array0 The elements of this array will be evaluated, and
     * the index of the smallest will be returned.
     *
     * @return The index of the smallest element of the array.
     */
    template <class Type>
    inline Index2D
    argmin2D(const Array2D<Type>& array0);


    /** 
     * This function returns an Index2D instance indicating which is
     * the smallest element of its input array, where smallness is
     * defined by the second argument.
     *
     * @param array0 The elements of this array will be evaluated, and
     * the index of the smallest will be returned.
     *
     * @param comparator This argument is a functor which supports the
     * std::binary_function<Type, Type, bool> interface, and returns
     * true if its first argument is smaller than its second argument.
     *
     * @return The index of the smallest element of the array.
     */
    template <class Type, class Functor>
    inline Index2D
    argmin2D(const Array2D<Type>& array0, Functor comparator);


    /** 
     * This function returns an array of indices, result, so that the
     * sequence (array0[result[0]], array0[result[1]],
     * array0[result[2]], ...) is sorted from smallest to largest using
     * operator<().
     *
     * @param array0 The of this array will be compared to each other in
     * order to establish the correct sequence of indices.
     *
     * @return An array of indices as described above.
     *
     * Note(xxx): provide a version that uses iterators.
     */
    template <class Type>
    Array1D<size_t>
    argsort(const Array1D<Type>& array0);

  
//   /** 
//    * This function returns an array of indices, result, so that the
//    * sequence (array0[result[0]], array0[result[1]],
//    * array0[result[2]], ...) is sorted from smallest to largest using
//    * the supplied comparison operator.
//    *
//    * @param array0 The of this array will be compared to each other in
//    * order to establish the correct sequence of indices.
//    *
//    * @param comparator This argument is a functor which supports the
//    * std::binary_function<Type, Type, bool> interface, and returns
//    * true if its first argument is smaller than its second argument.
//    *
//    * @return An array of indices as described above.
//    */
//   template <class Type, class Functor>
//   Array1D<size_t>
//   argsort(const Array1D<Type>& array0, Functor comparator);

  
    /** 
     * This function is just an alias for the function axisMaximum().
     * 
     * @param array0 See documentation for axisMaximum().
     *
     * @param axis See documentation for axisMaximum().
     *
     * @return See documentation for axisMaximum().
     */
    template <class Type>
    Array1D<Type>
    axisMax(const Array2D<Type>& array0,
            size_t axis) {return axisMaximum(array0, axis);}

    /** 
     * This function returns an Array1D in which each element has the
     * value of the largest element in one row or column of the input
     * Array2D.  The maximum is taken along the axis specified by the
     * second argument.  For more information see the documentation of
     * axisMaximum(const Array2D&, size_t, Functor).
     * 
     * @param array0 This is the input Array2D instance.
     *
     * @param axis This argument indicates along which axis the maximum
     * should be computed. Axis 0 is aligned with the columns of the
     * array, while axis 1 is aligned with the rows of the array.
     *
     * @return The return value is an Array1D instance containing the
     * maximum element from each row or column of the input array.
     */
    template <class Type>
    inline Array1D<Type>
    axisMaximum(const Array2D<Type>& array0, size_t axis);

    /** 
     * This function returns an Array1D in which each element has the
     * value of the largest element in one row or column of the input
     * Array2D, where largeness is defined by the third argument.  The
     * maximum is taken along the axis specified by the second argument.
     * For example:
     *
     *   Array2D<int> testArray("[[1, 3, 0], [5, 0, 2]]");
     *   std::cout << axisMaximum(testArray, 0, std::less<int>()) << ", "
     *             << axisMaximum(testArray, 1, std::less<int>()) << ".";
     *
     * will print "Array1D([5, 3, 2]), Array1D([3, 5])."
     * 
     * NOTE: Read the argument description for comparator carefully.  It
     * is consistent with the standard library convention, but many
     * people find it to be counterintuitive.
     *
     * @param array0 This is the input Array2D instance.
     *
     * @param axis This argument indicates along which axis the maximum
     * should be computed. Axis 0 is aligned with the columns of the
     * array, while axis 1 is aligned with the rows of the array.
     *
     * @param comparator This argument is a functor which supports the
     * std::binary_function<Type, Type, bool> interface, and returns
     * true if its first argument is smaller than its second argument.
     *
     * @return The return value is an Array1D instance containing the
     * maximum element from each row or column of the input array.
     */
    template <class Type, class Functor>
    Array1D<Type>
    axisMaximum(const Array2D<Type>& array0, size_t axis, Functor comparator);

    /** 
     * This function is just an alias for the function axisMinimum().
     * 
     * @param array0 See documentation for axisMinimum().
     *
     * @param axis See documentation for axisMinimum().
     *
     * @return See documentation for axisMinimum().
     */
    template <class Type>
    inline Array1D<Type>
    axisMin(const Array2D<Type>& array0,
            size_t axis) {return axisMinimum(array0, axis);}
  
    /** 
     * This function returns an Array1D in which each element has the
     * value of the smallest element in one row or column of the input
     * Array2D.  The minimum is taken along the axis specified by the
     * second argument.  For more information see the documentation of
     * axisMinimum(const Array2D&, size_t, Functor).
     * 
     * @param array0 This is the input Array2D instance.
     *
     * @param axis This argument indicates along which axis the minimum
     * should be computed. Axis 0 is aligned with the columns of the
     * array, while axis 1 is aligned with the rows of the array.
     *
     * @return The return value is an Array1D instance containing the
     * minimum element from each row or column of the input array.
     */
    template <class Type>
    inline Array1D<Type>
    axisMinimum(const Array2D<Type>& array0, size_t axis);

    /** 
     * This function returns an Array1D in which each element has the
     * value of the smallest element in one row or column of the input
     * Array2D, where smallness is defined by the return value of the
     * third argument.  The minimum is taken along the axis specified
     * by the second argument.  For example:
     *
     *   Array2D<int> testArray("[[1, 3, 0], [5, 0, 2]]");
     *   std::cout << axisMinimum(testArray, 0, std::less<int>()) << ", "
     *             << axisMinimum(testArray, 1, std::less<int>()) << ".";
     *
     * will print "Array1D([1, 0, 0]), Array1D([0, 0])."
     * 
     * NOTE: Read the argument description for comparator carefully.  It
     * is consistent with the standard library convention, but many
     * people find it to be counterintuitive.
     *
     * @param array0 This is the input Array2D instance.
     *
     * @param axis This argument indicates along which axis the minimum
     * should be computed. Axis 0 is aligned with the columns of the
     * array, while axis 1 is aligned with the rows of the array.
     *
     * @param comparator This argument is a functor which supports the
     * std::binary_function<Type, Type, bool> interface, and returns
     * true if its first argument is smaller than its second argument.
     *
     * @return The return value is an Array1D instance containing the
     * minimum element from each row or column of the input array.
     */
    template <class Type, class Functor>
    Array1D<Type>
    axisMinimum(const Array2D<Type>& array0, size_t axis, Functor comparator);

    /** 
     * This function returns an Array1D in which each element has the
     * sum of one row or column of the input Array2D.  The sum is taken
     * along the axis specified by the second argument.  For Example:
     *
     *   Array2D<int> testArray("[[1, 3, 0], [5, 0, 2]]");
     *   std::cout << axisSum(testArray, 0) << ", "
     *             << axisSum(testArray, 1) << ".";
     *
     * will print "Array1D([6, 3, 2]), Array1D([4, 7])."
     *
     * The summation is done using values of type
     * NumericTraits<Type>::SumType, where Type is the type of the
     * elements in the input array.
     * 
     * @param array0 This is the input Array2D instance.
     *
     * @param axis This argument indicates along which axis the sum
     * should be computed. Axis 0 is aligned with the columns of the
     * array, while axis 1 is aligned with the rows of the array.
     *
     * @return The return value is an Array1D instance containing the
     * sums of the rows or columns of the input array.
     */
    template <class Type>
    inline Array1D<typename NumericTraits<Type>::SumType>
    axisSum(const Array2D<Type>& array0, size_t axis);

    /** 
     * This function returns an Array1D in which each element has the
     * sum of one row or column of the input Array2D.  The sum is taken
     * along the axis specified by the second argument, as described for
     * axisSum(const Array2D&, size_t), above.
     *
     * This version of axisSum permits the user to control the precision
     * of the summation by specifying the type which will be used to do
     * the computation.
     * 
     * @param array0 This is the input Array2D instance.
     *
     * @param axis This argument indicates along which axis the sum
     * should be computed. Axis 0 is aligned with the columns of the
     * array, while axis 1 is aligned with the rows of the array.
     *
     * @param resultTag This argument is a type_tag
     * indicating which type is to be used to perform the computation.
     *
     * @return The return value is an Array1D instance containing the
     * sums of the rows or columns of the input array.
     */
    template <class Type, class ResultType>
    inline Array1D<ResultType>
    axisSum(const Array2D<Type>& array0, size_t axis,
            type_tag<ResultType> resultTag);

    /** 
     * This function returns an Array1D in which each element has the
     * sum of one row or column of the input Array2D.  The sum is taken
     * along the axis specified by the second argument, as described for
     * axisSum(const Array2D&, size_t), above, and addition is performed
     * using a user supplied binary operator.  For each row or column,
     * the sum is initialized to the value specified by initialValue
     * before the summation is performed.  After this initialization,
     * each row- or column- sum is updated according to the equation:
     *
     *   sum = adder(sum, nextElement)
     *
     * for each element of the row or column.  For more information, see
     * the documentation of other axisSum() versions.
     *
     * This version of axisSum also permits the user to control the
     * precision of the summation by specifying the type which will be
     * used to do the computation.
     * 
     * @param array0 This is the input Array2D instance.
     *
     * @param axis This argument indicates along which axis the sum
     * should be computed. Axis 0 is aligned with the columns of the
     * array, while axis 1 is aligned with the rows of the array.
     *
     * @param resultTag This argument is a type_tag
     * indicating which type is to be used to perform the computation.
     *
     * @param initialValue This argument specifies the value to which
     * each row/column sum will be initialized prior to accumulating the
     * elements of the array.
     *
     * @param adder This argument is a functor conforming to the
     * std::binary_function interface which will be used to add element
     * values.
     *
     * @return The return value is an Array1D instance containing the
     * sums of the rows or columns of the input array.
     */
    template <class Type, class ResultType, class Functor>
    Array1D<ResultType>
    axisSum(const Array2D<Type>& array0, size_t axis,
            type_tag<ResultType> resultTag,
            const ResultType& initialValue, Functor adder);

    /**
     * columnIndices(rows, columns): Returns an Array2D in which each
     * element contains the index of its column.  The element type of the
     * returned array depends on the type_tag argument.  For example:
     *
     *   columnIndices(2, 3, type_tag<int>());
     *
     * will return:
     *
     *   Array2D<int>([[0, 1, 2],
     *                 [0, 1, 2]]);
     *
     * 
     * @param rows This argument specifies the number of rows in the
     * returned array.
     *
     * @param columns This argument specifies the number of columns in
     * the returned array.
     *
     * @param typeTag This argument specifies the element type of
     * the returned array.
     *
     * @return The return value is an Array2D of the specified size and
     * type in which all elements in the first column have value
     * initialized to 0, all elements in the second column have value
     * initialized to 1, and so on.
     */
    template <class Type>
    Array2D<Type>
    columnIndices(size_t rows, size_t columns, type_tag<Type> typeTag);

    /** 
     * This function selects those elements of an input Array1D which
     * correspond to "true" values of a mask Array1D, and returns an
     * Array1D containing only those elements.  For example:
     *
     *   Array1D<int> a = compress(
     *     Array1D<bool>("1, 1, 0, 1, 0]"),
     *     Array1D<int>("1, 2, 3, 4, 5"));
     *   std::cout << a;
     *
     * will print:
     *
     *   "Array1D([1, 2, 4])"
     *
     * It is an error for the two array arguments to have different sizes.
     *
     * @param condition This argument is the mask which selects which
     * elements will be included in the output array.  Elements of
     * argument this array which evaluate to true will cause the
     * corresponding elements of the input array to be included in the
     * output.
     *
     * @param input Elements of this argument will be included in the
     * output array if the corresponding elements of the condition array
     * evaluate to true.  This array must have the same number of
     * elements as the condition array.
     *
     * @return The return value is an Array1D instance containing only
     * those elements of the input array which correspond to true
     * elements of the condition array.
     */
    template <class Type0, class Type1>
    inline Array1D<Type1>
    compress(const Array1D<Type0>& condition, const Array1D<Type1>& input);

    /** 
     * This function behaves in exactly the same way as compress(const
     * Array1D&, const Array1D&), above, but it permits the user to
     * specify the number of true elements in the condition array.  This
     * eliminates one pass through the condition array, slightly
     * speeding up the computation.  Use this version with caution,
     * since specifying an incorrect count can crash your program.
     *
     * @param condition This argument is the mask which selects which
     * elements will be included in the output array.  Elements of
     * argument this array which evaluate to true will cause the
     * corresponding elements of the input array to be included in the
     * output.
     *
     * @param input Elements of this argument will be included in the
     * output array if the corresponding elements of the condition array
     * evaluate to true.  This array must have the same number of
     * elements as the condition array.
     *
     * @param numTrue This argument specifies the number of "true"
     * elements in the condition array.
     *
     * @return The return value is an Array1D instance containing only
     * those elements of the input array which correspond to true
     * elements of the condition array.
     */
    template <class Type0, class Type1>
    Array1D<Type1>
    compress(const Array1D<Type0>& condition, const Array1D<Type1>& input,
             size_t numTrue);


    /** 
     * This element counts the number of elements of the input array
     * which evaluate to true, and returns that number.
     * 
     * @param array0 This argument is the array of elements to be
     * counted.
     *
     * @return The return value is the total number of elements in x
     * which, when cast to bool, evaluate to true.
     */
    template <class Type>
    inline size_t
    count(const Array1D<Type>& array0);
  
    /** 
     * This function computes the cross product of two Vector3D
     * instances.  That is, it computes a Vector3D which lies
     * perpendicular to each of the arguments and has magnitude equal
     * to the product of the magnitudes of the arguments times the sine
     * of the angle between them.
     * 
     * @param vector0 This is the first argument of the cross product.
     *
     * @param vector1 This is the first argument of the cross product.
     *
     * @return The cross product of the two arguments.
     */
    inline Vector3D
    cross(const Vector3D& vector0, const Vector3D& vector1);
  
    /** 
     * This function computes the inner product of two input arrays.
     * The computation is done using the ProductType specified by the
     * appropriate NumericTraits class.
     *
     * @param array0 A 1D array, the first argument of the dot product.
     *
     * @param array1 A 1D array, the second argument of the dot product.
     *
     * @return The inner product of the two arguments.
     */
    template <class Type>
    inline typename NumericTraits<Type>::ProductType
    dot(const Array1D<Type>& array0, const Array1D<Type>& array1);

    /** 
     * This function computes the inner product of two input arrays, and
     * allows the user to control which type is used to do the
     * calculation.
     *
     * @param array0 A 1D array, the first argument of the dot product.
     *
     * @param array1 A 1D array, the second argument of the dot product.
     *
     * @param typeTag The third argument is a type tag
     * indicating which type is to be used to perform the computation.
     *
     * @return The inner product of the two array arguments.
     */
    template <class Type0, class Type1, class Type2>
    inline Type2
    dot(const Array1D<Type0>& array0, const Array1D<Type1>& array1,
        type_tag<Type2> typeTag);

    /** 
     * This function computes the inner product of two Vector2D instances.
     * 
     * @param vector0 A Vector2D, the first argument of the dot product.
     *
     * @param vector1 A Vector2D, the second argument of the dot product.
     *
     * @return The inner product of the two Vector2D arguments.
     */
    inline double
    dot(const Vector2D& vector0, const Vector2D& vector1);

    /** 
     * This function computes the inner product of two Vector3D instances.
     * 
     * @param vector0 A Vector3D, the first argument of the dot product.
     *
     * @param vector1 A Vector3D, the second argument of the dot product.
     *
     * @return The inner product of the two Vector3D arguments.
     */
    inline double
    dot(const Vector3D& vector0, const Vector3D& vector1);

    /** 
     * This function computes from a vector, x, the matrix, X, such that
     * the matrix product A * x is equal to the matrix product X *
     * vec(A).  That is, if
     *
     *   X = equivalentMatrix(x, n);
     *
     * then
     *
     *   A * x == X * vec(A)
     *
     * where x is a vector of m elements, and A is a matrix with n rows
     * and m columns.
     * 
     * @param vector0 This argument represents the vector x.
     * 
     * @param rowsInMatrix This argument describes the matrix A.
     * 
     * @return The return value is an Array2D instance describing the
     * matrix X.
     */
    template <class Type>
    Array2D<Type>
    equivalentMatrix(const Array1D<Type>& vector0, size_t rowsInMatrix);
  

    /** 
     * This function estimates the mean and covariance of a set of
     * vectors, which are represented by the rows (or columns) of the
     * input 2D array.  Estimated mean and covariance are returned via
     * reference arguments.
     * 
     * @param sampleArray This argument is the array of sample vectors.
     * If argument majorAxis is set to 0, then each row of sampleArray
     * represents a sample.  If argument majorAxis is nonzero, then each
     * column of sampleArray represents a sample.
     * 
     * @param mean The estimated mean vector is returned by reference
     * using this argument.
     * 
     * @param covariance The estimated covariance matrix is returned by
     * reference using this argument.
     * 
     * @param majorAxis This argument is normally set to 0 or 1, and
     * specifies the arrangement of samples in sampleArray as described
     * above.  If majorAxis is set to a value which is not equal to 0 or
     * 1, computation will proceed as if it had been set to 1.
     */
    template <class Type>
    void
    getMeanAndCovariance(const Array2D<Type>& sampleArray,
                         Array1D<double>& mean,
                         Array2D<double>& covariance,
                         size_t majorAxis=0);
  
                     
    /** 
     * This function returns an Array2D instance with the specified
     * shape in which the elements on the diagonal are set to 1, and all
     * other elements are set to 0.
     * 
     * @param rows This argument specifies the number of rows in the
     * returned Array2D instance.
     *
     * @param columns This argument specifies the number of columns in
     * the returned Array2D instance.
     *
     * @param typeTag This argument controls the element type of
     * the returned Array2D instance.
     *
     * @return An Array2D in each element on the diagonal is set to 1,
     * and all other elements are set to 0.
     */
    template <class Type>
    inline Array2D<Type>
    identity(size_t rows, size_t columns, type_tag<Type> typeTag);


    /** 
     * This function returns an Array2D instance with the specified
     * shape in which the elements on the diagonal are set to 1, and all
     * other elements are set to 0.
     * 
     * @param rows This argument specifies the number of rows in the
     * returned Array2D instance.
     *
     * @param columns This argument specifies the number of columns in
     * the returned Array2D instance.
     *
     * @return An Array2D in each element on the diagonal is set to 1,
     * and all other elements are set to 0.
     */
    template <class Type>
    Array2D<Type>
    identity(size_t rows, size_t columns);


    /** 
     * This function returns an array in which each element is the
     * natural logarithm of the corresponding element of its input.
     * 
     * @param array0 This argument is an Array1D with arguments in a
     * common floating point type such as double or float.
     * 
     * @return The return value is an array of natural log values.
     */
    template <class Type>
    Array1D<Type>
    ln(const Array1D<Type>& array0);
  

    /** 
     * This function returns an array in which each element is the
     * natural logarithm of the corresponding element of its input.
     * 
     * @param array0 This argument is an Array2D with arguments in a
     * common floating point type such as double or float.
     * 
     * @return The return value is an array of natural log values.
     */
    template <class Type>
    Array2D<Type>
    ln(const Array2D<Type>& array0);
  

    /** 
     * This function returns an Array2D instance in which the value of
     * each element is the logical not of the corresponding element of
     * the input array.  For example:
     *
     *   Array2D<int> a = logical_not(Array2D<int>("[[1, 2], [0, -5]]");
     *   std::cout << a;
     *
     * will print:
     *
     *   "Array2D([[0, 0], [1, 0]])"
     * 
     * @param array0 This argument is an Array2D with elements of a type
     * which can be cast to bool.
     *
     * @return The return value is an Array2D instance of the same shape
     * and element type as the input array, in which each element is set
     * to the logical not of the corresponding element of the input
     * array.
     */
    template <class Type>
    Array2D<Type>
    logicalNot(const Array2D<Type>& array0);

    /** 
     * This function computes the magnitude of its input argument.  That
     * is, it computes the square root of the sum of the squares of the
     * elements of its input argument.
     * 
     * @param array0 An Array1D whose magnitude is to be calculated.
     *
     * @return The magnitude of the input argument.
     */  
    template <class Type>
    inline Type
    magnitude(const Array1D<Type>& array0);

    /** 
     * This function computes the magnitude of its input argument.  That
     * is, it computes the square root of the sum of the squares of the
     * elements of its input argument.
     * 
     * @param vector0 A Vector2D whose magnitude is to be calculated.
     *
     * @return The magnitude of the input argument.
     */  
    inline double
    magnitude(const Vector2D& vector0);

    /** 
     * This function computes the magnitude of its input argument.  That
     * is, it computes the square root of the sum of the squares of the
     * elements of its input argument.
     * 
     * @param vector0 A Vector3D whose magnitude is to be calculated.
     *
     * @return The magnitude of the input argument.
     */  
    inline double
    magnitude(const Vector3D& vector0);

  
    /** 
     * This function computes the square of the magnitude of its input
     * argument.  That is, it computes the sum of the squares of the
     * elements of its input argument.
     * 
     * @param array0 An Array1D whose squared magnitude is to be
     * calculated.
     *
     * @return The squared magnitude of the input argument.
     */  
    template <class Type>
    inline typename NumericTraits<Type>::ProductType
    magnitudeSquared(const Array1D<Type>& array0);

  
    /** 
     * This function computes the square of the magnitude of its input
     * argument.  That is, it computes the sum of the squares of the
     * elements of its input argument.
     * 
     * @param vector0 A Vector2D whose squared magnitude is to be
     * calculated.
     *
     * @return The squared magnitude of the input argument.
     */  
    inline double
    magnitudeSquared(const Vector2D& vector0);

  
    /** 
     * This function computes the square of the magnitude of its input
     * argument.  That is, it computes the sum of the squares of the
     * elements of its input argument.
     * 
     * @param vector0 A Vector3D whose squared magnitude is to be
     * calculated.
     *
     * @return The squared magnitude of the input argument.
     */  
    inline double
    magnitudeSquared(const Vector3D& vector0);
  

    /** 
     * This function computes a vector * matrix product.  Assuming the
     * first argument, vector0, represents a column vector and the
     * second argument, matrix0, represents a 2D array, then this function
     * computes the product:
     *  vector0' * matrix0
     * Where the single quote indicates transpose.
     * 
     * @param vector0 The first operand for the multiplication.
     *
     * @param matrix0 The second operand for the multiplication.
     *
     * @return A 1D array which is the matrix product of vector0 and
     * matrix0.  The element type of this array is determined by
     * NumericTraits<Type>::ProductType for the element type of the
     * arguments.
     */
    template <class Type>
    inline Array1D<typename NumericTraits<Type>::ProductType>
    matrixMultiply(const Array1D<Type>& vector0, const Array2D<Type>& matrix0);

    /** 
     * This function computes a vector * matrix product just like
     * matrixMultiply(const Array1D<Type>&, const Array2D<Type>&), above.
     * This function differs in that the element type of the return
     * value is set explicitly using a third argument.
     * 
     * @param vector0 The first operand for the multiplication.
     *
     * @param matrix0 The second operand for the multiplication.
     *
     * @param typeTag The third argument is a type tag
     * indicating which type is to be used to perform the computation.
     *
     * @return A 1D array which is the matrix product of vector0 and
     * matrix0.  The element type of this array is set by the type_tag 
     * argument.
     */
    template <class Type0, class Type1, class Type2>
    Array1D<Type2>
    matrixMultiply(const Array1D<Type0>& vector0, const Array2D<Type0>& matrix0,
                   type_tag<Type2> typeTag);

    /** 
     * This function computes a matrix * vector product.  Assuming the
     * first argument, matrix0, represents a matrix, and the and the
     * second argument, vector0, represents a column vector, then this
     * function computes the product:
     *  matrix0 * vector0
     * 
     * @param matrix0 The first operand for the multiplication.
     *
     * @param vector0 The second operand for the multiplication.
     *
     * @return A 1D array which is the matrix product of matrix0 and
     * vector0.  The element type of this array is determined by
     * NumericTraits<Type>::ProductType for the element type of the
     * arguments.
     */
    template <class Type>
    inline Array1D<typename NumericTraits<Type>::ProductType>
    matrixMultiply(const Array2D<Type>& matrix0, const Array1D<Type>& vector0);

    /** 
     * This function computes a matrix * vector product just like
     * matrixMultiply(const Array2D<Type>&, const Array1D<Type>&), above.
     * This function differs in that the element type of the return
     * value is set explicitly using a third argument.
     * 
     * @param matrix0 The first operand for the multiplication.
     *
     * @param vector0 The second operand for the multiplication.
     *
     * @param typeTag The third argument is a type tag
     * indicating which type is to be used to perform the computation.
     *
     * @return A 1D array which is the matrix product of matrix0 and
     * vector0.  The element type of this array is set by the type_tag
     * argument.
     */
    template <class Type0, class Type1, class Type2>
    Array1D<Type2>
    matrixMultiply(const Array2D<Type0>& matrix0, const Array1D<Type1>& vector0,
                   type_tag<Type2> typeTag);


    /** 
     * This function computes a matrix * matrix product.  That is,
     * the elements of the resulting array are the dot products of the
     * rows of the first argument with the columns of the second argument.
     * 
     * @param matrix0 The first operand for the multiplication.
     *
     * @param matrix1 The second operand for the multiplication.
     *
     * @return A 2D array which is the matrix product of matrix0 and
     * matrix1.  The element type of this array is determined by
     * NumericTraits<Type>::ProductType for the element type of the
     * arguments.
     */
    template <class Type>
    inline Array2D<typename NumericTraits<Type>::ProductType>
    matrixMultiply(const Array2D<Type>& matrix0, const Array2D<Type>& matrix1);
    
    /** 
     * This function computes a matrix * matrix product, just like
     * matrixMultiply(const Array2D<Type>&, const Array2D<Type>&), above.
     * This function differs in that the element type of the return
     * value is set explicitly using a third argument.
     * 
     * @param matrix0 The first operand for the multiplication.
     *
     * @param matrix1 The second operand for the multiplication.
     *
     * @param typeTag The third argument is a type tag
     * indicating which type is to be used to perform the computation.
     *
     * @return A 2D array which is the matrix product of matrix0 and
     * matrix1.  The element type of this array is set by the type_tag
     * argument.
     */
    template <class Type0, class Type1, class Type2>
    Array2D<Type2>
    matrixMultiply(const Array2D<Type0>& matrix0, const Array2D<Type1>& matrix1,
                   type_tag<Type2> typeTag);


    // The next function is commented out because some environments use
    // #define directives for min() and max(), which don't obey
    // namespaces, and makes the code not compile if we define our own
    // min/max.
    // 
    // 
    // /** 
    // * This function is an alias for the function maximum(const
    // * Array1D&) below.
    // * 
    // * @param array0 See documentation for maximum(const Array1D&).
    // *
    // * @return See documentation for maximum(const Array1D&).
    // */
    // template <class Type>
    // inline Type
    // max(const Array1D<Type>& array0) {return maximum(array0);}

    /** 
     * This function returns a copy of the largest element in the input
     * Array1D instance.
     * 
     * @param array0 This argument is the input array.
     *
     * @return A copy of an element of the input array such that no
     * other element of the input array is larger.
     */
    template <class Type>
    inline Type
    maximum(const Array1D<Type>& array0);

    /** 
     * This function returns a copy of the largest element in the input
     * Array1D instance, where largeness is defined by the return value
     * of the second argument.
     * 
     * NOTE: Read the argument description for comparator carefully.  It
     * is consistent with the standard library convention, but many
     * people find it to be counterintuitive.
     *
     * @param array0 This argument is the input array.
     *
     * @param comparator This argument is a functor which supports the
     * std::binary_function<Type, Type, bool> interface, and returns
     * true if its first argument is smaller than its second argument.
     *
     * @return A copy of an element of the input array such that no
     * other element of the input array is larger, where largeness is
     * defined by comparator.
     */
    template <class Type, class Functor>
    Type
    maximum(const Array1D<Type>& array0, Functor comparator);

    
    /** 
     * This function computes the average value, or geometric mean, of
     * the elements of its argument.  The computation is performed using
     * the precision specified by NumericTraits<Type>::SumType, but is
     * returned as an instance of Type.
     * 
     * @param array0 This argument is the input array, from the elements
     * of which the mean will be computed.
     *
     * @return The return value is the mean of the elements of array0.
     */
    template <class Iterator, class Type>
    Type
    mean(const Iterator& beginIter, const Iterator& endIter);

    
    /** 
     * This function computes the average value, or geometric mean, of
     * the elements of its argument.  The computation is performed using
     * the precision specified by NumericTraits<Type>::SumType, but is
     * returned as an instance of Type.
     * 
     * @param array0 This argument is the input array, from the elements
     * of which the mean will be computed.
     *
     * @return The return value is the mean of the elements of array0.
     */
    template <class Type>
    inline double
    mean(const Array1D<Type>& array0);

    
    /** 
     * This function computes the average value, or geometric mean, of
     * the elements of its argument, and allows the user to specify the
     * precision with which the computation is carried out.  The
     * computation is performed using the type specified by the second
     * argument to the mean() function call.
     * 
     * @param array0 This argument is the input array, from the elements
     * of which the mean will be computed.
     *
     * @param typeTag This argument specifies the type which will be
     * used to perform the computation, and also specifies the return
     * type of the function.
     *
     * @return The return value is the mean of the elements of array0.
     */
    template <class Type0, class Type1>
    inline Type1
    mean(const Array1D<Type0>& array0, type_tag<Type1> typeTag);
  

    // The next function is commented out because some environments use
    // #define directives for min() and max(), which don't obey
    // namespaces, and makes the code not compile if we define our own
    // min/max.
    // 
    // 
    // /** 
    // * This function is an alias for the function minimum(const
    // * Array1D&) below.
    // * 
    // * @param array0 See documentation for minimum(const Array1D&).
    // *
    // * @return See documentation for minimum(const Array1D&).
    // */
    // template <class Type>
    // inline Type
    // min(const Array1D<Type>& array0) {return minimum(array0);}

    /** 
     * This function returns a copy of the smallest element in the input
     * Array1D instance.
     * 
     * @param array0 This argument is the input array.
     *
     * @return A copy of an element of the input array such that no
     * other element of the input array is smaller.
     */
    template <class Type>
    inline Type
    minimum(const Array1D<Type>& array0);

    /** 
     * This function returns a copy of the smallest element in the input
     * Array1D instance, where largeness is defined by the return value
     * of the second argument.
     * 
     * NOTE: Read the argument description for comparator carefully.  It
     * is consistent with the standard library convention, but many
     * people find it to be counterintuitive.
     *
     * @param array0 This argument is the input array.
     *
     * @param comparator This argument is a functor which supports the
     * std::binary_function<Type, Type, bool> interface, and returns
     * true if its first argument is smaller than its second argument.
     *
     * @return A copy of an element of the input array such that no
     * other element of the input array is smaller, where largeness is
     * defined by comparator.
     */
    template <class Type, class Functor>
    Type
    minimum(const Array1D<Type>& array0, Functor comparator);


    /** 
     * This function uses the iterative method of Newton and Raphson to
     * search for a zero crossing of the supplied functor.  If convergence
     * fails, it throws a ValueException (blaming the argument for not
     * having a sufficiently findable zero crossing).
     * 
     * @param startPoint This argument specifies the point at which to
     * start the search.
     * 
     * @param objectiveFunction This argument is an instance of the
     * functor type.  The functor must accept a single argument of type
     * double, and return a double, and must provide a method
     * derivative(double), which returns the first derivative.
     * 
     * @param epsilon This argument specifies the convergence tolerance.
     * Iteration will stop when the algorithm finds a value which, when
     * passed to objectiveFunction, results in a return value with
     * magnitude less than epsilon.
     * 
     * @param maxIterations This argument specifies how many iterations
     * should be permitted before giving up the search (& throwing an
     * exception).
     * 
     * @return The return value is a value which, when passed to
     * objectiveFunction, results in a return value with magnitude less
     * than epsilon.
     */
    template <class FUNCTOR>
    double newtonRaphson(double startPoint, FUNCTOR objectiveFunction,
                         double epsilon, size_t maxIterations);
  
    /** 
     * This function computes the normalized correlation of two Array1D
     * arguments.  This is equivalent to (but slightly more efficient
     * than) independently normalizing the signal in each Array1D to
     * zero mean and unit variance, and then computing the dot product
     * of the two normalized Array1D elements.  The computation is
     * carried out in double precision floating point.  For more
     * information on normalized correlation, which is often called
     * "correlation coefficient," please refer to R. O. Duda and
     * P. E. Hart, Pattern Classification and Scene Analysis, Wiley, NY,
     * 1973.
     * 
     * @param signal0 This argument is an Array1D instance containing the
     * first of the signals to be correlated.
     *
     * @param signal1 This argument is an Array1D instance containing the
     * second of the signals to be correlated.
     *
     * @return The return value is the normalized correlation of the two
     * signals.
     */
    template <class Type>
    inline double
    normalizedCorrelation(const Array1D<Type>& signal0,
                          const Array1D<Type>& signal1);

    
    /** 
     * This function is equivalent to normalizedCorrelation(const
     * Array1D&, const Array1D&), except that the computation is carried
     * out using the type specified by the third argument.
     * 
     * @param signal0 This argument is an Array1D instance containing the
     * first of the signals to be correlated.
     *
     * @param signal1 This argument is an Array1D instance containing the
     * second of the signals to be correlated.
     *
     * @param typeTag This parameter specifies the type with
     * which the normalized correlation computation will be calculated.
     *
     * @return The return value is the normalized correlation of the two
     * signals.
     */
    template <class Type, class Type2>
    Type2
    normalizedCorrelation(const Array1D<Type>& signal0,
                          const Array1D<Type>& signal1,
                          type_tag<Type2> typeTag);

    /** 
     * This function returns an Array1D of the specified size and type
     * in which the value of every element is initialized to 1.
     * 
     * @param size This argument specifies the number of elements in the
     * returned array.
     *
     * @param typeTag This is a type_tag instance which specifies
     * the element type of the returned array.
     *
     * @return An Array1D instance in which each element is set to zero.
     */
    template <class Type>
    inline Array1D<Type>
    ones(int size, type_tag<Type> typeTag);

    /** 
     * This function returns an Array2D of the specified size and type
     * in which the value of every element is initialized to one.
     * 
     * @param rows This argument specifies the number of rows in the
     * returned array.
     *
     * @param columns This argument specifies the number of columns in the
     * returned array.
     *
     * @param typeTag This is a type_tag instance which specifies
     * the element type of the returned array.
     *
     * @return An Array2D instance in which each element is set to zero.
     */
    template <class Type>
    inline Array2D<Type>
    ones(int rows, int columns, type_tag<Type> typeTag);

    /** 
     * This function computes the outer product of two input Array1D
     * instances.  The computation is done using the ProductType
     * specified by the appropriate NumericTraits class.  The result is
     * Array2D instance with values such that the element in the i-th
     * column and j-th row has a value equal to the product of the i-th
     * element of the second input argument and the j-th element of the
     * first input argument.
     *
     * @param array0 A 1D array, the first argument of the outer product.
     *
     * @param array1 A 1D array, the second argument of the outer product.
     *
     * @return The outer product of the two arguments.
     */
    template <class Type>
    inline Array2D<typename NumericTraits<Type>::ProductType>
    outerProduct(const Array1D<Type>& array0, const Array1D<Type>& array1);

    /** 
     * This function computes the outer product of two input Array1D
     * instances and allows the user to control which type is used to do
     * the calculation.  The computation is done using the ProductType
     * specified by the appropriate NumericTraits class.  The result is
     * Array2D instance with values such that the element in the i-th
     * column and j-th row has a value equal to the product of the i-th
     * element of the second input argument and the j-th element of the
     * first input argument.
     *
     * @param array0 A 1D array, the first argument of the outer product.
     *
     * @param array1 A 1D array, the second argument of the outer product.
     *
     * @param typeTag The third argument is a type tag
     * indicating which type is to be used to perform the computation.
     *
     * @return The outer product of the two arguments.
     */
    template <class Type0, class Type1, class Type2>
    Array2D<Type2>
    outerProduct(const Array1D<Type0>& array0, const Array1D<Type1>& array1,
                 type_tag<Type2> typeTag);

    /** 
     * This function returns an Array1D in which the first element has
     * value equal to argument "start," and each subsequent element has
     * value equal to the previous element plus argument "stride."  The
     * length of the array is equal to largest integer less than the
     * quantity ((stop - start) / stride).  It is an error for stride to
     * have a different sign than (stop - start).  The element type of
     * the returned array is determined by the type of the arguments.
     * For example,
     *
     *  range(2, 7, 1);
     *
     * will return
     *
     *  Array1D([2, 3, 4, 5, 6]);
     *
     * and
     *
     *  range(2.0, -6.0, -1.5);
     *
     * will return
     *
     *  Array1D([2.0, 0.5, -1.0, -2.5, -4.0, -5.5]);
     * 
     * @param start This argument specifies the value of the first
     * element of the returned Array1D instance.
     *
     * @param stop This argument specifies an exclusive bound for the
     * element values of the returned array.  If stride is less than
     * 0.0, then this will be an exclusive lower bound, else it will be
     * an exclusive upper bound.
     *
     * @param stride This argument specifies the increment which will be
     * added to each subsequent element in the output array.  It is an
     * error for stride to be equal to 0.
     *
     * @return The return value is an Array1D instance having regularly
     * spaced elements progressing from start toward stop.
     */
    template <class Type>
    Array1D<Type>
    range(Type start, Type stop, Type stride=1);

    /** 
     * This function takes an Array2D argument and returns an Array1D
     * instance which references the same data.  That is, changing an
     * element of the Array1D instance will change the corresponding
     * element of the input Array2D instance.  The elements of the
     * returned Array1D are in native storage order for the input array.
     * Input arrays of type Array2D happen to have their elements in
     * row-major order.  Will this always be true?  Hmm...
     * 
     * @param inputArray This argument is the Array2D instance from
     * which to take the data.
     *
     * @return An Array1D instance referencing the same data as the
     * input Array2D instance.
     */
    template <class Type>
    inline Array1D<Type>
    ravel(Array2D<Type>& inputArray) {return inputArray.ravel();}

    /** 
     * This function takes a const Array2D argument and returns a const
     * Array1D instance which references the same data.  It is
     * equivalent to ravel(Array2D&), but for const arrays.
     * 
     * @param inputArray This argument is the Array2D instance from
     * which to take the data.
     *
     * @return An Array1D instance referencing the same data as the
     * input Array2D instance.
     */
    template <class Type>
    inline const Array1D<Type>
    ravel(const Array2D<Type>& inputArray) {return inputArray.ravel();}


    /** 
     * This function computes the RMS (Root Mean Square) value of the
     * elements of its argument.  The computation is performed using the
     * precision specified by NumericTraits<Type>::ProductType, but is
     * returned as an instance of Type.
     * 
     * @param array0 This argument is the input array, from which the
     * RMS value will be computed.
     *
     * @return The return is the RMS value of the elements of array0.
     */
    template <class Type>
    inline Type
    rms(const Array1D<Type>& array0);


    /** 
     * This function computes the RMS (Root Mean Square) value of the
     * elements of its argument, and allows the user to specify the
     * precision with which the computation is carried out.  The
     * computation is performed using the type specified by the second
     * argument to the rms() function call.
     * 
     * @param array0 This argument is the input array, from which the
     * RMS value will be computed.
     *
     * @param typeTag This argument specifies the type which
     * will be used to perform the computation, and also specifies the
     * return type of the function.
     *
     * @return The return is the RMS value of the elements of array0.
     */
    template <class Type0, class Type1>
    Type1
    rms(const Array1D<Type0>& array0, type_tag<Type1> typeTag);


    /**
     * rowIndices(rows, columns): Returns an Array2D in which each
     * element contains the index of its row.  The element type of the
     * returned array depends on the type_tag argument.  For example:
     *
     *   rowIndices(2, 3, type_tag<int>());
     *
     * will return:
     *
     *   Array2D<int>([[0, 0, 0],
     *                 [1, 1, 1]]);
     * 
     * @param rows This argument specifies the number of rows in the
     * returned array.
     *
     * @param columns This argument specifies the number of columns in
     * the returned array.
     *
     * @param typeTag This argument specifies the element type of
     * the returned array.
     *
     * @return The return value is an Array2D of the specified size and
     * type in which all elements in the first row have value
     * initialized to 0, all elements in the second row have value
     * initialized to 1, and so on.
     */
    template <class Type>
    Array2D<Type>
    rowIndices(size_t rows, size_t columns, type_tag<Type> typeTag);


    /** 
     * This function returns true if the two arrays have the same shape,
     * false otherwise.  Two arrays are considered to have the same
     * shape if their sizes are equal along each axis.
     * 
     * @param array0 This argument is the first of the two arrays to be
     * compared.
     *
     * @param array1 This argument is the second of the two arrays to be
     * compared.
     *
     * @return The return value is true if the two arrays have matching
     * shape, false otherwise.
     */
    template <class Type0, class Type1>
    inline bool
    shapeMatch(const Array1D<Type0>& array0, const Array1D<Type1>& array1);


    /** 
     * This function returns true if the two arrays have the same shape,
     * false otherwise.  Two arrays are considered to have the same
     * shape if their sizes are equal along each axis.
     * 
     * @param array0 This argument is the first of the two arrays to be
     * compared.
     *
     * @param array1 This argument is the second of the two arrays to be
     * compared.
     *
     * @return The return value is true if the two arrays have matching
     * shape, false otherwise.
     */
    template <class Type0, class Type1>
    inline bool
    shapeMatch(const Array2D<Type0>& array0, const Array2D<Type1>& array1);


    /** 
     * This function returns true if the two arrays have the same shape,
     * false otherwise.  Two arrays are considered to have the same
     * shape if their sizes are equal along each axis.
     * 
     * @param array0 This argument is the first of the two arrays to be
     * compared.
     *
     * @param array1 This argument is the second of the two arrays to be
     * compared.
     *
     * @return The return value is true if the two arrays have matching
     * shape, false otherwise.
     */
    template <class Type0, class Type1>
    inline bool
    shapeMatch(const Array3D<Type0>& array0, const Array3D<Type1>& array1);


    /**
     * skewSymmetric(x): Returns a skew symmetric matrix X such
     * that matrixMultiply(X, y) = cross(x, y)
     * 
     * @param vector0 An Array1D instance, which must have exactly 3 elements.
     *
     * @return An Array2D instance as described above.
     */
    template <class Type>
    inline Array2D<Type>
    skewSymmetric(const Array1D<Type>& vector0);


    /** 
     * This function computes the real roots of the quadratic polynomial
     * c0*x^2 + c1*x + c2 = 0.
     *
     * If c0, c1, and c2 are arrays, then the returned roots will be
     * arrays of the same dimension, and will contain the roots of an
     * array of scalar quadratic equations.
     *
     * Note that the two well known versions of the quadratic formula:
     * 
     *   x = (-c1 +|- sqrt(c1**2 - 4c0*c2)) / (2*c0)
     * 
     * and
     * 
     *   x = 2*c2 / (-c1 +|- sqrt(c1**2 - 4*c0*c2))
     * 
     * both tend to be inaccurate when c0 and/or c2 are small, since
     * then the quantity (-c1 +|- sqrt(c1**2 - 4*c0*c2)) gets very
     * small and loses significance.  Instead we use the form
     * advocated by Press and Flannery (Numerical Recipes):
     *
     *   q = -(1/2)(c1 + sgn(c1)*sqrt(c1**2 - 4*c1*c2))
     *   x1 = q/c0, x2 = c2/q
     *
     * This is just the same as using both of the first two forms, each
     * one for the root at which it is most numerically stable.
     *
     * If checkValidity is set to zero, valid will always be set to
     * (scalar) one, and no attempt will be made to detect whether
     * quadratics have real roots.  This will make your code run
     * faster, but could cause problems if your quadratics don't
     * always have real roots.
     * 
     * @param c0 This argument is the quadratic coefficient of the
     * polynomial.
     * 
     * @param c1 This argument is the linear coefficient of the
     * polynomial.
     * 
     * @param c2 This argument is the constant coefficient of the
     * polynomial.
     * 
     * @param root0 If the polynomial has real roots, this reference
     * argument is used to return the first root.
     * 
     * @param root1 If the polynomial has real roots, this reference
     * argument is used to return the second root.
     * 
     * @param valid If the polynomial has real roots, this reference
     * argument is set to true.  If the polynomial does not have real
     * roots, this polynomial is set to false.
     * 
     * @param checkValidity The first thing solveQuadratic does is check
     * to see if the polynomial has real roots.  By setting
     * checkValidity to false, the user can disable this check.  It
     * makes sense to do this only if the polynomial is already known to
     * have real roots, and execution time is at a premium.
     */
    template <class Type>
    void
    solveQuadratic(Type c0, Type c1, Type c2,
                   Type& root0, Type& root1, bool& valid,
                   bool checkValidity=true);
    
  
    /** 
     * This function computes the standard deviation of a group of
     * scalar samples.  Computation is done using double precision
     * floats, and the result is returned as a double.  This routine
     * eventually calls the function mean(const Array1D&).  This is a
     * little inefficient, since you are probably already calling
     * mean(...).  If this redundant computation is not OK, then we
     * should a add class called "SignalMoments" (or something similar)
     * will compute both mean and standard deviation without any wasted
     * cycles.  Perhaps by the time you read this, such a class will
     * have already been implemented.
     * 
     * @param array0 This argument is the input array, from the elements
     * of which the standard deviation will be computed.
     *
     * @return The return value is the mean of the elements of array0.
     */
    template <class Type>
    inline double
    standardDeviation(const Array1D<Type>& array0);

    /** 
     * This function computes the standard deviation of the elements of
     * its argument, and allows the user to specify the precision with
     * which the computation is carried out.  The computation is
     * performed using the type specified by the second argument to the
     * standardDeviation() function call.  This routine eventually calls
     * the function mean(const Array1D&, ...).  This is a little
     * inefficient, since you are probably already calling mean(...).
     * If this redundant computation is not OK, then we should add a
     * class called "SignalMoments" (or something similar) which will
     * compute both mean and standard deviation without any wasted
     * cycles.  Perhaps by the time you read this, such a class will
     * have already been implemented.
     * 
     * @param array0 This argument is the input array, from the elements
     * of which the standard deviation will be computed.
     *
     * @param typeTag This argument specifies the type which
     * will be used to perform the computation, and also specifies the
     * return type of the function.
     *
     * @return The return value is the mean of the elements of array0.
     */
    template <class Type0, class Type1>
    inline Type1
    standardDeviation(const Array1D<Type0>& array0, type_tag<Type1> typeTag);
  

    /** 
     * This function computes the sum of the elements of its argument.
     * The summation is accumulated into a variable of type
     * NumericTraits<Type>::SumType.
     * 
     * @param array0 This argument is the array to be summed.
     *
     * @return The summation of all the elements of array0.
     */
    template <class Type>
    inline typename NumericTraits<Type>::SumType
    sum(const Array1D<Type>& array0);

    
    /** 
     * This function computes the sum of the elements of its argument.
     * The summation is accumulated into a variable of type Type2,
     * allowing the user to control the precision of the internal
     * summation.
     * 
     * @param array0 This argument is the array to be summed.
     *
     * @param typeTag This is a type_tag instance which
     * specifies the return type.
     *
     * @return The summation of all the elements of array0.
     */
    template <class Type, class Type2>
    Type2
    sum(const Array1D<Type>& array0, type_tag<Type2> typeTag);


    /** 
     * This function computes the sum of those elements of its
     * argument which lie within a rectangular region of interest.
     * The summation is accumulated into a variable of type
     * NumericTraits<Type>::SumType.
     * 
     * @param array0 This argument is the array to be summed.
     *
     * @param upperLeftCorner This argument specifies the upper left
     * corner of the rectangular region to be summed.  The summed
     * region will include the array element corresponding to
     * upperLeftCorner.
     * 
     * @param lowerRightCorner This argument specifies the lower right
     * corner of the rectangular region to be summed.  The summed
     * region will stop one row/column short of lowerRightCorner.
     * 
     * @return The summation of all the elements in the region of
     * interest.
     */
    template <class Type>
    inline typename NumericTraits<Type>::SumType
    sum(const Array2D<Type>& array0,
        const Index2D& upperLeftCorner,
        const Index2D& lowerRightCorner);

    
    /** 
     * This function computes the sum of those elements of its
     * argument which lie within a rectangular region of interest.
     * The summation is accumulated into a variable of type Type2,
     * allowing the user to control the precision of the internal
     * summation.
     * 
     * @param array0 This argument is the array to be summed.
     *
     * @param upperLeftCorner This argument specifies the upper left
     * corner of the rectangular region to be summed.  The summed
     * region will include the array element corresponding to
     * upperLeftCorner.
     * 
     * @param lowerRightCorner This argument specifies the lower right
     * corner of the rectangular region to be summed.  The summed
     * region will stop one row/column short of lowerRightCorner.
     * 
     * @param typeTag This is a type_tag instance which
     * specifies the return type.
     *
     * @return The summation of all the elements in the region of
     * interest.
     */
    template <class Type, class Type2>
    Type2
    sum(const Array2D<Type>& array0,
        const Index2D& upperLeftCorner,
        const Index2D& lowerRightCorner,
        type_tag<Type2> typeTag);


    /** 
     * This function returns an array made up of only those elements
     * of dataArray that correspond to indices in indexArray.  For
     * example, in the code below, resultArray should end up with
     * four elements ([2.0, 1.0, 1.0, 4.0]):
     *
     * @code
     *   Array1D<double> dataArray("[0.0, 1.0, 2.0, 3.0, 4.0, 5.0]");
     *   Array1D<unsigned int> indexArray("[2, 1, 1, 4]");
     *   Array1D<double> resultArray = take(dataArray, indexArray);
     * @endCode
     *
     * It is an error if indexArray contains indices that are not
     * valid for dataArray.
     * 
     * @param dataArray This argument is the array from which to draw
     * elements.
     * 
     * @param indexArray This argument contains, in order, the indices
     * of the elements that should be included in the output.
     * 
     * @return The return value is an array containing those elements
     * selected by indexArray.
     */
    template <class Type, class IntegralType>
    Array1D<Type>
    take(const Array1D<Type>& dataArray,
         const Array1D<IntegralType>& indexArray);
    
         
    /** 
     * This function returns an array made up of only those elements
     * of dataArray that correspond to indices in indexArray.  Its
     * operation is just like that of take(Array1D const&, Array1D
     * const&), with the additional detail that the input array is
     * flattened before the take operation.  For example, in the code
     * below, resultArray should end up with four elements ([2.0, 1.0,
     * 1.0, 4.0]):
     *
     * @code
     *   Array2D<double> dataArray("[[0.0, 1.0, 2.0], [3.0, 4.0, 5.0]]");
     *   Array1D<unsigned int> indexArray("[2, 1, 1, 4]");
     *   Array1D<double> resultArray = take(dataArray, indexArray);
     * @endCode
     *
     * It is an error if indexArray contains indices that are less
     * than zero or greater than (dataArray.size() - 1).
     * 
     * @param dataArray This argument is the array from which to draw
     * elements.
     * 
     * @param indexArray This argument contains, in order, the indices
     * of the elements that should be included in the output.
     * 
     * @return The return value is an array containing those elements
     * selected by indexArray.
     */
    template <class Type, class IntegralType>
    inline Array1D<Type>
    take(const Array2D<Type>& dataArray,
         const Array1D<IntegralType>& indexArray);
    
         
    /** 
     * This function works just like take(Array2D const&, Array1D
     * const&), with the exception that the input array is not
     * flattened, and entire rows (or columns) are selected.  For
     * example, in the code below, resultArray0 should end up with two
     * rows and three columns ([[0.0, 1.0, 2.0], [6.0, 7.0, 8.0]]),
     * while resultArray1 should end up with four rows and two columns
     * ([[0.0, 2.0], [3.0, 5.0], [6.0, 8.0], [9.0, 11.0]]):
     *
     * @code
     *   Array2D<double> dataArray("[[0.0, 1.0, 2.0], "
     *                             " [3.0, 4.0, 5.0], "
     *                             " [6.0, 7.0, 8.0], "
     *                             " [9.0, 10.0, 11.0]]");
     *   Array1D<unsigned int> indexArray("[0, 2]");
     *   Array2D<double> resultArray0 = take(dataArray, indexArray, 0);
     *   Array2D<double> resultArray1 = take(dataArray, indexArray, 1);
     * @endCode
     *
     * It is an error if axis == 0 and indexArray contains indices
     * that are less than zero or greater than (dataArray.rows() - 1),
     * or if axis == 1 and indexArray contains indices that are less
     * than zero or greater than (dataArray.columns() - 1).
     * 
     * @param dataArray This argument is the array from which to draw
     * elements.
     * 
     * @param indexArray This argument contains, in order, the indices
     * of the elements that should be included in the output.
     *
     * @param axis This argument specifies the axis over which to
     * select values.  If axis == 0, then individual rows will be
     * taken.  If axis == 1, then individual columns will be taken.
     * Behavior is undefined if axis is not equal to either 0 or 1.
     * 
     * @return The return value is an array containing those elements
     * selected by indexArray.
     */
    template <class Type, class IntegralType>
    Array2D<Type>
    take(const Array2D<Type>& dataArray,
         const Array1D<IntegralType>& indexArray,
         unsigned int axis);
    
         
    /** 
     * This function computes the variance of a group of scalar samples.
     * Computation is done using double precision floats, and the result
     * is returned as a double.  This routine eventually calls the
     * function mean(const Array1D&).  This is a little inefficient,
     * since you are probably already calling mean(...).  If this
     * redundant computation is not OK, then we should add a class
     * called "SignalMoments" (or something similar) which will compute
     * both mean and variance without any wasted cycles.  Perhaps by the
     * time you read this, such a class will have already been
     * implemented.
     * 
     * @param array0 This argument is the input array, from the elements
     * of which the variance will be computed.
     *
     * @return The return value is the variance of the elements of array0.
     */
    template <class Type>
    inline double
    variance(const Array1D<Type>& array0);

    /** 
     * This function computes the variance, of the elements of its
     * argument, and allows the user to specify the precision with which
     * the computation is carried out.  The computation is performed
     * using the type specified by the second argument to the variance()
     * function call.  This routine eventually calls the function
     * mean(const Array1D&, ...).  This is a little inefficient, since
     * you are probably already calling mean(...).  If this redundant
     * computation is not OK, then we should add a class called
     * "SignalMoments" (or something similar) which will compute both
     * mean and variance without any wasted cycles.  Perhaps by the time
     * you read this, such a class will have already been implemented.
     * 
     * @param array0 This argument is the input array, from the elements
     * of which the variance will be computed.
     *
     * @param typeTag This argument specifies the type which
     * will be used to perform the computation, and also specifies the
     * return type of the function.
     *
     * @return The return value is the mean of the elements of array0.
     */
    template <class Type0, class Type1>
    inline Type1
    variance(const Array1D<Type0>& array0, type_tag<Type1> typeTag);

    /** 
     * This function returns an Array1D of the specified size and type
     * in which the value of every element is zero.
     * 
     * @param size This argument specifies the number of elements in the
     * returned array.
     *
     * @param typeTag This is a type_tag instance which specifies
     * the element type of the returned array.
     *
     * @return An Array1D instance in which each element is set to zero.
     */
    template <class Type>
    inline Array1D<Type>
    zeros(size_t size, type_tag<Type> typeTag);

    /** 
     * This function returns an Array2D of the specified size and type
     * in which the value of every element is zero.
     * 
     * @param rows This argument specifies the number of rows in the
     * returned array.
     *
     * @param columns This argument specifies the number of columns in the
     * returned array.
     *
     * @param typeTag This is a type_tag instance which specifies
     * the element type of the returned array.
     *
     * @return An Array2D instance in which each element is set to zero.
     */
    template <class Type>
    inline Array2D<Type>
    zeros(size_t rows, size_t columns, type_tag<Type> typeTag);

    /** 
     * This function returns an Array3D of the specified size and type
     * in which the value of every element is zero.
     * 
     * @param shape0 This argument specifies the number of slices in the
     * returned array.
     *
     * @param shape1 This argument specifies the number of rows in the
     * returned array.
     *
     * @param shape2 This argument specifies the number of columns in the
     * returned array.
     *
     * @param typeTag This is a type_tag instance which specifies
     * the element type of the returned array.
     *
     * @return An Array3D instance in which each element is set to zero.
     */
    template <class Type>
    inline Array3D<Type>
    zeros(size_t shape0, size_t shape1, size_t shape2, type_tag<Type> typeTag);

  } // namespace numeric

} // namespace dlr


/* ======= Declarations to maintain compatibility with legacy code. ======= */

namespace dlr {

  using numeric::abs;
  using numeric::abs;
  using numeric::allFalse;
  using numeric::allTrue;
  using numeric::anyFalse;
  using numeric::anyTrue;
  using numeric::argmax;
  using numeric::argmin;
  using numeric::argsort;
  using numeric::axisMax;
  using numeric::axisMaximum;
  using numeric::axisMin;
  using numeric::axisMinimum;
  using numeric::axisSum;
  using numeric::columnIndices;
  using numeric::compress;
  using numeric::count;
  using numeric::cross;
  using numeric::dot;
  using numeric::equivalentMatrix;
  using numeric::getMeanAndCovariance;
  using numeric::identity;
  using numeric::ln;
  using numeric::logicalNot;
  using numeric::magnitude;
  using numeric::magnitudeSquared;
  using numeric::matrixMultiply;
  using numeric::maximum;
  using numeric::mean;
  using numeric::minimum;
  using numeric::newtonRaphson;
  using numeric::normalizedCorrelation;
  using numeric::ones;
  using numeric::outerProduct;
  using numeric::range;
  using numeric::ravel;
  using numeric::rms;
  using numeric::rowIndices;
  using numeric::shapeMatch;
  using numeric::skewSymmetric;
  using numeric::solveQuadratic;
  using numeric::standardDeviation;
  using numeric::sum;
  using numeric::variance;
  using numeric::zeros;

} // namespace dlr


/* ================ Definitions follow ================ */

#include <cmath>
#include <sstream>
#include <numeric>
//#include <algorithm>

namespace dlr {

  namespace numeric {
    
    /**
     ** @cond privateCode
     **
     ** "Private" namespace containing some functors to make
     ** implementation easier in utilities.h.  Note, since this
     ** (regrettably) has to be part of the .h file, it's important not
     ** to put anything in this namespace which is not either a template
     ** or an inline function.  If you violate this rule, you will
     ** probably get multiply defined symbols when you link your
     ** program.
     **/
    namespace privateCode {

      /**
       ** This is the generic absFunctor implementation.  The purpose of
       ** absFunctor is to compute the absolute value of the
       ** operator()(const Type&) input argument.
       **/
      template <class Type>
      struct absFunctor : public std::unary_function<Type, Type> {
        /** 
         * This is the operator which does all the work.  Since there's
         * no generic way to compute the absolute value of a type, the
         * generic implementation simply throws an exception.
         * 
         * @return This function throws an exception, and so does not return.
         */
        inline Type operator()(const Type& input) {
          DLR_THROW3(NotImplementedException,
                     "absFunctor<Type>::operator()(const Type&)",
                     "absFunctor must be specialized for each type.");
          return static_cast<Type>(0);
        }
      };

      /**
       ** This functor specializes absFunctor for long double floats.
       **/
      template <>
      struct absFunctor<long double>
        : public std::unary_function<long double, long double> {
        /** 
         * This is the operator which does all the work.  It returns the
         * absolute value of its input argument.
         * 
         * @return This function throws an exception, and so does not return.
         */
        inline double operator()(const long double& input) {
          // fabsl doesn't appear to be commonly available.
          // return std::fabsl(input);
          return (input > 0.0) ? input : -input;
        }
      };


      /**
       ** This functor specializes absFunctor for double precision floats.
       **/
      template <>
      struct absFunctor<double> : public std::unary_function<double, double> {
        /** 
         * This is the operator which does all the work.  It returns the
         * absolute value of its input argument.
         * 
         * @return This function throws an exception, and so does not return.
         */
        inline double operator()(const double& input) {
          return std::fabs(input);
        }
      };


      /**
       ** This functor specializes absFunctor for single precision floats.
       **/
      template <>
      struct absFunctor<float> : public std::unary_function<float, float> {
        /** 
         * This is the operator which does all the work.  It returns the
         * absolute value of its input argument.
         * 
         * @return This function throws an exception, and so does not return.
         */
        inline float operator()(const float& input) {
          // Strange.  fabsf shows up in the global namespace.  Is this
          // a bug in the g++ 3.3 stand library?
          return fabsf(input);
        }
      };


      /**
       ** This functor specializes absFunctor for long ints.
       **/
      template <>
      struct absFunctor<long int>
        : public std::unary_function<long int, long int> {
        /** 
         * This is the operator which does all the work.  It returns the
         * absolute value of its input argument.
         * 
         * @return This function throws an exception, and so does not return.
         */
        inline long int operator()(const long int& input) {
          return std::labs(input);
        }
      };


      /**
       ** This functor specializes absFunctor for long long ints.
       **/
      template <>
      struct absFunctor<long long int>
        : public std::unary_function<long long int, long long int> {
        /** 
         * This is the operator which does all the work.  It returns the
         * absolute value of its input argument.
         * 
         * @return This function throws an exception, and so does not return.
         */
        inline long long int operator()(const long long int& input) {
          // Many compilers don't support C99.
          // return std::llabs(input);
          return input >= 0LL ? input : -input;
        }
      };


      /**
       ** This functor specializes absFunctor for ints.
       **/
      template <>
      struct absFunctor<int> : public std::unary_function<int, int> {
        /** 
         * This is the operator which does all the work.  It returns the
         * absolute value of its input argument.
         * 
         * @return This function throws an exception, and so does not return.
         */
        inline int operator()(const int& input) {
          return std::abs(input);
        }
      };


    }
    /** @endcond **/


    // This function returns an array of the same size and element type
    // as its input argument, in which each element is set to the
    // absolute value of the corresponding element of the input array.
    template <class Type>
    Array1D<Type>
    abs(const Array1D<Type>& array0)
    {
      Array1D<Type> result(array0.size());
      std::transform(array0.begin(), array0.end(), result.begin(),
                     privateCode::absFunctor<Type>());
      return result;
    }

    // This function returns an array of the same size and element type
    // as its input argument, in which each element is set to the
    // absolute value of the corresponding element of the input array.
    template <class Type>
    Array2D<Type>
    abs(const Array2D<Type>& array0)
    {
      Array2D<Type> result(array0.rows(), array0.columns());
      std::transform(array0.begin(), array0.end(), result.begin(),
                     privateCode::absFunctor<Type>());
      return result;
    }


    // This function returns true if each element of its argument is
    // false, and returns false otherwise.
    template <class Type>
    bool
    allFalse(const Array1D<Type>& array0)
    {
      for(typename Array1D<Type>::const_iterator iter = array0.begin();
          iter != array0.end();
          ++iter) {
        if(*iter) {
          return false;
        }
      }
      return true;
    }

  
    // This function returns true if each element of its argument is
    // true, and returns false otherwise.
    template <class Type>
    bool
    allTrue(const Array1D<Type>& array0)
    {
      for(typename Array1D<Type>::const_iterator iter = array0.begin();
          iter != array0.end();
          ++iter) {
        if(!(*iter)) {
          return false;
        }
      }
      return true;
    }

  
    // This function returns true if any element of its argument is
    // false, and returns false otherwise.
    template <class Type>
    inline bool
    anyFalse(const Array1D<Type>& array0)
    {
      return !allTrue(array0);
    }

  
    // This function returns true if any element of its argument is
    // true, and returns false otherwise.
    template <class Type>
    inline bool
    anyTrue(const Array1D<Type>& array0)
    {
      return !allFalse(array0);
    }

  
    // This function returns the index of the largest element of its input
    // array.
    template <class Type>
    inline size_t
    argmax(const Array1D<Type>& array0)
    {
      return argmax(array0, std::less<Type>());
    }


    // This function returns the index of the largest element of its input
    // sequence.
    template <class IterType>
    inline size_t
    argmax(IterType beginIter, IterType endIter)
    {
      return argmax(beginIter, endIter, std::less<IterType>());
    }


    // This function returns the index of the largest element of its
    // input array, where largeness is defined by the second argument.
    template <class Type, class Functor>
    inline size_t
    argmax(const Array1D<Type>& array0, Functor comparator)
    {
      return argmax(array0.begin(), array0.end(), comparator);
    }
  

    // This function returns the index of the largest element of its
    // input sequence, where largeness is defined by the second argument.
    template <class IterType, class Functor>
    inline size_t
    argmax(IterType beginIter, IterType endIter, Functor comparator)
    {
      return static_cast<size_t>(
        std::max_element(beginIter, endIter, comparator) - beginIter);
    }


    // This function returns an Index2D instance indicating which is
    // the largest element of its input array.
    template <class Type>
    inline Index2D
    argmax2D(const Array2D<Type>& array0)
    {
      return argmax2D(array0, std::less<Type>());
    }


    // This function returns an Index2D instance indicating which is
    // the largest element of its input array, where largeness is
    // defined by the second argument.
    template <class Type, class Functor>
    inline Index2D
    argmax2D(const Array2D<Type>& array0, Functor comparator)
    {
      size_t ravelIndex = argmax(array0.ravel(), comparator);
      size_t row = ravelIndex / array0.columns(); // Int division.
      size_t column = ravelIndex - row * array0.columns();
      return Index2D(row, column);
    }
  

    // This function returns the index of the smallest element of its input
    // array.
    template <class Type>
    inline size_t
    argmin(const Array1D<Type>& array0)
    {
      return argmin(array0, std::less<Type>());
    }


    // This function returns the index of the smallest element of its
    // input array, where largeness is defined by the second argument.
    template <class Type, class Functor>
    inline size_t
    argmin(const Array1D<Type>& array0, Functor comparator)
    {
      const Type* minPtr = std::min_element(array0.begin(), array0.end(),
                                            comparator);
      return static_cast<size_t>(minPtr - array0.begin());
    }


    // This function returns an Index2D instance indicating which is
    // the smallest element of its input array.
    template <class Type>
    inline Index2D
    argmin2D(const Array2D<Type>& array0)
    {
      return argmin2D(array0, std::less<Type>());
    }


    // This function returns an Index2D instance indicating which is
    // the smallest element of its input array, where smallness is
    // defined by the second argument.
    template <class Type, class Functor>
    inline Index2D
    argmin2D(const Array2D<Type>& array0, Functor comparator)
    {
      size_t ravelIndex = argmin(array0.ravel(), comparator);
      size_t row = ravelIndex / array0.columns(); // Int division.
      size_t column = ravelIndex - row * array0.columns();
      return Index2D(row, column);
    }


    // This function returns an array of indices, result, so that the
    // sequence (array0[result[0]], array0[result[1]],
    // array0[result[2]], ...) is sorted from smallest to largest using
    // operator<().
    template <class Type>
    Array1D<size_t>
    argsort(const Array1D<Type>& array0)
    {
      Array1D< std::pair<Type, size_t> > sortVector(array0.size());
      Array1D<size_t> resultVector(array0.size());

      for(size_t index0 = 0; index0 < array0.size(); ++index0) {
        sortVector[index0] = std::pair<Type, size_t>(array0[index0], index0);
      }
      std::sort(sortVector.begin(), sortVector.end());
      std::transform(sortVector.begin(), sortVector.end(), resultVector.begin(),
                     ExtractSecondFunctor<Type, size_t>());
      return resultVector;    
    }

  
//   // This function returns an array of indices, result, so that the
//   // sequence (array0[result[0]], array0[result[1]],
//   // array0[result[2]], ...) is sorted from smallest to largest using
//   // the supplied comparison operator.
//   template <class Type, class Functor>
//   Array1D<size_t>
//   argsort(const Array1D<Type>& array0, Functor comparator)
//   {
//     Array1D< std::pair<Type, size_t> > sortVector(array0.size());
//     Array1D<size_t> resultVector(array0.size());

//     for(size_t index0 = 0; index0 < array0.size(); ++index0) {
//       sortVector[index0] = std::pair<Type, size_t>(array0[index0], index0);
//     }
//     std::sort(sortVector.begin(), sortVector.end(), comparator);
//     std::transform(sortVector.begin(), sortVector.end(), resultVector.begin(),
//                    ExtractSecondFunctor<Type, size_t>());
//     return resultVector;    
//   }

  
    // This function returns an Array1D in which each element has the
    // value of the largest element in one row or column of the input
    // Array2D.
    template <class Type>
    inline Array1D<Type>
    axisMaximum(const Array2D<Type>& array0, size_t axis)
    {
      return axisMaximum(array0, axis, std::less<Type>());
    }


    // This function returns an Array1D in which each element has the
    // value of the largest element in one row or column of the input
    // Array2D, where largeness is defined by the third argument.
    template <class Type, class Functor>
    Array1D<Type>
    axisMaximum(const Array2D<Type>& array0, size_t axis, Functor comparator)
    {
      Array1D<Type> result;
      switch(axis) {
      case 0:
        result.reinit(array0.columns());
        for(size_t column = 0; column < array0.columns(); ++column) {
          const Type* dataPtr = array0.data(0, column);
          Type columnMax = *dataPtr;
          size_t stride = array0.columns();
          for(size_t row = 0; row < array0.rows(); ++row) {
            if(!comparator(*dataPtr, columnMax)) {
              columnMax = *dataPtr;
            }
            dataPtr += stride;
          }
          result(column) = columnMax;
        }
        break;
      case 1:
        result.reinit(array0.rows());
        for(size_t row = 0; row < array0.rows(); ++row) {
          const Type* dataPtr = array0.data(row, 0);
          Type rowMax = *dataPtr;
          for(size_t column = 0; column < array0.columns(); ++column) {
            if(!comparator(*dataPtr, rowMax)) {
              rowMax = *dataPtr;
            }
            ++dataPtr;
          }
          result(row) = rowMax;
        }
        break;
      default:
        std::ostringstream message;
        message << "Axis " << axis << " is invalid for an Array2D.";
        DLR_THROW3(IndexException, "axisMaximum(const Array2D&, size_t, ...)",
                   message.str().c_str());
        break;
      }
      return result;  
    }


    // This function returns an Array1D in which each element has the
    // value of the smallest element in one row or column of the input
    // Array2D.
    template <class Type>
    inline Array1D<Type>
    axisMinimum(const Array2D<Type>& array0, size_t axis)
    {
      return axisMinimum(array0, axis, std::less<Type>());
    }


    // This function returns an Array1D in which each element has the
    // value of the smallest element in one row or column of the input
    // Array2D, where smallness is defined by the third argument.
    template <class Type, class Functor>
    Array1D<Type>
    axisMinimum(const Array2D<Type>& array0, size_t axis, Functor comparator)
    {
      Array1D<Type> result;
      switch(axis) {
      case 0:
        result.reinit(array0.columns());
        for(size_t column = 0; column < array0.columns(); ++column) {
          const Type* dataPtr = array0.data(0, column);
          Type columnMax = *dataPtr;
          size_t stride = array0.columns();
          for(size_t row = 0; row < array0.rows(); ++row) {
            if(comparator(*dataPtr, columnMax)) {
              columnMax = *dataPtr;
            }
            dataPtr += stride;
          }
          result(column) = columnMax;
        }
        break;
      case 1:
        result.reinit(array0.rows());
        for(size_t row = 0; row < array0.rows(); ++row) {
          const Type* dataPtr = array0.data(row, 0);
          Type rowMax = *dataPtr;
          for(size_t column = 0; column < array0.columns(); ++column) {
            if(comparator(*dataPtr, rowMax)) {
              rowMax = *dataPtr;
            }
            ++dataPtr;
          }
          result(row) = rowMax;
        }
        break;
      default:
        std::ostringstream message;
        message << "Axis " << axis << " is invalid for an Array2D.";
        DLR_THROW3(IndexException, "axisMinimum(const Array2D&, size_t, ...)",
                   message.str().c_str());
        break;
      }
      return result;  
    }


    // This function returns an Array1D in which each element has the
    // sum of one row or column of the input Array2D.
    template <class Type>
    inline Array1D<typename NumericTraits<Type>::SumType>
    axisSum(const Array2D<Type>& array0, size_t axis)
    {
      return axisSum(array0, axis,
                     type_tag<typename NumericTraits<Type>::SumType>());
    }

    // This function returns an Array1D in which each element has the
    // sum of one row or column of the input Array2D.
    template <class Type, class ResultType>
    inline Array1D<ResultType>
    axisSum(const Array2D<Type>& array0, size_t axis, type_tag<ResultType>)
    {
      return axisSum(array0, axis, type_tag<ResultType>(),
                     static_cast<ResultType>(0), std::plus<ResultType>());
    }

    // This function returns an Array1D in which each element has the
    // sum of one row or column of the input Array2D.
    template <class Type, class ResultType, class Functor>
    Array1D<ResultType>
    axisSum(const Array2D<Type>& array0, size_t axis, type_tag<ResultType>,
            const ResultType& initialValue, Functor adder)
    {
      Array1D<ResultType> result;
      switch(axis) {
      case 0:
        result.reinit(array0.columns());
        for(size_t column = 0; column < array0.columns(); ++column) {
          ResultType columnSum = initialValue;
          const Type* dataPtr = array0.data(0, column);
          size_t stride = array0.columns();
          for(size_t row = 0; row < array0.rows(); ++row) {
            columnSum = adder(columnSum, *dataPtr);
            dataPtr += stride;
          }
          result(column) = columnSum;
        }
        break;
      case 1:
        result.reinit(array0.rows());
        for(size_t row = 0; row < array0.rows(); ++row) {
          ResultType rowSum = initialValue;
          const Type* dataPtr = array0.data(row, 0);
          for(size_t column = 0; column < array0.columns(); ++column) {
            rowSum = adder(rowSum, *dataPtr);
            ++dataPtr;
          }
          result(row) = rowSum;
        }
        break;
      default:
        std::ostringstream message;
        message << "Axis " << axis << " is invalid for an Array2D.";
        DLR_THROW3(IndexException, "axisSum(const Array2D&, size_t, ...)",
                   message.str().c_str());
        break;
      }
      return result;  
    }

    // columnIndices(rows, columns): Returns an Array2D in which each
    // element contains the index of its column.
    template <class Type>
    Array2D<Type>
    columnIndices(size_t rows, size_t columns, type_tag<Type>)
    {
      Array2D<Type> returnValue(rows, columns);
      Array1D<Type> modelRow = range(static_cast<Type>(0),
                                     static_cast<Type>(columns),
                                     static_cast<Type>(1));
      for(size_t row=0; row < rows; ++row) {
        std::copy(modelRow.begin(), modelRow.end(), returnValue.data(row, 0));
      }
      return returnValue;
    }

    // This function selects those elements of an input Array1D which
    // correspond to "true" values of a mask Array1D, and returns an
    // Array1D containing only those elements.
    template <class Type0, class Type1>
    inline Array1D<Type1>
    compress(const Array1D<Type0>& condition,
             const Array1D<Type1>& input)
    {
      size_t numTrue = count(condition);
      return compress(condition, input, numTrue);
    }

    // This function behaves in exactly the same way as compress(const
    // Array1D&, const Array1D&), above, but it permits the user to
    // specify the number of true elements in the condition array.
    template <class Type0, class Type1>
    Array1D<Type1>
    compress(const Array1D<Type0>& condition,
             const Array1D<Type1>& input,
             size_t numTrue)
    {
      if(condition.size() != input.size()) {
        std::ostringstream message;
        message << "Condition and input arguments must have the same "
                << "size, but condition has size = " << condition.size()
                << ", while input has size = " << input.size() << ".";
        DLR_THROW3(ValueException,
                   "compress(const Array1D&, const Array1D&, size_t)",
                   message.str().c_str());
      }
      Array1D<Type1> result(numTrue);
      // copy_mask(input.begin(), input.end(), condition.begin(),
      //           result.begin());
      //
      // Hmm... copy_mask was previously defined in dlrLib3 as follows:
      //
      // template <class In0, class In1, class Out>
      // Out copy_mask(In0 first, In0 last, In1 mask, Out res)
      // {
      //   while(first != last) {
      //     if(*mask) {
      //       *res++ = *first;
      //     }
      //     ++mask;
      //     ++first;
      //   }
      //   return res;
      // }

      typename Array1D<Type1>::const_iterator first = input.begin();
      typename Array1D<Type1>::const_iterator last = input.end();
      typename Array1D<Type0>::const_iterator mask = condition.begin();
      typename Array1D<Type1>::iterator target = result.begin();
      while(first != last) {
        if(*mask) {
          *(target++) = *first;
        }
        ++mask;
        ++first;
      }

      return result;
    }

    // This element counts the number of elements of the input array
    // which evaluate to true, and returns that number.
    template <class Type>
    inline size_t
    count(const Array1D<Type>& x)
    {
      return std::count_if(x.begin(), x.end(), StaticCastFunctor<Type, bool>());
    }

    // This function computes the cross product of two Vector3D
    // instances.
    inline Vector3D
    cross(const Vector3D& vector0, const Vector3D& vector1) {
      return Vector3D((vector0.y() * vector1.z()) - (vector0.z() * vector1.y()),
                      (vector0.z() * vector1.x()) - (vector0.x() * vector1.z()),
                      (vector0.x() * vector1.y()) - (vector0.y() * vector1.x()));
    }

    // This function computes the inner product of two input arrays.
    // The computation is done using the ProductType specified by
    // the appropriate NumericTraits class.
    template <class Type>
    inline typename NumericTraits<Type>::ProductType
    dot(const Array1D<Type>& x, const Array1D<Type>& y)
    {
      return dot(x, y, type_tag<typename NumericTraits<Type>::ProductType>());
    }

    // This function computes the inner product of two input arrays, and
    // allows the user to control which type is used to do the
    // calculation.
    template <class Type0, class Type1, class Type2>
    inline Type2
    dot(const Array1D<Type0>& x, const Array1D<Type1>& y, type_tag<Type2>)
    {
      if(x.size() != y.size()) {
        std::ostringstream message;
        message << "Input arguments must have matching sizes, but x.size() == "
                << x.size() << " and y.size() == " << y.size() << "."
                << std::endl;
        DLR_THROW3(ValueException, "dot()", message.str().c_str());
      }
      return std::inner_product(x.begin(), x.end(), y.begin(),
                                static_cast<Type2>(0));
    }

    // This function computes the inner product of two Vector2D instances.
    inline double
    dot(const Vector2D& vector0, const Vector2D& vector1)
    {
      return vector0.x() * vector1.x() + vector0.y() * vector1.y();
    }

    // This function computes the inner product of two Vector3D instances.
    inline double
    dot(const Vector3D& vector0, const Vector3D& vector1)
    {
      return (vector0.x() * vector1.x() + vector0.y() * vector1.y()
              + vector0.z() * vector1.z());
    }

    // This function computes the matrix X such that A * x = X * vec(A).
    template <class Type>
    Array2D<Type>
    equivalentMatrix(const Array1D<Type>& vector0, size_t rowsInMatrix)
    {
      Array2D<Type> XMatrix = zeros(rowsInMatrix, vector0.size() * rowsInMatrix,
                                    type_tag<Type>());
      for(size_t XRow = 0; XRow != rowsInMatrix; ++XRow) {
        for(size_t index0 = 0; index0 < vector0.length(); ++index0) {
          XMatrix(XRow, index0 * rowsInMatrix + XRow) = vector0(index0);
        }
      }
      return XMatrix;
    }
  
    // This function returns a 2D matrix with the specified shape in
    // which the elements on the diagonal are set to 1, and all other
    // elements are set to 0.
    template <class Type>
    Array2D<Type>
    identity(size_t rows, size_t columns, type_tag<Type>)
    {
      return identity<Type>(rows, columns);
    }


    // This function returns a 2D matrix with the specified shape in
    // which the elements on the diagonal are set to 1, and all other
    // elements are set to 0.
    template <class Type>
    Array2D<Type>
    identity(size_t rows, size_t columns)
    {
      Array2D<Type> returnArray = zeros(rows, columns, type_tag<Type>());
      size_t rank = std::min(rows, columns);
      for(size_t index = 0; index < rank; ++index) {
        returnArray(index, index) = static_cast<Type>(1);
      } 
      return returnArray;
    }


    // This function returns an array in which each element is the
    // natural logarithm of the corresponding element of its input.
    template <class Type>
    Array1D<Type>
    ln(const Array1D<Type>& array0)
    {
      Array1D<Type> result(array0.size());
      // std::transform(array0.begin(), array0.end(), std::log);
      for(size_t index0 = 0; index0 < array0.size(); ++index0) {
        result[index0] = std::log(array0[index0]);      
      }
      return result;
    }


//   // Strange troubles compiling these next two functions.
//
//   // This function returns an array in which each element is the
//   // natural logarithm of the corresponding element of its input.
//   template <>
//   Array1D<float>
//   ln(const Array1D<float>& array0)
//   {
//     Array1D<float> result(array0.size());
//     std::transform(array0.begin(), array0.end(), std::logf);
//     return result;
//   }


//   // This function returns an array in which each element is the
//   // natural logarithm of the corresponding element of its input.
//   template <>
//   Array1D<long double>
//   ln(const Array1D<long double>& array0)
//   {
//     Array1D<long double> result(array0.size());
//     std::transform(array0.begin(), array0.end(), std::logl);
//     return result;
//   }


    // This function returns an array in which each element is the
    // natural logarithm of the corresponding element of its input.
    template <class Type>
    Array2D<Type>
    ln(const Array2D<Type>& array0)
    {
      Array2D<Type> result(array0.size());
      // std::transform(array0.begin(), array0.end(), std::log);
      for(size_t index0 = 0; index0 < array0.size(); ++index0) {
        result[index0] = std::log(array0[index0]);      
      }
      return result;
    }


    // This function returns an Array2D instance in which the value of
    // each element is the logical not of the corresponding element of
    // the input array.
    template <class Type>
    Array2D<Type>
    logicalNot(const Array2D<Type>& array0)
    {
      Array2D<Type> result(array0.rows(), array0.columns());
      std::transform(array0.begin(), array0.end(), result.begin(),
                     unaryComposeFunctor(StaticCastFunctor<bool, Type>(),
                                         std::logical_not<Type>()));
      return result;
    }

  
    // This function computes the magnitude of its input argument.
    template <class Type>
    inline Type
    magnitude(const Array1D<Type>& array0)
    {
      return static_cast<Type>(std::sqrt(magnitudeSquared(array0)));
    }

  
    // This function computes the magnitude of its input argument.
    inline double
    magnitude(const Vector2D& vector0) {
      return std::sqrt(magnitudeSquared(vector0));
    }

  
    // This function computes the magnitude of its input argument.
    inline double
    magnitude(const Vector3D& vector0) {
      return std::sqrt(magnitudeSquared(vector0));
    }


    // This function computes the square of the magnitude of its input
    // argument.
    template <class Type>
    inline typename NumericTraits<Type>::ProductType
    magnitudeSquared(const Array1D<Type>& array0)
    {
      typedef typename NumericTraits<Type>::ProductType ProductType;
      return std::inner_product(array0.begin(), array0.end(),
                                array0.begin(), static_cast<ProductType>(0));
    }


    // This function computes the square of the magnitude of its input
    // argument.
    inline double
    magnitudeSquared(const Vector2D& vector0) {
      return (vector0.x() * vector0.x() + vector0.y() * vector0.y());
    }

  
    // This function computes the square of the magnitude of its input
    // argument.
    inline double
    magnitudeSquared(const Vector3D& vector0) {
      return (vector0.x() * vector0.x() + vector0.y() * vector0.y()
              + vector0.z() * vector0.z());
    }

  
    // This function computes a vector * matrix product.  Assuming the
    // first argument, vector0, represents a column vector and the
    // second argument, matrix0, represents a 2D array, then this function
    // computes the product:
    //   vector0' * matrix0
    // Where the single quote indicates transpose.
    template <class Type>
    inline Array1D<typename NumericTraits<Type>::ProductType>
    matrixMultiply(const Array1D<Type>& vector0, const Array2D<Type>& matrix0)
    {
      return matrixMultiply(
        vector0, matrix0, type_tag<typename NumericTraits<Type>::ProductType>());
    }

    // This function computes a vector * matrix product just like
    // matrixMultiply(const Array1D<Type>&, const Array2D<Type>&), above.
    // This function differs in that the element type of the return
    // value is set explicitly using a third argument.
    template <class Type0, class Type1, class Type2>
    Array1D<Type2>
    matrixMultiply(const Array1D<Type0>& vector0, const Array2D<Type0>& matrix0,
                   type_tag<Type2>)
    {
      if(vector0.size() != matrix0.rows()) {
        std::ostringstream message;
        message << "Can't left-multiply a "
                << matrix0.rows() << " x " << matrix0.columns()
                << " matrix by a " << vector0.size() << " element vector\n";
        DLR_THROW(ValueException, "matrixMultiply()", message.str().c_str());
      }
      Array1D<Type2> result = zeros(matrix0.columns(), type_tag<Type2>());
      size_t index = 0;
      for(size_t row = 0; row < matrix0.rows(); ++row) {
        for(size_t column = 0; column < matrix0.columns(); ++column) {
          result[column] += (static_cast<Type2>(vector0[row])
                             * static_cast<Type2>(matrix0[index]));
          ++index;
        }
      }
      return result;
    }

    // This function computes a matrix * vector product.  Assuming the
    // first argument, matrix0, represents a matrix, and the and the
    // second argument, vector0, represents a column vector, then this
    // function computes the product:
    //  matrix0 * vector0
    template <class Type>
    inline Array1D<typename NumericTraits<Type>::ProductType>
    matrixMultiply(const Array2D<Type>& matrix0, const Array1D<Type>& vector0)
    {
      return matrixMultiply(
        matrix0, vector0, type_tag<typename NumericTraits<Type>::ProductType>());
    }


    // This function computes a matrix * vector product just like
    // matrixMultiply(const Array2D<Type>&, const Array1D<Type>&), above.
    // This function differs in that the element type of the return
    // value is set explicitly using a third argument.
    template <class Type0, class Type1, class Type2>
    Array1D<Type2>
    matrixMultiply(const Array2D<Type0>& matrix0, const Array1D<Type1>& vector0,
                   type_tag<Type2>)
    {
      if(vector0.size() != matrix0.columns()) {
        std::ostringstream message;
        message << "matrixMultiply() -- can't right-multiply a "
                << matrix0.rows() << " x " << matrix0.columns()
                << " matrix by a " << vector0.size() << " element vector\n";
        DLR_THROW(ValueException, "matrixMultiply()", message.str().c_str());
      }
      Array1D<Type2> result(matrix0.rows());
      for(size_t row = 0; row < matrix0.rows(); ++row) {
        result[row] = dot(vector0, matrix0.row(row), type_tag<Type2>());
      }
      return result;
    }

    // This function computes a matrix * matrix product.  That is,
    // the elements of the resulting array are the dot products of the
    // rows of the first argument with the columns of the second argument.
    template <class Type>
    inline Array2D<typename NumericTraits<Type>::ProductType>
    matrixMultiply(const Array2D<Type>& matrix0, const Array2D<Type>& matrix1)
    {
      return matrixMultiply(
        matrix0, matrix1, type_tag<typename NumericTraits<Type>::ProductType>());
    }
    
    // This function computes a matrix * matrix product, just like
    // matrixMultiply(const Array2D<Type>&, const Array2D<Type>&), above.
    // This function differs in that the element type of the return
    // value is set explicitly using a third argument.
    template <class Type0, class Type1, class Type2>
    Array2D<Type2>
    matrixMultiply(const Array2D<Type0>& matrix0, const Array2D<Type1>& matrix1,
                   type_tag<Type2>)
    {
      if(matrix1.rows() != matrix0.columns()) {
        std::ostringstream message;
        message << "matrixMultiply() -- can't left-multiply a "
                << matrix1.rows() << " x " << matrix1.columns()
                << " matrix by a "
                << matrix0.rows() << " x " << matrix0.columns()
                << " matrix\n";
        DLR_THROW(ValueException, "matrixMultiply()", message.str().c_str());
      }
      Array2D<Type2> result = zeros(matrix0.rows(), matrix1.columns(),
                                    type_tag<Type2>());
      for(size_t resultRow = 0; resultRow < result.rows(); ++resultRow) {
        for(size_t resultColumn = 0; resultColumn < result.columns();
            ++resultColumn) {
          for(size_t index = 0; index < matrix0.columns(); ++index) {
            result(resultRow, resultColumn) +=
              (static_cast<Type2>(matrix0(resultRow, index))
               * static_cast<Type2>(matrix1(index, resultColumn)));
          }
        }
      }
      return result;
    }

    // This function returns a copy of the largest element in the input
    // Array1D instance.
    template <class Type>
    inline Type
    maximum(const Array1D<Type>& array0)
    {
      return maximum(array0, std::less<Type>());
    }

    // This function returns a copy of the largest element in the input
    // Array1D instance, where largeness is defined by the return value
    // of the second argument.
    template <class Type, class Functor>
    Type
    maximum(const Array1D<Type>& array0, Functor comparator)
    {
      if(array0.size() == 0) {
        DLR_THROW3(ValueException, "maximum()",
                   "Can't find the maximum element of an empty array.");
      }
      return *std::max_element(array0.begin(), array0.end(), comparator);
    }


    // This function computes the average value, or geometric mean, of
    // the elements of input sequence.
    template <class Iterator, class Type>
    Type
    mean(const Iterator& beginIter, const Iterator& endIter)
    {
      Type meanValue;
      size_t count = 0;
      while(beginIter != endIter) {
        meanValue += *beginIter;
        ++count;
        ++beginIter;
      }
      return meanValue / count;
    }

    
    // This function computes the average value, or geometric mean, of
    // the elements of its argument.
    template <class Type>
    inline double
    mean(const Array1D<Type>& array0)
    {
      if(array0.size() == 0) {
        DLR_THROW3(ValueException, "mean()",
                   "Can't find the mean of an empty array.");
      }
      NumericTypeConversionFunctor<typename NumericTraits<Type>::SumType,
        Type> functor;
      return functor(
        mean(array0, type_tag<typename NumericTraits<Type>::SumType>()));
    }
  

    // This function computes the average value, or geometric mean, of
    // the elements of its argument, and allows the user to specify the
    // precision with which the computation is carried out.
    template <class Type0, class Type1>
    inline Type1
    mean(const Array1D<Type0>& array0, type_tag<Type1>)
    {
      return sum(array0, type_tag<Type1>()) / static_cast<Type1>(array0.size());
    }
  

    // This function estimates the mean and covariance of a set of
    // vectors, which are represented by the rows (or columns) of the
    // input 2D array.
    template <class Type>
    void
    getMeanAndCovariance(const Array2D<Type>& sampleArray,
                         Array1D<double>& meanArray,
                         Array2D<double>& covarianceArray,
                         size_t majorAxis)
    {
      // Check input.
      if(sampleArray.size() == 0) {
        meanArray.reinit(0);
        covarianceArray.reinit(0, 0);
        return;
      }

      // Compute number of samples and sample dimensionality.
      size_t dimensionality;
      size_t numberOfSamples;
      if(majorAxis == 0) {
        dimensionality = sampleArray.columns();
        numberOfSamples = sampleArray.rows();
      } else {
        dimensionality = sampleArray.rows();
        numberOfSamples = sampleArray.columns();
      }
      
      // Now compute the mean value of the sample points.
      // For efficiency, we simultaneously compute covariance.
      // We make use of the identity:
      // 
      //   covariance = E[(x - u)*(x - u)^T] = E[x * x^T] - u * u^T
      // 
      // where E[.] denotes expected value and u is the mean.
      //
      // Note that the sample mean is an unbiased estimator for u, but
      // substituting the sample mean for u in the covariance equation
      // gives a biased estimator of covariance.  We resolve this by
      // using the unbiased estimator:
      //
      //   covariance = (1/(N-1))sum(x * x^T) - (N/(N-1)) * uHat * uHat^T
      //
      // where uHat is the sample mean.

      meanArray.reinit(dimensionality);
      meanArray = 0.0;
      covarianceArray.reinit(dimensionality, dimensionality);
      covarianceArray = 0.0;
      typename Array2D<Type>::const_iterator sampleIterator =
        sampleArray.begin();

      if(majorAxis == 0) {
        for(size_t rowIndex = 0; rowIndex < sampleArray.rows(); ++rowIndex) {
          for(size_t columnIndex = 0; columnIndex < sampleArray.columns();
              ++columnIndex) {
            // Update accumulator for mean.
            meanArray[columnIndex] += *sampleIterator;

            // Update accumulator for covariance.  We only compute the top
            // half of the covariance matrix for now, since it's symmetric.
            typename Array2D<Type>::const_iterator sampleIterator2 =
              sampleIterator;
            for(size_t covarianceIndex = columnIndex;
                covarianceIndex < dimensionality;
                ++covarianceIndex) {
              // Note(xxx): Should fix this to run faster.
              covarianceArray(columnIndex, covarianceIndex) +=
                *sampleIterator * *sampleIterator2;
              ++sampleIterator2;
            }
            ++sampleIterator;
          }
        }
      } else {
        for(size_t rowIndex = 0; rowIndex < sampleArray.rows(); ++rowIndex) {
          for(size_t columnIndex = 0; columnIndex < sampleArray.columns();
              ++columnIndex) {
            // Update accumulator for mean.
            meanArray[rowIndex] += *sampleIterator;

            // Update accumulator for covariance.  We only compute the top
            // half of the covariance matrix for now, since it's symmetric.
            typename Array2D<Type>::const_iterator sampleIterator2 =
              sampleIterator;
            for(size_t covarianceIndex = rowIndex;
                covarianceIndex < dimensionality;
                ++covarianceIndex) {
              // Note(xxx): Should fix this to run faster.
              covarianceArray(rowIndex, covarianceIndex) +=
                *sampleIterator * *sampleIterator2;
              sampleIterator2 += numberOfSamples;
            }
            ++sampleIterator;
          }
        }
      }
    
      // Finish up the computation of the mean.
      meanArray /= static_cast<double>(numberOfSamples);

      // Finish up the computation of the covariance matrix.
      for(size_t index0 = 0; index0 < dimensionality; ++index0) {
        for(size_t index1 = index0; index1 < dimensionality; ++index1) {
          // Add correction to the top half of the covariance matrix.
          covarianceArray(index0, index1) -=
            numberOfSamples * meanArray[index0] * meanArray[index1];
          covarianceArray(index0, index1) /= (numberOfSamples - 1);

          // And copy to the bottom half.
          if(index0 != index1) {
            covarianceArray(index1, index0) = covarianceArray(index0, index1);
          }
        }
      }

    }
  
                    
    // This function returns a copy of the smallest element in the input
    // Array1D instance.
    template <class Type>
    inline Type
    minimum(const Array1D<Type>& array0)
    {
      if(array0.size() == 0) {
        DLR_THROW3(ValueException, "minimum()",
                   "Can't find the minimum element of an empty array.");
      }
      return minimum(array0, std::less<Type>());
    }

    // This function returns a copy of the smallest element in the input
    // Array1D instance, where largeness is defined by the value of the
    // second argument.
    template <class Type, class Functor>
    Type
    minimum(const Array1D<Type>& array0, Functor comparator)
    {
      return *std::min_element(array0.begin(), array0.end(), comparator);
    }


    // This function uses the iterative method of Newton and Raphson to
    // search for a zero crossing of the supplied functor.
    template <class FUNCTOR>
    double newtonRaphson(double startPoint, FUNCTOR objectiveFunction,
                         double epsilon, size_t maxIterations)
    {
      double argument = startPoint;
      for(size_t count = 0; count < maxIterations; ++count) {
        double result = objectiveFunction(argument);
        if(std::fabs(result) < epsilon) {
          return argument;
        }
        double resultPrime = objectiveFunction.derivative(argument);
        if(resultPrime == 0.0) {
          DLR_THROW(ValueException, "newtonRaphson()", "Zero derivative.");
        }
        argument = argument - (result / resultPrime);
      }
      DLR_THROW(ValueException, "newtonRaphson()",
                "Root finding failed to converge.");
      return 0.0;  // keep compiler happy
    }

    // This function computes the normalized correlation of two Array1D
    // arguments.
    template <class Type>
    inline double
    normalizedCorrelation(const Array1D<Type>& signal0,
                          const Array1D<Type>& signal1)
    {
      return normalizedCorrelation(signal0, signal1, type_tag<double>());
    }

    // This function is equivalent to normalizedCorrelation(const
    // Array1D&, const Array1D&), except that the computation is carried
    // out using the type specified by the third argument.
    template <class Type, class Type2>
    Type2
    normalizedCorrelation(const Array1D<Type>& signal0,
                          const Array1D<Type>& signal1,
                          type_tag<Type2>)
    {
      if(signal0.size() != signal1.size()) {
        DLR_THROW(ValueException, "normalizedCorrelation()",
                  "Input arrays must have the same size.");
      }
      Type2 oneOverN =
        static_cast<Type2>(1) / static_cast<Type2>(signal0.size());
      Type2 sum0 = static_cast<Type2>(0);
      Type2 sum1 = static_cast<Type2>(0);
      Type2 sum00 = static_cast<Type2>(0);
      Type2 sum01 = static_cast<Type2>(0);
      Type2 sum11 = static_cast<Type2>(0);
      typedef typename Array1D<Type>::const_iterator SignalIter;
      SignalIter iter0 = signal0.begin();
      SignalIter iter1 = signal1.begin();
      SignalIter end0 = signal0.end();
      while(iter0 != end0) {
        sum0 += *iter0;
        sum1 += *iter1;
        sum00 += (*iter0) * (*iter0);
        sum01 += (*iter0) * (*iter1);
        sum11 += (*iter1) * (*iter1);
        ++iter0;
        ++iter1;
      }
      Type2 numerator = sum01 - oneOverN * sum0 * sum1;
      Type2 denominator = std::sqrt((sum00 - oneOverN * sum0 * sum0)
                                    * (sum11 - oneOverN * sum1 * sum1));
      return numerator / denominator;
    }

    // This function returns an Array1D of the specified size and type
    // in which the value of every element is initialized to 1.
    template <class Type>
    inline Array1D<Type>
    ones(int size, type_tag<Type>)
    {
      Array1D<Type> result(size);
      result = 1;
      return result;
    }

    // This function returns an Array2D of the specified size and type
    // in which the value of every element is initialized to one.
    template <class Type>
    inline Array2D<Type>
    ones(int rows, int columns, type_tag<Type>)
    {
      Array2D<Type> result(rows, columns);
      result = 1;
      return result;
    }

    // This function computes the outer product of two input Array1D
    // instances. 
    template <class Type>
    inline Array2D<typename NumericTraits<Type>::ProductType>
    outerProduct(const Array1D<Type>& x, const Array1D<Type>& y)
    {
      return outerProduct(
        x, y, type_tag<typename NumericTraits<Type>::ProductType>());
    }

    // This function computes the outer product of two input Array1D
    // instances and allows the user to control which type is used to do the
    // computation.
    template <class Type0, class Type1, class Type2>
    Array2D<Type2>
    outerProduct(const Array1D<Type0>& x, const Array1D<Type1>& y,
                 type_tag<Type2>)
    {
      Array2D<Type2> result(x.size(), y.size());
      typename Array2D<Type2>::iterator runningIterator = result.begin();
      typename Array1D<Type1>::const_iterator yBegin = y.begin();
      typename Array1D<Type1>::const_iterator yEnd = y.end();
      for(size_t index = 0; index < x.size(); ++index) {
        std::transform(yBegin, yEnd, runningIterator,
                       std::bind2nd(std::multiplies<Type2>(), x[index]));
        runningIterator += y.size();
      }
      return result;
    }

    // This function returns an Array1D in which the first element has
    // value equal to argument "start," and each subsequent element has
    // value equal to the previous element plus argument "stride."
    template <class Type>
    Array1D<Type>
    range(Type start, Type stop, Type stride)
    {
      if(stride == 0) {
        DLR_THROW3(ValueException, "range(Type, Type, Type)",
                   "Argument \"stride\" must not be equal to 0.");
      }
      int length = static_cast<int>((stop - start) / stride);
      if(stride > 0) {
        if((length * stride) < (stop - start)) {      
          ++length;      
        }
      } else {
        if((length * stride) > (stop - start)) {
          ++length;      
        }
      }
      Array1D<Type> result(length);
      for(int index = 0; index < length; ++index) {
        result(index) = start;
        start += stride;
      }
      return result;
    }
  

    // This function computes the RMS (Root Mean Square) value of the
    // elements of its argument.
    template <class Type>
    inline Type
    rms(const Array1D<Type>& array0)
    {
      NumericTypeConversionFunctor<
        typename NumericTraits<Type>::SumType, Type> functor;
      return functor(rms(array0,
                         type_tag<typename NumericTraits<Type>::ProductType>()));
    }

    // This function computes the RMS (Root Mean Square) value of the
    // elements of its argument, and allows the user to specify the
    // precision with which the computation is carried out.
    template <class Type0, class Type1>
    Type1
    rms(const Array1D<Type0>& array0, type_tag<Type1>)
    {
      Type1 accumulator = 0;
      for(int index = 0; index < array0.size(); ++index) {
        Type1 element = static_cast<Type1>(array0[index]);
        accumulator += element * element;
      }
      return ::sqrt(accumulator / static_cast<Type1>(array0.size()));
    }

    // rowIndices(rows, columns): Returns an Array2D in which each
    // element contains the index of its row.
    template <class Type>
    Array2D<Type>
    rowIndices(size_t rows, size_t columns, type_tag<Type>)
    {
      Array2D<Type> returnValue(rows, columns);
      for(size_t row=0; row < rows; ++row) {
        Type* rowPtr = returnValue.data(row, 0);
        std::fill(rowPtr, rowPtr + columns, static_cast<Type>(row));
      }
      return returnValue;
    }

    // This function returns true if the two arrays have the same shape,
    // false otherwise.
    template <class Type0, class Type1>
    inline bool
    shapeMatch(const Array1D<Type0>& array0, const Array1D<Type1>& array1)
    {
      return array0.size() == array1.size();
    }

    // This function returns true if the two arrays have the same shape,
    // false otherwise.
    template <class Type0, class Type1>
    inline bool
    shapeMatch(const Array2D<Type0>& array0, const Array2D<Type1>& array1)
    {
      return ((array0.rows() == array1.rows())
              && (array0.columns() == array1.columns()));
    }

    // This function returns true if the two arrays have the same shape,
    // false otherwise.
    template <class Type0, class Type1>
    inline bool
    shapeMatch(const Array3D<Type0>& array0, const Array3D<Type1>& array1)
    {
      return ((array0.shape0() == array1.shape0())
              && (array0.shape1() == array1.shape1())
              && (array0.shape2() == array1.shape2()));            
    }

    // skewSymmetric(x): Returns a skew symmetric matrix X such
    // that matrixMultiply(X, y) = cross(x, y).
    template <class Type>
    inline Array2D<Type>
    skewSymmetric(const Array1D<Type>& vector0)
    {
      if(vector0.size() != 3) {
        std::ostringstream message;
        message << "Argument must have exactly 3 elements, but instead has "
                << vector0.size() << "elements.";
        DLR_THROW(ValueException, "skewSymmetric()", message.str().c_str());
      }
      Array2D<Type> returnVal(3, 3);
      returnVal(0, 0) = 0;
      returnVal(0, 1) = -vector0(2);
      returnVal(0, 2) = vector0(1);
  
      returnVal(1, 0) = vector0(2);
      returnVal(1, 1) = 0;
      returnVal(1, 2) = -vector0(0);

      returnVal(2, 0) = -vector0(1);
      returnVal(2, 1) = vector0(0);
      returnVal(2, 2) = 0;
      return returnVal;
    }


    // This function computes the real roots of the quadratic polynomial
    // c0*x^2 + c1*x + c0 = 0.
    template <class Type>
    void
    solveQuadratic(Type c0, Type c1, Type c2,
                   Type& root0, Type& root1, bool& valid, bool)
    {
      valid = solveQuadratic(c0, c1, c2, root0, root1);
    }

    
    // This function computes the standard deviation of a group of
    // scalar samples.
    template <class Type>
    inline double
    standardDeviation(const Array1D<Type>& array0)
    {
      return standardDeviation(array0, type_tag<double>());
    }

    // This function computes the standard deviation, of the elements of
    // its argument, and allows the user to specify the precision with
    // which the computation is carried out.
    template <class Type0, class Type1>
    inline Type1
    standardDeviation(const Array1D<Type0>& array0, type_tag<Type1>)
    {
      return ::sqrt(variance(array0, type_tag<Type1>()));
    }

    // This function computes the sum of all elements in the input
    // array.  The summation is done using the SumType specified by the
    // appropriate NumericTraits class.
    template <class Type>
    inline typename NumericTraits<Type>::SumType
    sum(const Array1D<Type>& array0)
    {
      return sum(array0, type_tag<typename NumericTraits<Type>::SumType>());
    }

    // This function computes the sum of all elements in the input
    // array, and allows the user to control which type is used to do the
    // summation.
    template <class Type, class Type2>
    Type2
    sum(const Array1D<Type>& array0, type_tag<Type2>)
    {
      return std::accumulate(array0.begin(), array0.end(),
                             static_cast<Type2>(0));
    }


    // This function computes the sum of those elements of its
    // argument which lie within a rectangular region of interest.
    template <class Type>
    inline typename NumericTraits<Type>::SumType
    sum(const Array2D<Type>& array0,
        const Index2D& upperLeftCorner,
        const Index2D& lowerRightCorner)
    {
      return sum(array0, upperLeftCorner, lowerRightCorner,
                 type_tag<typename NumericTraits<Type>::SumType>());
    }
      
    
    // This function computes the sum of those elements of its
    // argument which lie within a rectangular region of interest.
    template <class Type, class Type2>
    Type2
    sum(const Array2D<Type>& array0,
        const Index2D& upperLeftCorner,
        const Index2D& lowerRightCorner,
        type_tag<Type2>)
    {
      Type2 result = static_cast<Type2>(0);
      for(int row = upperLeftCorner.getRow();
          row < lowerRightCorner.getRow();
          ++row) {
        typename Array2D<Type>::const_iterator rowBegin = array0.rowBegin(row);
        result += std::accumulate(
          rowBegin + upperLeftCorner.getColumn(),
          rowBegin + lowerRightCorner.getColumn(),
          static_cast<Type2>(0));
      }
      return result;
    }

    
    // This function returns an array made up of only those elements
    // of dataArray that correspond to indices in indexArray.
    template <class Type, class IntegralType>
    Array1D<Type>
    take(const Array1D<Type>& dataArray,
         const Array1D<IntegralType>& indexArray)
    {
      Array1D<Type> resultArray(indexArray.size());
      for(unsigned int ii = 0; ii < indexArray.size(); ++ii) {
        resultArray[ii] = dataArray[indexArray[ii]];
      }
      return resultArray;
    }
    
    
    // This function returns an array made up of only those elements
    // of dataArray that correspond to indices in indexArray.
    template <class Type, class IntegralType>
    Array1D<Type>
    take(const Array2D<Type>& dataArray,
         const Array1D<IntegralType>& indexArray)
    {
      return take(dataArray.ravel(), indexArray);
    }
    
         
    // This function works just like take(Array2D const&, Array1D
    // const&), with the exception that the input array is not
    // flattened, and entire rows (or columns) are selected.
    template <class Type, class IntegralType>
    Array2D<Type>
    take(const Array2D<Type>& dataArray,
         const Array1D<IntegralType>& indexArray,
         unsigned int axis)
    {
      Array2D<Type> resultArray;
      switch(axis) {
      case 0:
        resultArray.reinit(indexArray.size(), dataArray.columns());
        for(unsigned int ii = 0; ii < indexArray.size(); ++ii) {
          unsigned int jj = indexArray[ii];
          std::copy(dataArray.rowBegin(jj), dataArray.rowEnd(jj),
                    resultArray.rowBegin(ii));
        }
        break;
      default:
        resultArray.reinit(dataArray.rows(), indexArray.size());
        for(unsigned int ii = 0; ii < dataArray.rows(); ++ii) {
          typename Array2D<Type>::iterator resultIterator =
            resultArray.rowBegin(ii);
          typename Array2D<Type>::const_iterator dataRowBegin =
            dataArray.rowBegin(ii);
          for(unsigned int jj = 0; jj < indexArray.size(); ++jj) {
            *resultIterator = *(dataRowBegin + indexArray[jj]);
            ++resultIterator;
          }
        }
        break;
      }
      return resultArray;
    }
    
         
    // This function computes the standard deviation of a group of
    // scalar samples.
    template <class Type>
    inline double
    variance(const Array1D<Type>& array0)
    {
      return variance(array0, type_tag<double>());
    }

    // This function computes the standard deviation, of the elements of
    // its argument, and allows the user to specify the precision with
    // which the computation is carried out.
    template <class Type0, class Type1>
    inline Type1
    variance(const Array1D<Type0>& array0, type_tag<Type1>)
    {
      Type1 meanValue = mean(array0, type_tag<Type1>());
      Type1 accumulator = 0;
      for(size_t index0 = 0; index0 < array0.size(); ++index0) {
        Type1 offset = static_cast<Type1>(array0[index0]) - meanValue;
        accumulator += offset * offset;
      }
      return accumulator / static_cast<Type1>(array0.size());
    }

    // This function returns an Array1D of the specified size and type
    // in which the value of every element is zero.
    template <class Type>
    inline Array1D<Type>
    zeros(size_t size, type_tag<Type>)
    {
      Array1D<Type> result(size);
      result = 0;
      return result;
    }

    // This function returns an Array2D of the specified size and type
    // in which the value of every element is zero.
    template <class Type>
    inline Array2D<Type>
    zeros(size_t rows, size_t columns, type_tag<Type>)
    {
      Array2D<Type> result(rows, columns);
      result = 0;
      return result;
    }


    // This function returns an Array3D of the specified size and type
    // in which the value of every element is zero.
    template <class Type>
    inline Array3D<Type>
    zeros(size_t shape0, size_t shape1, size_t shape2, type_tag<Type>)
    {
      Array3D<Type> result(shape0, shape1, shape2);
      result = 0;
      return result;
    }
  
  } // namespace numeric

} // namespace dlr

#endif /* #ifndef DLR_NUMERIC_UTILITIES_H */
