/**
***************************************************************************
* @file dlrNumeric/transform3D.h
*
* Header file declaring Transform3D class.
*
* Copyright (C) 2001-2007 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 1137 $
* $Date: 2009-04-02 18:07:08 -0400 (Thu, 02 Apr 2009) $
***************************************************************************
**/

#ifndef _DLR_TRANSFORM3D_H_
#define _DLR_TRANSFORM3D_H_

#include <dlrNumeric/array2D.h>
#include <dlrNumeric/vector3D.h>

namespace dlr {

  namespace numeric {
    
    // Forward declaration.
    class Transform3DFunctor;

  
    /**
     ** The Transform3D class represents a homogeneous coordinate
     ** transform from one 3D coordinate system to another 3D
     ** coordinate system.
     **/
    class Transform3D {
    public:

      /** 
       * Default constructor.  Initializes to identity.
       */
      inline
      Transform3D()
        : m_00(1.0), m_01(0.0), m_02(0.0), m_03(0.0),
          m_10(0.0), m_11(1.0), m_12(0.0), m_13(0.0),
          m_20(0.0), m_21(0.0), m_22(1.0), m_23(0.0),
          m_30(0.0), m_31(0.0), m_32(0.0) {}

      /** 
       * Build a Transform3D instance by explicitly setting element values
       * as if setting the elements of a 4x4 transformation matrix:
       *    [[a00, a01, a02, a03],
       *     [a10, a11, a12, a13],
       *     [a20, a21, a22, a23],
       *     [a30, a31, a32, a33]]
       * 
       * @param a00 The value of one element of the transformation matrix.
       * @param a01 The value of one element of the transformation matrix.
       * @param a02 The value of one element of the transformation matrix.
       * @param a03 The value of one element of the transformation matrix.
       * @param a10 The value of one element of the transformation matrix.
       * @param a11 The value of one element of the transformation matrix.
       * @param a12 The value of one element of the transformation matrix.
       * @param a13 The value of one element of the transformation matrix.
       * @param a20 The value of one element of the transformation matrix.
       * @param a21 The value of one element of the transformation matrix.
       * @param a22 The value of one element of the transformation matrix.
       * @param a23 The value of one element of the transformation matrix.
       * @param a30 The value of one element of the transformation matrix.
       * @param a31 The value of one element of the transformation matrix.
       * @param a32 The value of one element of the transformation matrix.
       * @param a33 The value of one element of the transformation matrix.
       */
      inline
      Transform3D(double a00, double a01, double a02, double a03,
                  double a10, double a11, double a12, double a13,
                  double a20, double a21, double a22, double a23,
                  double a30, double a31, double a32, double a33)
        : m_00(a00), m_01(a01), m_02(a02), m_03(a03),
          m_10(a10), m_11(a11), m_12(a12), m_13(a13),
          m_20(a20), m_21(a21), m_22(a22), m_23(a23),
          m_30(a30), m_31(a31), m_32(a32) {
        this->normalize(a33);
      }

      /** 
       * Build a Transform3D from a homogeneous 4x4 matrix.
       *
       * @param source A 2D array containing the elements of the desired
       * homogeneous transformation.
       */
      Transform3D(const Array2D<double>& source);

      /** 
       * The copy constructor simply duplicates its argument.
       * 
       * @param src This is the Transform3D instance to be copied.
       */
      inline
      Transform3D(const Transform3D& src)
        : m_00(src.m_00), m_01(src.m_01), m_02(src.m_02), m_03(src.m_03),
          m_10(src.m_10), m_11(src.m_11), m_12(src.m_12), m_13(src.m_13),
          m_20(src.m_20), m_21(src.m_21), m_22(src.m_22), m_23(src.m_23),
          m_30(src.m_30), m_31(src.m_31), m_32(src.m_32) {}

      /** 
       * Destructor.
       */
      ~Transform3D() {}


      /** 
       * This member function returns a functor which makes it easier to
       * transform arrays of points using algorithms such as
       * std::transform().  For example:
       *
       * @code
       * std::transform(myPoints.begin(), myPoints.end(), myNewPoints.begin(),
       *                myTransform.getFunctor());
       * @endcode
       * 
       * @return The return value is a functor instance which will
       * transform points according to the current value of *this.  The
       * functor will contain a copy of *this, so that subsequent
       * changes to *this will not affect the functor.
       */
      Transform3DFunctor
      getFunctor() const;
    

      /** 
       * This member function returns one element from the matrix
       * representation of the coordinate transform by value.  Note
       * that indexing is zero-based.  For example, calling
       * myTransform3D.value<1, 2>() will return the element from the
       * 2nd row, 3rd column of the matrix representation of the
       * coordinate transformation.  Indexing using this function is
       * faster than operator()(), but the indices must be known at
       * compile time.  This is an alias for member function template
       * value<row, column>().
       * 
       * @return The value of the requested element.
       */
      template <size_t row, size_t column>
      double
      getValue() const {return this->value<row, column>();}


      /** 
       * This member function returns the inverse of *this.  It is an
       * error if *this is not invertible.
       * 
       * @return The return value is the inverse of *this.
       */
      Transform3D
      invert() const;

    
      /** 
       * Change the Transform3D value by explicitly setting element values
       * as if setting the elements of a 4x4 transformation matrix:
       *    [[a00, a01, a02, a03],
       *     [a10, a11, a12, a13],
       *     [a20, a21, a22, a23],
       *     [a30, a31, a32, a33]]
       * 
       * @param a00 The value of one element of the transformation matrix.
       * @param a01 The value of one element of the transformation matrix.
       * @param a02 The value of one element of the transformation matrix.
       * @param a03 The value of one element of the transformation matrix.
       * @param a10 The value of one element of the transformation matrix.
       * @param a11 The value of one element of the transformation matrix.
       * @param a12 The value of one element of the transformation matrix.
       * @param a13 The value of one element of the transformation matrix.
       * @param a20 The value of one element of the transformation matrix.
       * @param a21 The value of one element of the transformation matrix.
       * @param a22 The value of one element of the transformation matrix.
       * @param a23 The value of one element of the transformation matrix.
       * @param a30 The value of one element of the transformation matrix.
       * @param a31 The value of one element of the transformation matrix.
       * @param a32 The value of one element of the transformation matrix.
       * @param a33 The value of one element of the transformation matrix.
       */
      void
      setTransform(double a00, double a01, double a02, double a03,
                   double a10, double a11, double a12, double a13,
                   double a20, double a21, double a22, double a23,
                   double a30, double a31, double a32, double a33);



      /** 
       * This member function sets one element of the matrix
       * representation of the coordinate transform.  Note that indexing
       * is zero-based.  For example, calling myTransform3D.setValue(1,
       * 2, 5.0) will set the element from the 2nd row, 3rd column of
       * the matrix representation of the coordinate transformation to
       * 5.0.  If blindingly fast execution is important, consider using
       * the setValue<size_t, size_t>(double) member function instead.
       *
       * It is not permitted to set the value of the lower-right element
       * of the matrix: setValue(3, 3, ...) will throw an
       * IndexException.
       * 
       * @param row This argument specifies the row of the matrix
       * element to be modified.
       * 
       * @param column This argument specifies the row of the matrix
       * element to be modified.
       * 
       * @param value This argument specifies the value to be copied
       * into the matrix element at the specified row & column.
       */
      void
      setValue(size_t row, size_t column, double value);


      /** 
       * This member function sets one element from the matrix
       * representation of the coordinate transform.  Note that indexing
       * is zero-based.  For example, calling myTransform3D.setValue<1,
       * 2>(5.0) will set the element from the 2nd row, 3rd column of
       * the matrix representation of the coordinate transformation to
       * 5.0.
       *
       * It is not permitted to set the value of the lower-right element
       * of the matrix: setValue<3, 3>(double) will throw an
       * IndexException.
       * 
       * @param value This argument specifies the value to be copied
       * into the matrix element at the specified row & column.
       */
      template <size_t row, size_t column>
      void
      setValue(double value);


      /** 
       * This member function is an alias for member function setTransform().
       */
      void
      setValue(double a00, double a01, double a02, double a03,
               double a10, double a11, double a12, double a13,
               double a20, double a21, double a22, double a23,
               double a30, double a31, double a32, double a33) {
        this->setTransform(a00, a01, a02, a03, a10, a11, a12, a13,
                           a20, a21, a22, a23, a30, a31, a32, a33);
      }

      
      /** 
       * This member function returns one element from the matrix
       * representation of the coordinate transform by value.  Note that
       * indexing is zero-based.  For example, calling
       * myTransform3D.value<1, 2>() will return the element from the
       * 2nd row, 3rd column of the matrix representation of the
       * coordinate transformation.
       * 
       * @return The value of the requested element.
       */
      template <size_t row, size_t column>
      double
      value() const;


      /** 
       * This operator returns one element from the matrix
       * representation of the coordinate transform by value.
       * If blindingly fast execution is important, consider using
       * value<size_t, size_t>() member function.
       *
       * @param row The row of the requested element.
       * @param column The column of the requested element.
       * @return The value of the requested element.
       */
      double
      operator()(size_t row, size_t column) const;

    
      /** 
       * This operator takes a point and applies the coordinate
       * transform, returning the result.
       * 
       * @param vector0 The point to be transformed.
       * @return The result of the transformation.
       */
      Vector3D
      operator*(const Vector3D& vector0) const;

    
      /** 
       * The assignment operator simply duplicates its argument.
       * 
       * @param source This is the Transform3D instance to be copied.
       * @return A reference to *this.
       */
      Transform3D&
      operator=(const Transform3D& source);

    
    private:
      void normalize(double scaleFactor);
      
      double m_00, m_01, m_02, m_03;
      double m_10, m_11, m_12, m_13;
      double m_20, m_21, m_22, m_23;
      double m_30, m_31, m_32;

    }; // class Transform3D


    /* ============== Helper classes ============== */


    /**
     ** This helper class works with Transform3D::getFunctor()
     **/
    class Transform3DFunctor
      : public std::unary_function<Vector3D, Vector3D> {

    public:

      /** 
       * The constructor deep-copies its argument.
       * 
       *  @param transform This is the transform instance to be copied.
       */
      Transform3DFunctor(const Transform3D& transform)
        : m_transform(transform) {}

    
      /** 
       * The application operator transforms its argument.
       * 
       *  @param vec This is the vector to be transformed.
       * 
       *  @return The transformed vector.
       */
      inline Vector3D
      operator()(const Vector3D& vec) const {return m_transform * vec;}
    
    private:
      Transform3D m_transform;
    };
  
  
    /* ============== Non-member functions ============== */
  
    /** 
     * This operator composes two Transform3D instances.  The resulting
     * transform satisfies the equation:
     *
     *  (transform0 * transform1) * v0 = transform0 * (transform1 * v0),
     *
     * where v0 is a Vector3D instance.
     * 
     * @param transform0 This is the first of the two Transform3D instances to
     * be composed.
     * @param transform1 This is the second of the two Transform3D instances to
     * be composed.
     * @return A Transform3D instance which is equivalent to the composition
     * of the two arguments.
     */
    Transform3D
    operator*(const Transform3D& transform0, const Transform3D& transform1);

  
    /** 
     * Outputs a text representation of a Transform3D instance to a
     * std::ostream.  The output format looks like this:
     *
     * Transform3D(1.0, 2.0, 3.0, 4.0,
     *             5.0, 6.0, 7.0, 8.0,
     *             9.0, 10.0, 11.0, 12.0,
     *             13.0, 14.0, 15.0, 1.0)
     *
     * @param stream Reference to the the output stream.
     *
     * @param transform0 Const reference to the Transform3D instance to be
     * output.
     *
     * @return Reference to the output stream.
     */
    std::ostream&
    operator<<(std::ostream& stream, const Transform3D& transform0);

  
    /** 
     * Sets the value of a Transform3D instance from a std::istream.
     * The input format is as described for
     * operator<<(std::ostream&, const Transform3D&), above.
     * 
     * @param stream Reference to the the input stream.
     *
     * @param transform0 Reference to the Transform3D that will take
     * the input.
     *
     * @return Reference to the input stream.
     */
    std::istream&
    operator>>(std::istream& stream, Transform3D& transform0);
  
  } // namespace numeric

} // namespace dlr


/* ======= Declarations to maintain compatibility with legacy code. ======= */

namespace dlr {

  using numeric::Transform3D;
  using numeric::Transform3DFunctor;

} // namespace dlr


/*******************************************************************
 * Member function definitions follow.  This would be a .C file
 * if it weren't templated.
 *******************************************************************/

namespace dlr {

  namespace numeric {
    
    // This operator sets one element of the matrix representation of
    // the coordinate transform.  The case statements should optimize
    // away, since row and column are both known at compile time.
    template <size_t ROW, size_t COLUMN>
    void
    Transform3D::
    setValue(double inValue)
    {
      switch(ROW) {
      case 0:
        switch(COLUMN) {
        case 0: m_00 = inValue; return; break;
        case 1: m_01 = inValue; return; break;
        case 2: m_02 = inValue; return; break;
        case 3: m_03 = inValue; return; break;
        default: break;
        }
        break;
      case 1:
        switch(COLUMN) {
        case 0: m_10 = inValue; return; break;
        case 1: m_11 = inValue; return; break;
        case 2: m_12 = inValue; return; break;
        case 3: m_13 = inValue; return; break;
        default: break;
        }
        break;
      case 2:
        switch(COLUMN) {
        case 0: m_20 = inValue; return; break;
        case 1: m_21 = inValue; return; break;
        case 2: m_22 = inValue; return; break;
        case 3: m_23 = inValue; return; break;
        default: break;
        }
        break;
      case 3:
        switch(COLUMN) {
        case 0: m_30 = inValue; return; break;
        case 1: m_31 = inValue; return; break;
        case 2: m_32 = inValue; return; break;
        default: break;
        }
        break;
      default:
        break;
      }
      std::ostringstream message;
      message << "Indices (" << ROW << ", " << COLUMN << ") are out of bounds.";
      DLR_THROW(IndexException, "Transform3D::setValue<size_t, size_t>(double)",
                message.str().c_str());
    }

  
    // This operator returns one element from the matrix
    // representation of the coordinate transform by value.
    // The case statements should optimize away, since row and column
    // are both known at compile time.
    template <size_t ROW, size_t COLUMN>
    double
    Transform3D::
    value() const
    {
      switch(ROW) {
      case 0:
        switch(COLUMN) {
        case 0: return m_00; break;
        case 1: return m_01; break;
        case 2: return m_02; break;
        case 3: return m_03; break;
        default: break;
        }
        break;
      case 1:
        switch(COLUMN) {
        case 0: return m_10; break;
        case 1: return m_11; break;
        case 2: return m_12; break;
        case 3: return m_13; break;
        default: break;
        }
        break;
      case 2:
        switch(COLUMN) {
        case 0: return m_20; break;
        case 1: return m_21; break;
        case 2: return m_22; break;
        case 3: return m_23; break;
        default: break;
        }
        break;
      case 3:
        switch(COLUMN) {
        case 0: return m_30; break;
        case 1: return m_31; break;
        case 2: return m_32; break;
        case 3: return 1.0; break;
        default: break;
        }
        break;
      default:
        break;
      }
      std::ostringstream message;
      message << "Indices (" << ROW << ", " << COLUMN << ") are out of bounds.";
      DLR_THROW(IndexException, "Transform3D::value<size_t, size_t>()",
                message.str().c_str());
      return 0.0; // Dummy return to keep the compiler happy.
    }

  } // namespace numeric

} // namespace dlr
  
#endif // #ifndef _DLR_TRANSFORM3D_H_
