/**
***************************************************************************
* @file dlrNumeric/solveQuadratic.h
*
* Header file declaring a function for solving quadratic polynomial
* equations of a single variable.
*
* Copyright (C) 2001-2009 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: $
* $Date: $
***************************************************************************
**/

#ifndef DLR_NUMERIC_SOLVEQUADRATIC_H
#define DLR_NUMERIC_SOLVEQUADRATIC_H

#include <complex>

namespace dlr {

  namespace numeric {
    
    /** 
     * This function computes the real roots of the quadratic polynomial
     * c0*x^2 + c1*x + c2 = 0.
     *
     * Note that the two well known versions of the quadratic formula:
     * 
     *   x = (-c1 +|- sqrt(c1**2 - 4c0*c2)) / (2*c0)
     * 
     * and
     * 
     *   x = 2*c2 / (-c1 +|- sqrt(c1**2 - 4*c0*c2))
     * 
     * both tend to be inaccurate when c0 and/or c2 are small, since
     * then the quantity (-c1 +|- sqrt(c1**2 - 4*c0*c2)) gets very
     * small and loses significance.  Instead we use the form
     * advocated by Press and Flannery (Numerical Recipes):
     *
     *   q = -(1/2)(c1 + sgn(c1)*sqrt(c1**2 - 4*c1*c2))
     *   x1 = q/c0, x2 = c2/q
     *
     * This is just the same as using both of the first two forms, each
     * one for the root at which it is most numerically stable.
     *
     * @param c0 This argument is the quadratic coefficient of the
     * polynomial.
     * 
     * @param c1 This argument is the linear coefficient of the
     * polynomial.
     * 
     * @param c2 This argument is the constant coefficient of the
     * polynomial.
     * 
     * @param root0 If the polynomial has real roots, this reference
     * argument is used to return the first root.
     * 
     * @param root1 If the polynomial has real roots, this reference
     * argument is used to return the second root.
     * 
     * @return If the polynomial has real roots, the return value is
     * true.  If the polynomial does not have real roots, the return
     * value is false, and arguments root0 and root1 are not changed.
     */
    template <class Type>
    bool
    solveQuadratic(Type c0, Type c1, Type c2,
                   Type& root0, Type& root1);

  
    /** 
     * This function computes the (possibly complex) roots of the
     * quadratic polynomial c0*x^2 + c1*x + c2 = 0.
     * 
     * @param c0 This argument is the quadratic coefficient of the
     * polynomial.
     * 
     * @param c1 This argument is the linear coefficient of the
     * polynomial.
     * 
     * @param c2 This argument is the constant coefficient of the
     * polynomial.
     * 
     * @param root0 This reference argument is used to return the
     * first root of the polynomial.
     * 
     * @param root1 This reference argument is used to return the
     * second root of the polynomial.
     */
    template <class Type>
    void
    solveQuadratic(Type c0, Type c1, Type c2,
                   std::complex<Type>& root0, std::complex<Type>& root1);


    /** 
     * This function computes the roots of the quadratic polynomial
     * x^2 + c0*x + c1 = 0, where c0 and c1 are complex.
     * 
     * @param c0 This argument is the linear coefficient of the
     * polynomial.
     * 
     * @param c1 This argument is the constant coefficient of the
     * polynomial.
     * 
     * @param root0 This reference argument is used to return the
     * first root of the polynomial.
     * 
     * @param root1 This reference argument is used to return the
     * second root of the polynomial.
     */
    template <class Type>
    void
    solveQuadratic(std::complex<Type> c0, std::complex<Type> c1,
                   std::complex<Type>& root0, std::complex<Type>& root1);
    
  
  } // namespace numeric

} // namespace dlr


/* ======== Inline and template definitions below. ======== */

#include <cmath>

namespace dlr {

  namespace numeric {

    // This function computes the real roots of the quadratic polynomial
    // c0*x^2 + c1*x + c0 = 0.
    template <class Type>
    bool
    solveQuadratic(Type c0, Type c1, Type c2,
                   Type& root0, Type& root1)
    {
      Type ss = (c1 * c1) - (static_cast<Type>(4.0) * c0 * c2);
      if(ss < static_cast<Type>(0.0)) {
        return false;
      }
      Type rt = std::sqrt(ss);
      if(c1 < static_cast<Type>(0.0)) {
        rt = -rt;
      }
      Type qq = static_cast<Type>(-0.5) * (rt + c1);
      root0 = qq / c0;
      root1 = c2 / qq;
      return true;
    }



    // This function computes the (possibly complex) roots of the
    // quadratic polynomial c0*x^2 + c1*x + c2 = 0.
    template <class Type>
    void
    solveQuadratic(Type c0, Type c1, Type c2,
                   std::complex<Type>& root0, std::complex<Type>& root1)
    {
      Type ss = (c1 * c1) - (static_cast<Type>(4.0) * c0 * c2);
      std::complex<Type> rt;
      if(ss >= static_cast<Type>(0.0)) {
        rt.real() = std::sqrt(ss);
        rt.imag() = static_cast<Type>(0.0);
      } else {
        rt.real() = static_cast<Type>(0.0);
        rt.imag() = std::sqrt(-ss);
      }
      if(c1 < static_cast<Type>(0.0)) {
        rt = -rt;
      }
      std::complex<Type> qq = static_cast<Type>(-0.5) * (rt + c1);
      root0 = qq / c0;
      root1 = c2 / qq;
    }


    // This function computes the roots of the quadratic polynomial
    // x^2 + c0*x + c1 = 0, where c0 and c1 are complex.
    template <class Type>
    void
    solveQuadratic(std::complex<Type> c0, std::complex<Type> c1,
                   std::complex<Type>& root0, std::complex<Type>& root1)
    {
      std::complex<Type> ss = (c0 * c0) - (static_cast<Type>(4.0) * c1);
      std::complex<Type> rt = std::sqrt(ss);
      if((c0.real() * rt.real() + c0.imag() * rt.imag())
         < static_cast<Type>(0.0)) {
        rt = -rt;
      }
      std::complex<Type> qq = static_cast<Type>(-0.5) * (rt + c0);
      root0 = qq;

      // If qq is zero, then c0 and c1 are zero, and we know the
      // second root.
      if(qq.real() || qq.imag()) {
        root1 = c1 / qq;
      } else {
        root1.real() = Type(0.0);
        root1.imag() = Type(0.0);
      }
    } 
    
  } // namespace numeric

} // namespace dlr

#endif /* #ifndef DLR_NUMERIC_UTILITIES_H */
